#!/usr/bin/env python3
"""
Script de migration pour ajouter la colonne error_details
"""
import sqlite3
import sys
import os

def check_column_exists(cursor, table_name, column_name):
    """Vérifie si une colonne existe dans une table"""
    cursor.execute(f"PRAGMA table_info({table_name})")
    columns = [row[1] for row in cursor.fetchall()]
    return column_name in columns

def migrate_database(dbpath):
    """Applique la migration"""
    if not os.path.exists(dbpath):
        print(f"Erreur: Base de données introuvable: {dbpath}")
        return False

    try:
        conn = sqlite3.connect(dbpath)
        cursor = conn.cursor()

        # Vérifier si la colonne existe déjà
        if check_column_exists(cursor, 'queue', 'error_details'):
            print("[OK] La colonne 'error_details' existe déjà dans la table 'queue'")
            conn.close()
            return True

        # Ajouter la colonne
        print("Ajout de la colonne 'error_details' à la table 'queue'...")
        cursor.execute("ALTER TABLE queue ADD COLUMN error_details TEXT")
        conn.commit()

        # Vérifier que l'ajout a réussi
        if check_column_exists(cursor, 'queue', 'error_details'):
            print("[OK] Migration réussie")
            conn.close()
            return True
        else:
            print("[ERREUR] La migration a échoué")
            conn.close()
            return False

    except sqlite3.Error as e:
        print(f"[ERREUR] SQLite: {e}")
        return False
    except Exception as e:
        print(f"[ERREUR] Inattendue: {e}")
        return False

if __name__ == "__main__":
    # Par défaut, utiliser le chemin de la base de données du middleware
    dbpath = "./data/middleware.db"

    if len(sys.argv) > 1:
        dbpath = sys.argv[1]

    print(f"Migration de la base de données: {dbpath}")
    print("-" * 60)

    success = migrate_database(dbpath)

    if success:
        print("-" * 60)
        print("Migration terminée avec succès!")
        sys.exit(0)
    else:
        print("-" * 60)
        print("Migration échouée")
        sys.exit(1)
