"""
Worker pour traiter les requêtes en queue
"""
import time
from datetime import datetime, timedelta
from typing import List
from app.database import Database
from app.api_client import APIClient
from app.models import QueuedRequest, APIRequest, APIResponse, RequestStatus
from app.config import API_DESTINATIONS, ROUTING_RULES, QUEUE_CONFIG
from app.formatters import get_formatter
from app.logger import get_logger
from app.validators.registry import validate_data

logger = get_logger(__name__)

# Import optionnel du module email
try:
    from app.email_notifications import MiddlewareEmailNotifications
    EMAIL_ENABLED = True
except ImportError:
    EMAIL_ENABLED = False
    logger.warning("email_module_not_available", message="Module email_sender non disponible")

class Worker:
    def __init__(self, worker_id: int = 1, config: dict = None):
        self.worker_id = worker_id
        # Utiliser fne.db depuis la config si disponible
        fne_config = config.get("fne_cdi", {})
        dbpath = fne_config.get("db")
        if dbpath is None:
            raise ValueError("fne.db doit être fourni dans la configuration du worker")
        self.db = Database(dbpath=dbpath)
        self.api_client = APIClient(
            max_retries=3,
            retry_delays=QUEUE_CONFIG['retry_delays']
        )
        self.running = False

        # Initialiser le module email si disponible et configuré
        self.email_notif = None
        if EMAIL_ENABLED and config:
            try:
                self.email_notif = MiddlewareEmailNotifications(config)
                logger.info("email_notifications_enabled", worker_id=worker_id)
            except Exception as e:
                logger.critical("email_notifications_init_failed", error=str(e))

        self.config = config
        logger.info("worker_initialized", worker_id=worker_id, dbpath=dbpath)
    
    def start(self):
        """Démarre le worker"""
        self.running = True
        logger.info("worker_started", worker_id=self.worker_id)
        
        while self.running:
            try:
                # Récupérer les requêtes en attente
                pending = self.db.get_pending_requests(
                    limit=QUEUE_CONFIG['batch_size']
                )
                
                if pending:
                    logger.info(
                        "processing_batch",
                        worker_id=self.worker_id,
                        request_count=len(pending)
                    )
                    for request in pending:
                        self.process_request(request)
                else:
                    # Pas de requêtes, attendre un peu
                    time.sleep(2)
            
            except Exception as e:
                logger.error("worker_error", worker_id=self.worker_id, error=str(e))
                time.sleep(5)

        logger.info("worker_stopped", worker_id=self.worker_id)
    
    def stop(self):
        """Arrête le worker"""
        self.running = False
    
    def process_request(self, request: QueuedRequest):
        """Traite une requête"""
        logger.info(
            "processing_request",
            request_id=request.id,
            event_type=request.event_type,
            worker_id=self.worker_id
        )
        
        # Marquer comme en cours de traitement
        self.db.update_request_status(request.id, RequestStatus.PROCESSING)
        
        try:
            # Récupérer les règles de routing
            routing_rule = ROUTING_RULES.get(request.event_type)
            
            if not routing_rule:
                logger.warning(
                    "no_routing_rule",
                    event_type=request.event_type,
                    request_id=request.id
                )
                self.db.update_request_status(request.id, RequestStatus.FAILED)
                return
            
            # === VALIDATION DES DONNÉES ===
            validator_name = routing_rule.get('validator')
            if validator_name:
                logger.debug(
                    "validating_data",
                    request_id=request.id,
                    validator=validator_name
                )

                validation_result = validate_data(validator_name, request.data)

                if not validation_result.valid:
                    # Validation échouée - sauvegarder les détails
                    error_details = {
                        "type": "validation_error",
                        "validator": validator_name,
                        "errors": validation_result.errors,
                        "warnings": validation_result.warnings,
                        "timestamp": datetime.utcnow().isoformat()
                    }

                    logger.error(
                        "validation_failed",
                        request_id=request.id,
                        errors=validation_result.errors,
                        warnings=validation_result.warnings
                    )

                    self.db.update_request_status(
                        request.id,
                        RequestStatus.FAILED,
                        error_details=error_details
                    )
                    return

                # Log des warnings si présents
                if validation_result.warnings:
                    logger.warning(
                        "validation_warnings",
                        request_id=request.id,
                        warnings=validation_result.warnings
                    )

                # Utiliser les données validées et normalisées
                request.data = validation_result.data

                logger.info(
                    "validation_passed",
                    request_id=request.id,
                    warnings_count=len(validation_result.warnings)
                )

            # Vérifier la condition si c'est un routing conditionnel
            if routing_rule['type'] == 'conditional':
                condition = routing_rule.get('condition')
                if condition and not condition(request.data):
                    logger.info(
                        "condition_not_met",
                        request_id=request.id,
                        event_type=request.event_type
                    )
                    self.db.update_request_status(request.id, RequestStatus.COMPLETED)
                    return

            # Préparer les requêtes API
            destinations = routing_rule['destinations']
            formatter_name = routing_rule.get('formatter', 'format_generic')
            formatter = get_formatter(formatter_name)
            
            api_requests = []
            for dest in destinations:
                dest_config = API_DESTINATIONS.get(dest)
                if not dest_config:
                    logger.warning("unknown_destination", destination=dest, request_id=request.id)
                    continue
                
                # Formater les données pour cette destination
                formatted_data = formatter(request.data, dest)
                
                # Créer la requête API
                api_req = APIRequest(
                    queue_id=request.id,
                    destination=dest,
                    url=dest_config['base_url'],
                    method="POST",
                    headers=dest_config['headers'],
                    payload=formatted_data,
                    timeout=dest_config['timeout']
                )
                api_requests.append(api_req)
            
            # Envoyer les requêtes
            all_success = True
            responses_list = []
            for api_req in api_requests:
                response = self.api_client.send_request(api_req)

                # Sauvegarder la réponse
                self.db.save_api_response(response)
                responses_list.append(response)

                # Vérifier le succès
                if response.error or (response.status_code and response.status_code >= 400):
                    all_success = False

            # Mettre à jour le statut final
            if all_success:
                self.db.update_request_status(request.id, RequestStatus.COMPLETED)
                logger.info(
                    "request_completed",
                    request_id=request.id,
                    event_type=request.event_type
                )

                # Envoyer les notifications email si configuré
                if self.email_notif:
                    self._send_email_notifications(request, responses_list)

            else:
                # Vérifier si on doit retenter
                if request.retry_count < request.max_retries:
                    retry_count = request.retry_count + 1
                    delay_index = min(retry_count - 1, len(QUEUE_CONFIG['retry_delays']) - 1)
                    delay_seconds = QUEUE_CONFIG['retry_delays'][delay_index]
                    next_retry = datetime.utcnow() + timedelta(seconds=delay_seconds)
                    
                    self.db.update_request_status(
                        request.id,
                        RequestStatus.RETRYING,
                        retry_count=retry_count,
                        next_retry_at=next_retry
                    )
                    logger.warning(
                        "request_retry_scheduled",
                        request_id=request.id,
                        retry_count=retry_count,
                        delay_seconds=delay_seconds,
                        next_retry_at=next_retry.isoformat()
                    )
                else:
                    self.db.update_request_status(request.id, RequestStatus.FAILED)
                    logger.error(
                        "request_failed",
                        request_id=request.id,
                        max_retries=request.max_retries,
                        event_type=request.event_type
                    )
        
        except Exception as e:
            # Capturer les détails de l'erreur
            import traceback
            error_details = {
                "type": "processing_error",
                "error": str(e),
                "traceback": traceback.format_exc(),
                "timestamp": datetime.utcnow().isoformat()
            }

            logger.error(
                "request_processing_error",
                request_id=request.id,
                error=str(e),
                exc_info=True
            )

            # Tentative de retry si possible
            if request.retry_count < request.max_retries:
                retry_count = request.retry_count + 1
                next_retry = datetime.utcnow() + timedelta(seconds=30)
                self.db.update_request_status(
                    request.id,
                    RequestStatus.RETRYING,
                    retry_count=retry_count,
                    next_retry_at=next_retry,
                    error_details=error_details
                )
            else:
                self.db.update_request_status(
                    request.id,
                    RequestStatus.FAILED,
                    error_details=error_details
                )

    def _send_email_notifications(self, request: QueuedRequest, responses: List[APIResponse]):
        """
        Envoie les notifications email après un traitement réussi

        Args:
            request: Requête traitée
            responses: Liste des réponses API
        """
        try:
            # Pour chaque réponse réussie, envoyer les notifications appropriées
            for response in responses:
                if not response.response_data:
                    continue

                response_data = response.response_data

                logger.debug("hello", request=request.data)
                logger.debug("hello", response=response.response_data)

                # 1. Email au client si token présent
                # Note: token est une URL complète (ex: "http://54.247.95.108/fr/verification/019a799e...")
                download_url = response_data.get("token")
                if download_url:
                    # Extraire les infos client depuis les données originales
                    client_email = self._extract_client_email(request.data)
                    client_name = self._extract_client_name(request.data)
                    invoice_num = request.data.get("fac_num", "N/A")
                    invoice_date = request.data.get("fac_date", "N/A")
                    invoice_amount = self._extract_invoice_amount(request.data)

                    if not client_email:
                        client_email = self._extract_email_if_client_email()

                    if self._ignore_client_email:
                        client_email, _ = self._replace_client_email_with_redirect_emails()

                    logger.debug("hello", client_email=client_email)
                    if client_email:
                        try:
                            logger.debug(
                                "sending_invoice_email",
                                request_id=request.id,
                                client_email=client_email,
                                invoice_number=invoice_num,
                                download_url_length=len(download_url) if download_url else 0
                            )
                            rslt = self.email_notif.send_invoice_to_client(
                                client_email=client_email,
                                client_name=client_name,
                                invoice_number=invoice_num,
                                invoice_date=invoice_date,
                                invoice_amount=invoice_amount,
                                download_url=download_url
                            )
                            logger.info(
                                "invoice_email_sent",
                                request_id=request.id,
                                client_email=client_email,
                                response=rslt
                            )
                        except Exception as e:
                            logger.error(
                                "invoice_email_failed",
                                request_id=request.id,
                                client_email=client_email,
                                error=str(e),
                                exc_info=True
                            )
                    else:
                        logger.warning(
                            "no_client_email",
                            request_id=request.id,
                            invoice_number=invoice_num
                        )

                # 2. Email admin si warnings ou balance faible
                warnings = response_data.get("warning", [])
                balance_funds = response_data.get("balance_funds")

                # Envoyer alerte admin si warnings présents OU balance < 5000
                should_alert = (warnings and len(warnings) > 0) or \
                              (balance_funds is not None and balance_funds < 5000)

                if should_alert:
                    try:
                        self.email_notif.send_admin_alert(
                            warnings=warnings if warnings else None,
                            balance_funds=balance_funds,
                            invoice_number=request.data.get("fac_num"),
                            invoice_date=request.data.get("fac_date"),
                            invoice_amount=self._extract_invoice_amount(request.data),
                            client_name=self._extract_client_name(request.data),
                            request_id=request.id,
                            download_url=download_url
                        )
                        logger.info(
                            "admin_alert_sent",
                            request_id=request.id,
                            warnings_count=len(warnings) if warnings else 0,
                            balance=balance_funds
                        )
                    except Exception as e:
                        logger.error(
                            "admin_alert_failed",
                            request_id=request.id,
                            error=str(e)
                        )

        except Exception as e:
            # Ne pas faire échouer le traitement si l'email échoue
            logger.error(
                "email_notifications_error",
                request_id=request.id,
                error=str(e),
                exc_info=True
            )

    def _extract_client_email(self, data: dict) -> str:
        """Extrait l'email client des données GCVX"""
        try:
            return data.get("client", {}).get("adresses", {}).get("facturation", {}).get("email", "")
        except (AttributeError, KeyError):
            return ""

    def _extract_email_if_client_email(self) -> str:
        """Extrait l'email à mettre par défaut si le client ne possède pas d'email valide"""
        try:
            return self.config.get("email",{})\
                              .get("notifications", {})\
                              .get("if_invalid","")
        except (AttributeError, KeyError):
            return ""

    def _extract_client_name(self, data: dict) -> str:
        """Extrait le nom client des données GCVX"""
        try:
            return data.get("client", {}).get("adresses", {}).get("facturation", {}).get("raison_sociale", "Client")
        except (AttributeError, KeyError):
            return "Client"

    def _extract_invoice_amount(self, data: dict) -> float:
        """Extrait le montant TTC de la facture"""
        try:
            return float(data.get("pied", {}).get("montant_ttc", 0))
        except (AttributeError, KeyError, ValueError, TypeError):
            return 0.0

    @property
    def _ignore_client_email(self):
        return self.config.get("fne_cdi", {}).get("dev",{}).get("enabled", False)

    def _get_fne_cdi_dev_mode(self):
        return self.config.get("fne_cdi", {}).get("dev",{}).get("enabled", False)

    def _get_fne_cdi_redirect_emails(self):
        return self.config.get("fne_cdi", {}).get("dev",{}).get("redirect_emails", [])

    def _replace_client_email_with_redirect_emails(self):
        # Mode développement : rediriger les emails vers les adresses de test
        dev_mode = self._get_fne_cdi_dev_mode()
        dev_redirect_emails = self._get_fne_cdi_redirect_emails()
        if dev_mode and dev_redirect_emails:
            email = dev_redirect_emails[0]
            cc = dev_redirect_emails[1:]
            return email, cc
        return None, []
        # else:
        #     return email, cc
        #     # Les autres emails de la liste sont mis en CC
        #     if len(self.dev_redirect_emails) > 1:
        #         dev_cc_emails = self.dev_redirect_emails[1:]
        #     logger.info(
        #         "dev_mode_email_redirect",
        #         original_email=original_email,
        #         redirected_to=client_email,
        #         cc=dev_cc_emails,
        #         invoice_number=invoice_number
        #     )
