"""
Couche de compatibilité Pydantic v1 et v2
Permet au code de fonctionner avec Pydantic < 2.0 (CentOS 6) et >= 2.0

Ce module centralise toutes les fonctions de compatibilité entre les versions
de Pydantic pour assurer le bon fonctionnement sur différents environnements.
"""
import sys
from typing import Any, Callable

try:
    from pydantic import VERSION
    PYDANTIC_VERSION = int(VERSION.split('.')[0])
except (ImportError, AttributeError):
    PYDANTIC_VERSION = 1

PYDANTIC_V2 = PYDANTIC_VERSION >= 2


def get_validator_decorator():
    """
    Retourne le décorateur de validation approprié selon la version de Pydantic

    Pydantic v1: utilise @validator
    Pydantic v2: utilise @field_validator
    """
    if PYDANTIC_V2:
        from pydantic import field_validator
        return field_validator
    else:
        from pydantic import validator
        return validator


def get_model_validator_decorator():
    """
    Retourne le décorateur de validation de modèle selon la version

    Pydantic v1: utilise @root_validator
    Pydantic v2: utilise @model_validator
    """
    if PYDANTIC_V2:
        from pydantic import model_validator
        return model_validator
    else:
        from pydantic import root_validator
        return root_validator


def make_field_validator(field_names, *args, **kwargs):
    """
    Crée un validateur de champ compatible avec v1 et v2

    Usage:
        @make_field_validator('amount')
        def validate_amount(cls, v):
            return v
    """
    if PYDANTIC_V2:
        from pydantic import field_validator
        # En v2, field_validator prend le nom du champ comme premier argument
        if isinstance(field_names, str):
            return field_validator(field_names, *args, **kwargs)
        else:
            return field_validator(*field_names, *args, **kwargs)
    else:
        from pydantic import validator
        # En v1, validator prend le nom du champ comme premier argument
        if isinstance(field_names, str):
            return validator(field_names, *args, **kwargs)
        else:
            return validator(*field_names, *args, **kwargs)


def make_model_validator(mode='after', *args, **kwargs):
    """
    Crée un validateur de modèle compatible avec v1 et v2

    Usage:
        @make_model_validator(mode='after')
        def validate_model(cls, values):
            return values
    """
    if PYDANTIC_V2:
        from pydantic import model_validator
        return model_validator(mode=mode, *args, **kwargs)
    else:
        from pydantic import root_validator
        # En v1, on utilise pre=False pour mode='after'
        pre = (mode == 'before')
        return root_validator(pre=pre, *args, **kwargs)


def get_model_config(config_dict):
    """
    Retourne la configuration appropriée selon la version de Pydantic

    Pydantic v1: retourne une classe Config
    Pydantic v2: retourne model_config dict
    """
    if PYDANTIC_V2:
        # En v2, on utilise model_config
        return config_dict
    else:
        # En v1, on doit créer une classe Config
        class Config:
            anystr_strip_whitespace = config_dict.get('str_strip_whitespace', False)
            schema_extra = config_dict.get('json_schema_extra', {})
        return Config


def get_model_dict(model):
    """
    Retourne le dictionnaire d'un modèle Pydantic de manière compatible

    Pydantic v1: model.dict()
    Pydantic v2: model.model_dump()
    """
    if PYDANTIC_V2:
        return model.model_dump()
    else:
        return model.dict()


def create_literal_type(*values):
    """
    Crée un type Literal compatible
    """
    from typing import Literal
    return Literal[values]
