"""
API FastAPI principale - Middleware de routing
"""
import os
from fastapi import FastAPI, HTTPException, BackgroundTasks, status
from fastapi.responses import JSONResponse
from contextlib import asynccontextmanager
import threading
from typing import List, Optional

from app.models import IncomingRequest, QueuedRequest, RequestResult, RequestStatus
from app.database import Database
from app.worker import Worker
from app.config import QUEUE_CONFIG, API_DESTINATIONS, ROUTING_RULES, get_destinations_info, MIDDLEWARE_CONFIG
from app.logger import get_logger

logger = get_logger(__name__)

# Variables globales pour les workers et la base de données
workers = []
worker_threads = []
db: Database = None  # Initialisé dans lifespan


def get_db() -> Database:
    """Retourne l'instance de la base de données (lazy-loading)"""
    global db
    if db is None:
        fne_config = MIDDLEWARE_CONFIG.get("fne", {})
        dbpath = fne_config.get("db")
        if dbpath is None:
            raise ValueError("fne.db must be specified in MIDDLEWARE_CONFIG")
        db = Database(dbpath=dbpath)
    return db


@asynccontextmanager
async def lifespan(app: FastAPI):
    """Gestion du cycle de vie de l'application"""
    global db

    # Startup
    logger.info("middleware_starting")

    # Initialiser la base de données
    db = get_db()

    # Démarrer les workers avec la config pour les emails
    for i in range(QUEUE_CONFIG['worker_count']):
        worker = Worker(worker_id=i+1, config=MIDDLEWARE_CONFIG)
        workers.append(worker)

        thread = threading.Thread(target=worker.start, daemon=True)
        thread.start()
        worker_threads.append(thread)

    logger.info("middleware_started", worker_count=len(workers))

    yield

    # Shutdown
    logger.info("middleware_shutting_down")
    for worker in workers:
        worker.stop()

    logger.info("middleware_stopped")

# Créer l'application FastAPI
app = FastAPI(
    title="API Middleware",
    description="Middleware pour router et traiter des requêtes vers plusieurs APIs",
    version="1.0.0",
    lifespan=lifespan
)

@app.get("/")
def root():
    """Point d'entrée de l'API"""
    return {
        "name": "API Middleware",
        "version": "1.0.0",
        "status": "running",
        "workers": len(workers)
    }

@app.get("/health")
def health_check():
    """Vérification de santé"""
    stats = db.get_queue_stats()
    return {
        "status": "healthy",
        "workers": len(workers),
        "queue_stats": stats
    }

@app.post("/submit", status_code=status.HTTP_202_ACCEPTED)
def submit_request(request: IncomingRequest):
    """
    Soumet une requête au middleware pour traitement
    
    La requête sera mise en queue et traitée de manière asynchrone
    """
    try:
        # Valider que le type d'événement existe
        if request.event_type not in ROUTING_RULES:
            raise HTTPException(
                status_code=400,
                detail=f"Unknown event type: {request.event_type}. Available: {list(ROUTING_RULES.keys())}"
            )
        
        # Créer la requête en queue
        queued_request = QueuedRequest(
            event_type=request.event_type,
            data=request.data,
            priority=request.priority,
            metadata=request.metadata,
            status=RequestStatus.PENDING
        )
        
        # Ajouter à la base de données
        request_id = db.add_to_queue(queued_request)

        logger.info(
            "request_submitted",
            request_id=request_id,
            event_type=request.event_type,
            priority=request.priority
        )

        return {
            "request_id": request_id,
            "status": "accepted",
            "message": "Request queued for processing"
        }
    
    except HTTPException:
        raise
    except Exception as e:
        logger.error("error_submitting_request", error=str(e), exc_info=True)
        raise HTTPException(status_code=500, detail=str(e))

@app.get("/request/{request_id}")
def get_request_status(request_id: int):
    """
    Récupère le statut et les résultats d'une requête avec détails d'erreurs
    """
    from db_models import QueueModel
    import json

    with db.get_session() as session:
        queue_item = session.query(QueueModel).filter(
            QueueModel.id == request_id
        ).first()

        if not queue_item:
            raise HTTPException(status_code=404, detail="Request not found")

        # Récupérer les réponses API
        responses = db.get_request_responses(request_id)

        # Récupérer les destinations
        routing_rule = ROUTING_RULES.get(queue_item.event_type)
        destinations = routing_rule['destinations'] if routing_rule else []

        # Parser les détails d'erreur si présents
        error_details = None
        if queue_item.error_details:
            try:
                error_details = json.loads(queue_item.error_details)
            except:
                error_details = {"raw": queue_item.error_details}

        # Parser les données originales
        original_data = None
        if queue_item.data:
            try:
                original_data = json.loads(queue_item.data)
            except:
                original_data = {"raw": queue_item.data}

        return {
            "request_id": queue_item.id,
            "event_type": queue_item.event_type,
            "status": queue_item.status.value,
            "priority": queue_item.priority,
            "retry_count": queue_item.retry_count,
            "max_retries": queue_item.max_retries,
            "created_at": queue_item.created_at.isoformat() if queue_item.created_at else None,
            "processed_at": queue_item.processed_at.isoformat() if queue_item.processed_at else None,
            "next_retry_at": queue_item.next_retry_at.isoformat() if queue_item.next_retry_at else None,
            "original_data": original_data,
            "destinations": destinations,
            "error_details": error_details,
            "api_responses": [
                {
                    "destination": r.destination,
                    "request_payload": r.request_payload,
                    "status_code": r.status_code,
                    "response_data": r.response_data,
                    "error": r.error,
                    "duration_ms": r.duration_ms,
                    "timestamp": r.timestamp.isoformat() if r.timestamp else None
                }
                for r in responses
            ]
        }

@app.get("/queue/stats")
def get_queue_stats():
    """
    Statistiques de la queue
    """
    from db_models import QueueModel
    from sqlalchemy import func

    stats = db.get_queue_stats()

    with db.get_session() as session:
        # Requests par type d'événement
        events_result = session.query(
            QueueModel.event_type,
            func.count(QueueModel.id).label("count")
        ).group_by(QueueModel.event_type).all()
        events = {row.event_type: row.count for row in events_result}

        # Temps moyen de traitement
        avg_result = session.query(
            func.avg(
                func.julianday(QueueModel.processed_at) - func.julianday(QueueModel.created_at)
            ) * 86400
        ).filter(QueueModel.processed_at.isnot(None)).scalar()

        avg_processing = avg_result or 0

    return {
        "status_breakdown": stats,
        "events_breakdown": events,
        "avg_processing_time_seconds": round(avg_processing, 2),
        "workers_active": len(workers)
    }

@app.get("/destinations")
def list_destinations():
    """Liste toutes les destinations disponibles avec leurs modules sources"""
    # Informations sur les modules de destinations
    destinations_info = get_destinations_info()

    # Informations sur les APIs configurées
    api_configs = []
    for dest_id, config in API_DESTINATIONS.items():
        api_configs.append({
            "id": dest_id,
            "name": config['name'],
            "base_url": config['base_url'],
            "timeout": config['timeout'],
            "max_retries": config['max_retries']
        })

    return {
        "destinations_modules": destinations_info,
        "api_configurations": api_configs
    }

@app.get("/routing-rules")
def list_routing_rules():
    """Liste toutes les règles de routing"""
    rules = []
    for event_type, rule in ROUTING_RULES.items():
        rules.append({
            "event_type": event_type,
            "type": rule['type'],
            "destinations": rule['destinations'],
            "formatter": rule['formatter']
        })
    return {"routing_rules": rules}

@app.get("/queue/pending")
def list_pending_requests(limit: int = 50, offset: int = 0):
    """
    Liste les requêtes en attente ou en retry

    Args:
        limit: Nombre maximum de requêtes à retourner (défaut: 50, max: 200)
        offset: Nombre de requêtes à ignorer (pagination)
    """
    from db_models import QueueModel
    from sqlalchemy import or_, and_
    from datetime import datetime

    # Limiter le nombre de résultats
    limit = min(limit, 200)

    with db.get_session() as session:
        now = datetime.utcnow()

        # Requêtes pending ou retrying
        query = session.query(QueueModel).filter(
            or_(
                QueueModel.status == RequestStatus.PENDING,
                and_(
                    QueueModel.status == RequestStatus.RETRYING,
                    QueueModel.next_retry_at <= now
                )
            )
        ).order_by(
            QueueModel.priority.asc(),
            QueueModel.created_at.asc()
        )

        # Compter le total
        total_count = query.count()

        # Appliquer pagination
        results = query.offset(offset).limit(limit).all()

        # Formater les résultats
        pending_items = []
        for item in results:
            pending_items.append({
                "request_id": item.id,
                "event_type": item.event_type,
                "status": item.status.value,
                "priority": item.priority,
                "retry_count": item.retry_count,
                "max_retries": item.max_retries,
                "created_at": item.created_at.isoformat() if item.created_at else None,
                "next_retry_at": item.next_retry_at.isoformat() if item.next_retry_at else None
            })

        return {
            "total": total_count,
            "limit": limit,
            "offset": offset,
            "items": pending_items
        }

@app.get("/queue/all")
def list_all_requests(
    status: Optional[str] = None,
    limit: int = 50,
    offset: int = 0
):
    """
    Liste toutes les requêtes avec filtrage optionnel par statut

    Args:
        status: Filtrer par statut (pending, processing, completed, failed, retrying)
        limit: Nombre maximum de requêtes (défaut: 50, max: 200)
        offset: Pagination offset
    """
    from db_models import QueueModel

    # Limiter le nombre de résultats
    limit = min(limit, 200)

    with db.get_session() as session:
        query = session.query(QueueModel)

        # Filtrer par statut si spécifié
        if status:
            try:
                status_enum = RequestStatus(status)
                query = query.filter(QueueModel.status == status_enum)
            except ValueError:
                raise HTTPException(
                    status_code=400,
                    detail=f"Invalid status: {status}. Valid values: {[s.value for s in RequestStatus]}"
                )

        # Trier par date de création (plus récents en premier)
        query = query.order_by(QueueModel.created_at.desc())

        # Compter le total
        total_count = query.count()

        # Appliquer pagination
        results = query.offset(offset).limit(limit).all()

        # Formater les résultats
        import json
        items = []
        for item in results:
            # Parser les détails d'erreur si présents
            error_details = None
            if item.error_details:
                try:
                    error_details = json.loads(item.error_details)
                except:
                    error_details = {"raw": item.error_details}

            items.append({
                "request_id": item.id,
                "event_type": item.event_type,
                "status": item.status.value,
                "priority": item.priority,
                "retry_count": item.retry_count,
                "max_retries": item.max_retries,
                "created_at": item.created_at.isoformat() if item.created_at else None,
                "processed_at": item.processed_at.isoformat() if item.processed_at else None,
                "next_retry_at": item.next_retry_at.isoformat() if item.next_retry_at else None,
                "error_details": error_details
            })

        return {
            "total": total_count,
            "limit": limit,
            "offset": offset,
            "status_filter": status,
            "items": items
        }

@app.post("/admin/cleanup")
def cleanup_old_records(days: int = 30):
    """
    Nettoie les anciens enregistrements
    Nécessite authentification en production
    """
    deleted = db.cleanup_old_records(days)
    return {
        "deleted_count": deleted,
        "message": f"Cleaned up records older than {days} days"
    }

@app.delete("/request/{request_id}")
def cancel_request(request_id: int):
    """
    Annule une requête en attente
    """
    request = db.get_request_by_id(request_id)

    if not request:
        raise HTTPException(status_code=404, detail="Request not found")

    if request.status not in [RequestStatus.PENDING, RequestStatus.RETRYING]:
        raise HTTPException(
            status_code=400,
            detail=f"Cannot cancel request with status: {request.status.value}"
        )

    db.update_request_status(request_id, RequestStatus.FAILED)

    return {"message": "Request cancelled"}

@app.get("/destinations/{destination_id}/requests")
def get_destination_requests(
    destination_id: str,
    status: Optional[str] = None,
    limit: int = 50,
    offset: int = 0
):
    """
    Liste toutes les requêtes envoyées vers une destination spécifique
    avec statistiques de succès/échec

    Args:
        destination_id: ID de la destination (ex: fne_cdi_dev, httpbin, etc.)
        status: Filtrer par statut de la requête (pending, processing, completed, failed, retrying)
        limit: Nombre maximum de requêtes (défaut: 50, max: 200)
        offset: Pagination offset
    """
    from db_models import QueueModel, APIResponseModel

    # Vérifier que la destination existe
    if destination_id not in API_DESTINATIONS:
        raise HTTPException(
            status_code=404,
            detail=f"Destination not found: {destination_id}. Available: {list(API_DESTINATIONS.keys())}"
        )

    # Limiter le nombre de résultats
    limit = min(limit, 200)

    with db.get_session() as session:
        # Trouver toutes les requêtes qui ont été envoyées à cette destination
        # via la table api_responses
        query = session.query(QueueModel).join(
            APIResponseModel,
            QueueModel.id == APIResponseModel.queue_id
        ).filter(
            APIResponseModel.destination == destination_id
        )

        # Filtrer par statut si spécifié
        if status:
            try:
                status_enum = RequestStatus(status)
                query = query.filter(QueueModel.status == status_enum)
            except ValueError:
                raise HTTPException(
                    status_code=400,
                    detail=f"Invalid status: {status}. Valid values: {[s.value for s in RequestStatus]}"
                )

        # Supprimer les doublons (une requête peut avoir plusieurs réponses)
        query = query.distinct()

        # Trier par date de création (plus récents en premier)
        query = query.order_by(QueueModel.created_at.desc())

        # Compter le total
        total_count = query.count()

        # Appliquer pagination
        results = query.offset(offset).limit(limit).all()

        # Formater les résultats avec les réponses API
        items = []
        for item in results:
            # Récupérer la réponse API pour cette destination
            api_response = session.query(APIResponseModel).filter(
                APIResponseModel.queue_id == item.id,
                APIResponseModel.destination == destination_id
            ).order_by(APIResponseModel.timestamp.desc()).first()

            items.append({
                "request_id": item.id,
                "event_type": item.event_type,
                "status": item.status.value,
                "priority": item.priority,
                "retry_count": item.retry_count,
                "max_retries": item.max_retries,
                "created_at": item.created_at.isoformat() if item.created_at else None,
                "processed_at": item.processed_at.isoformat() if item.processed_at else None,
                "api_response": {
                    "status_code": api_response.status_code,
                    "error": api_response.error,
                    "duration_ms": api_response.duration_ms,
                    "timestamp": api_response.timestamp.isoformat() if api_response.timestamp else None
                } if api_response else None
            })

        # Calculer les statistiques manuellement pour éviter les problèmes SQL
        all_responses = session.query(APIResponseModel).filter(
            APIResponseModel.destination == destination_id
        ).all()

        total_requests = len(all_responses)
        successful = sum(1 for r in all_responses if r.status_code and 200 <= r.status_code < 300)
        errors = sum(1 for r in all_responses if r.status_code and r.status_code >= 400)
        exceptions = sum(1 for r in all_responses if r.error is not None)
        durations = [r.duration_ms for r in all_responses if r.duration_ms is not None]
        avg_duration = sum(durations) / len(durations) if durations else 0

        return {
            "destination": {
                "id": destination_id,
                "name": API_DESTINATIONS[destination_id]["name"],
                "base_url": API_DESTINATIONS[destination_id]["base_url"]
            },
            "statistics": {
                "total_requests": total_requests,
                "successful": successful,
                "errors": errors,
                "exceptions": exceptions,
                "avg_duration_ms": round(avg_duration, 2)
            },
            "pagination": {
                "total": total_count,
                "limit": limit,
                "offset": offset,
                "status_filter": status
            },
            "items": items
        }

@app.get("/destinations/{destination_id}/stats")
def get_destination_stats(destination_id: str):
    """
    Statistiques détaillées pour une destination spécifique

    Args:
        destination_id: ID de la destination (ex: fne_cdi_dev, httpbin, etc.)
    """
    from db_models import APIResponseModel
    from collections import Counter

    # Vérifier que la destination existe
    if destination_id not in API_DESTINATIONS:
        raise HTTPException(
            status_code=404,
            detail=f"Destination not found: {destination_id}. Available: {list(API_DESTINATIONS.keys())}"
        )

    with db.get_session() as session:
        # Récupérer toutes les réponses pour cette destination
        all_responses = session.query(APIResponseModel).filter(
            APIResponseModel.destination == destination_id
        ).all()

        # Calculer les statistiques manuellement
        total_requests = len(all_responses)
        successful = sum(1 for r in all_responses if r.status_code and 200 <= r.status_code < 300)
        client_errors = sum(1 for r in all_responses if r.status_code and 400 <= r.status_code < 500)
        server_errors = sum(1 for r in all_responses if r.status_code and r.status_code >= 500)
        exceptions = sum(1 for r in all_responses if r.error is not None)

        durations = [r.duration_ms for r in all_responses if r.duration_ms is not None]
        avg_duration = sum(durations) / len(durations) if durations else 0
        min_duration = min(durations) if durations else 0
        max_duration = max(durations) if durations else 0

        # Breakdown des codes de statut
        status_codes_list = [r.status_code if r.status_code else None for r in all_responses]
        status_code_counter = Counter(status_codes_list)
        status_code_breakdown = {
            str(code) if code is not None else "null": count
            for code, count in status_code_counter.items()
        }

        success_rate = round((successful / total_requests * 100), 2) if total_requests > 0 else 0

        return {
            "destination": {
                "id": destination_id,
                "name": API_DESTINATIONS[destination_id]["name"],
                "base_url": API_DESTINATIONS[destination_id]["base_url"],
                "timeout": API_DESTINATIONS[destination_id]["timeout"],
                "max_retries": API_DESTINATIONS[destination_id]["max_retries"]
            },
            "statistics": {
                "total_requests": total_requests,
                "successful": successful,
                "client_errors": client_errors,
                "server_errors": server_errors,
                "exceptions": exceptions,
                "success_rate": success_rate
            },
            "performance": {
                "avg_duration_ms": round(avg_duration, 2),
                "min_duration_ms": round(min_duration, 2),
                "max_duration_ms": round(max_duration, 2)
            },
            "status_codes": status_code_breakdown
        }

if __name__ == "__main__":
    import uvicorn
    uvicorn.run(app, host="0.0.0.0", port=8000)