"""
Module d'intégration email_sender pour le middleware
Gère l'envoi d'emails de facture et d'alertes admin
"""
import sys
from pathlib import Path
from typing import Dict, Any, Optional
from datetime import datetime
from app.logger import get_logger


try:
    from email_sender import EmailSender
    from email_sender.qrcode_generator import generate_qrcode
except ImportError:
    # Ajouter le dossier parent au path pour importer email_sender. Utile en dev local
    sys.path.insert(0, str(Path(__file__).parent.parent.parent))
    from email_sender import EmailSender
    from email_sender.qrcode_generator import generate_qrcode

logger = get_logger("middleware.email_notifications")

class MiddlewareEmailNotifications:
    """
    Gestionnaire d'envoi d'emails pour le middleware FNE
    Utilise le module email_sender générique
    """

    def __init__(self, config: Dict[str, Any]):
        """
        Initialise le gestionnaire d'emails

        Args:
            config: Configuration complète (depuis config.toml)
        """
        self.config = config
        self.sender = EmailSender.from_config(config)

        logger.debug("middleware_email_sender_initialized", config=config)
        
        # Récupérer les paramètres de notification
        self.notifications_config = config.get("email", {}).get("notifications", {})
        self.admin_email = self.notifications_config.get("admin_email")

        # Récupérer les template IDs (Brevo uniquement)
        email_config = config.get("email", {})
        brevo_config = email_config.get("brevo", {})
        self.invoice_template_id = brevo_config.get("invoice_template_id")
        self.use_template = self.invoice_template_id is not None and email_config.get("provider") == "brevo"

        self.provider_config = self.get_provider_config(email_config)

        # Configuration mode développement (depuis fne_cdi.dev)
        fne_cdi_config = config.get("fne_cdi", {})
        self.dev_config = fne_cdi_config.get("dev", {})
        self.dev_mode = self.dev_config.get("enabled", False)
        self.dev_redirect_emails = self.dev_config.get("redirect_emails", [])
        self.dev_subject_prefix = self.dev_config.get("subject_prefix", "[DEV] ")

        # Debug: log la config fne_cdi reçue
        logger.info(
            "email_notifications_dev_config",
            fne_cdi_keys=list(fne_cdi_config.keys()) if fne_cdi_config else [],
            dev_config=self.dev_config,
            dev_mode=self.dev_mode,
            redirect_emails=self.dev_redirect_emails
        )

        if self.dev_mode:
            logger.info("dev_mode_enabled", redirect_emails=self.dev_redirect_emails)

    def get_provider_config(self, email_config: Dict[str, Any]) -> Dict[str, Any]:
        provider = email_config.get("provider")
        if not provider:
            return {}

        return email_config.get(provider, {})

    def send_invoice_to_client(
        self,
        client_email: str,
        client_name: str,
        invoice_number: str,
        invoice_date: str,
        invoice_amount: float,
        download_url: str
    ) -> Dict[str, Any]:
        """
        Envoie la facture au client avec QR code ou via template

        Args:
            client_email: Email du client
            client_name: Nom du client
            invoice_number: Numéro de facture
            invoice_date: Date de facture (format JJ/MM/AAAA)
            invoice_amount: Montant TTC
            download_url: URL complète de téléchargement depuis response_data.token
                         (ex: "http://54.247.95.108/fr/verification/019a799e...")

        Returns:
            Résultat de l'envoi
        """

        logger.debug("hello send_invoice_to_client",
                     client_email=client_email,
                     dev_mode=self.dev_mode,
                     redirect_emails=self.dev_redirect_emails)

        # Si un template Brevo est configuré, l'utiliser
        if self.use_template:
            template_params = {
                "firstname": client_name,
                "order_number": invoice_number,
                "order_date": invoice_date,
                "invoice_url": download_url,
                "shop_name": self.provider_config.get("shop_name"),
                "shop_address": self.provider_config.get("shop_address"),
                "shop_email": self.provider_config.get("shop_email", ""),
                "shop_phone": self.provider_config.get("shop_phone", ""),
                "shop_logo_url": self.provider_config.get("shop_logo_url", ""),
                "shop_url": self.provider_config.get("shop_url", ""),
                "facebook_url": self.provider_config.get("facebook_url", ""),
                "instagram_url": self.provider_config.get("instagram_url", ""),
                "website_url": self.provider_config.get("website_url", "")
            }

            return self.sender.send_template(
                to_email=client_email,
                template_id=self.invoice_template_id,
                template_params=template_params
                # cc=dev_cc_emails
            )

        # Sinon, utiliser le HTML embarqué
        # Générer le QR code
        # qrcode_bytes = generate_qrcode(download_url, size=300)

        # Construire le corps HTML de l'email
        html_body = self._build_invoice_email_html(
            client_name=client_name,
            invoice_number=invoice_number,
            invoice_date=invoice_date,
            invoice_amount=invoice_amount,
            download_url=download_url
        )

        # Envoyer l'email avec QR code inline
        subject = f"Votre facture {invoice_number}"
        if self.dev_mode:
            subject = f"{self.dev_subject_prefix}{subject}"

        return self.sender.send_html(
            to_email=client_email,
            subject=subject,
            html_body=html_body
        )
        #     inline_images={"qrcode": qrcode_bytes}
        # )

    def send_admin_alert(
        self,
        warnings: Optional[list] = None,
        balance_funds: Optional[float] = None,
        invoice_number: Optional[str] = None,
        invoice_date: Optional[str] = None,
        invoice_amount: Optional[float] = None,
        client_name: Optional[str] = None,
        request_id: Optional[int] = None,
        download_url: Optional[str] = None
    ) -> Dict[str, Any]:
        """
        Envoie une alerte admin

        Args:
            warnings: Liste des avertissements depuis response_data.warning
            balance_funds: Solde restant depuis response_data.balance_funds
            invoice_number: Numéro de facture
            invoice_date: Date de facture
            invoice_amount: Montant
            client_name: Nom du client
            request_id: ID de la requête middleware
            download_url: URL complète de téléchargement (optionnel)

        Returns:
            Résultat de l'envoi

        Raises:
            ValueError: Si admin_email n'est pas configuré ou si ni warnings ni balance_funds fournis
        """
        if not self.admin_email:
            raise ValueError("admin_email non configuré dans [email.notifications]")

        if not warnings and balance_funds is None:
            raise ValueError("Au moins warnings ou balance_funds doit être fourni")

        # Construire le corps HTML
        html_body = self._build_admin_alert_html(
            warnings=warnings,
            balance_funds=balance_funds,
            invoice_number=invoice_number,
            invoice_date=invoice_date,
            invoice_amount=invoice_amount,
            client_name=client_name,
            request_id=request_id,
            download_url=download_url
        )

        # Construire le sujet
        subject_parts = []
        if warnings:
            subject_parts.append(f"{len(warnings)} avertissement(s)")
        if balance_funds is not None:
            if balance_funds < 1000:
                subject_parts.append(f"SOLDE CRITIQUE: {balance_funds} FCFA")
            else:
                subject_parts.append(f"Solde: {balance_funds} FCFA")

        subject = f"[FNE-CDI] {' - '.join(subject_parts)}"

        return self.sender.send_html(
            to_email=self.admin_email,
            subject=subject,
            html_body=html_body
        )

    def _build_invoice_email_html(
        self,
        client_name: str,
        invoice_number: str,
        invoice_date: str,
        invoice_amount: float,
        download_url: str,
        original_email: Optional[str] = None
    ) -> str:
        """Construit le HTML de l'email client"""
        # Bannière mode développement
        dev_banner = ""
        if self.dev_mode and original_email:
            dev_banner = f"""
    <div style="background-color: #ff6b6b; color: white; padding: 10px; text-align: center; font-weight: bold;">
         MODE DÉVELOPPEMENT - Email destiné à : {original_email}
    </div>"""

        return f"""<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <style>
        body {{ font-family: Arial, sans-serif; line-height: 1.6; color: #333; max-width: 600px; margin: 0 auto; padding: 20px; }}
        .header {{ background-color: #2c3e50; color: white; padding: 20px; text-align: center; border-radius: 5px 5px 0 0; }}
        .content {{ background-color: #f9f9f9; padding: 30px; border: 1px solid #ddd; border-top: none; }}
        .qrcode-section {{ text-align: center; margin: 30px 0; padding: 20px; background-color: white; border-radius: 5px; }}
        .qrcode-section img {{ max-width: 300px; height: auto; }}
        .download-button {{ display: inline-block; background-color: #3498db; color: white; padding: 15px 30px; text-decoration: none; border-radius: 5px; margin: 20px 0; font-weight: bold; }}
        .invoice-details {{ background-color: white; padding: 15px; border-radius: 5px; margin: 20px 0; }}
        .invoice-details table {{ width: 100%; }}
        .invoice-details td {{ padding: 5px; }}
        .invoice-details td:first-child {{ font-weight: bold; width: 40%; }}
        .footer {{ margin-top: 30px; padding-top: 20px; border-top: 1px solid #ddd; font-size: 12px; color: #777; text-align: center; }}
    </style>
</head>
<body>{dev_banner}
    <div class="header">
        <h1>Facture Électronique</h1>
        <p>{self.provider_config.get('shop_name', 'GCV6 Gestion Commerciale')}</p>
    </div>
    <div class="content">
        <p>Bonjour {client_name},</p>
        <p>Votre facture électronique est maintenant disponible.</p>
        <div class="invoice-details">
            <table>
                <tr><td>Numéro de facture:</td><td>{invoice_number}</td></tr>
                <tr><td>Date:</td><td>{invoice_date}</td></tr>
                <tr><td>Montant TTC:</td><td>{invoice_amount:,.0f} FCFA</td></tr>
            </table>
        </div>
        <div class="qrcode-section">
            <h3>Scanner pour télécharger</h3>
            <img src="cid:qrcode" alt="QR Code">
            <p style="font-size: 12px; color: #666;">Scannez ce code pour accéder à votre facture</p>
        </div>
        <div style="text-align: center;">
            <a href="{download_url}" class="download-button">Télécharger la facture</a>
        </div>
        <p style="margin-top: 30px; font-size: 14px;">Cette facture est conforme à la réglementation fiscale.</p>
    </div>
    <div class="footer">
        <p>{self.provider_config.get('shop_name', 'GCV6 Gestion Commerciale')}<br>
        {self.provider_config.get('shop_address', '')}<br>
        {self.provider_config.get('shop_phone', '')} - {self.provider_config.get('shop_email', '')}</p>
        <p style="margin-top: 10px;">Email généré automatiquement.</p>
    </div>
</body>
</html>"""

    def _build_admin_alert_html(
        self,
        warnings: Optional[list],
        balance_funds: Optional[float],
        invoice_number: Optional[str],
        invoice_date: Optional[str],
        invoice_amount: Optional[float],
        client_name: Optional[str],
        request_id: Optional[int],
        download_url: Optional[str]
    ) -> str:
        """Construit le HTML de l'alerte admin"""
        warning_type = "error" if balance_funds is not None and balance_funds < 1000 else "warning"

        warnings_html = ""
        if warnings:
            warnings_items = "".join([f"<li>{w}</li>" for w in warnings])
            warnings_html = f"""
        <div class="alert-box">
            <h3>Avertissements</h3>
            <ul>{warnings_items}</ul>
        </div>"""

        balance_html = ""
        if balance_funds is not None:
            balance_class = "error" if balance_funds < 1000 else ""
            balance_msg = ""
            if balance_funds < 1000:
                balance_msg = '<p style="color: #e74c3c; font-weight: bold;">⚠️ ATTENTION: Solde faible!</p>'
            elif balance_funds < 5000:
                balance_msg = '<p style="color: #f39c12;">Le solde approche du seuil critique.</p>'

            balance_html = f"""
        <div class="alert-box {balance_class}">
            <h3>Solde du compte</h3>
            <p style="font-size: 24px; font-weight: bold; margin: 10px 0;">{balance_funds:,.0f} FCFA</p>
            {balance_msg}
        </div>"""

        download_link_html = ""
        if download_url:
            download_link_html = f'<p><strong>Lien:</strong><br><a href="{download_url}">{download_url}</a></p>'

        return f"""<!DOCTYPE html>
<html lang="fr">
<head>
    <meta charset="UTF-8">
    <style>
        body {{ font-family: Arial, sans-serif; line-height: 1.6; color: #333; max-width: 600px; margin: 0 auto; padding: 20px; }}
        .header {{ background-color: {{'#e74c3c' if warning_type == 'error' else '#f39c12'}}; color: white; padding: 20px; text-align: center; border-radius: 5px 5px 0 0; }}
        .content {{ background-color: #f9f9f9; padding: 30px; border: 1px solid #ddd; border-top: none; }}
        .alert-box {{ background-color: #fff3cd; border-left: 4px solid #f39c12; padding: 15px; margin: 20px 0; }}
        .alert-box.error {{ background-color: #f8d7da; border-left-color: #e74c3c; }}
        .details {{ background-color: white; padding: 15px; border-radius: 5px; margin: 20px 0; font-family: monospace; font-size: 12px; }}
        .details table {{ width: 100%; border-collapse: collapse; }}
        .details td {{ padding: 8px; border-bottom: 1px solid #eee; }}
        .details td:first-child {{ font-weight: bold; width: 30%; color: #555; }}
        .footer {{ margin-top: 30px; padding-top: 20px; border-top: 1px solid #ddd; font-size: 12px; color: #777; text-align: center; }}
    </style>
</head>
<body>
    <div class="header">
        <h1>Alerte FNE-CDI</h1>
    </div>
    <div class="content">
        <p>Bonjour Administrateur,</p>
        <p>Une alerte a été déclenchée lors du traitement d'une facture.</p>
        {warnings_html}
        {balance_html}
        <div class="details">
            <h3>Détails de la facture</h3>
            <table>
                <tr><td>Numéro:</td><td>{invoice_number or 'N/A'}</td></tr>
                <tr><td>Date:</td><td>{invoice_date or 'N/A'}</td></tr>
                <tr><td>Montant:</td><td>{f"{invoice_amount:,.0f} FCFA" if invoice_amount else "N/A"}</td></tr>
                <tr><td>Client:</td><td>{client_name or 'N/A'}</td></tr>
                <tr><td>Request ID:</td><td>{request_id or 'N/A'}</td></tr>
                <tr><td>Timestamp:</td><td>{datetime.now().strftime('%Y-%m-%d %H:%M:%S')}</td></tr>
            </table>
        </div>
        {download_link_html}
    </div>
    <div class="footer">
        <p>GCV Middleware - Système de facturation électronique</p>
    </div>
</body>
</html>"""

    def verify(self) -> bool:
        """Vérifie la configuration email"""
        return self.sender.verify_configuration()
