"""
Formatters pour les APIs de test
"""
from typing import Dict, Any
from datetime import datetime
from zoneinfo import ZoneInfo

# Timezone par défaut
TZ_MARTINIQUE = ZoneInfo("America/Martinique")


def format_product(data: Dict[str, Any], destination: str) -> Dict[str, Any]:
    """
    Formate les données de produit pour restful-api.dev

    Args:
        data: Données du produit
        destination: ID de la destination

    Returns:
        Données formatées
    """
    if destination == "restful_api":
        return {
            "name": data.get("name", "Unknown Product"),
            "data": {
                "price": data.get("price", 0),
                "color": data.get("color", "N/A"),
                "category": data.get("category", "general"),
                "stock": data.get("stock", 0),
                "created_at": datetime.now(TZ_MARTINIQUE).isoformat()
            }
        }

    elif destination == "httpbin":
        return {
            "event": "product_created",
            "product": data,
            "timestamp": datetime.now(TZ_MARTINIQUE).isoformat(),
            "source": "middleware"
        }

    return data


def format_user(data: Dict[str, Any], destination: str) -> Dict[str, Any]:
    """
    Formate les données utilisateur pour Reqres.in

    Args:
        data: Données utilisateur
        destination: ID de la destination

    Returns:
        Données formatées
    """
    if destination == "reqres":
        return {
            "name": data.get("name", "Unknown"),
            "job": data.get("job", "user"),
            "email": data.get("email")
        }

    elif destination == "httpbin":
        return {
            "event": "user_created",
            "user": {
                "name": data.get("name"),
                "email": data.get("email"),
                "job": data.get("job")
            },
            "metadata": {
                "timestamp": datetime.now(TZ_MARTINIQUE).isoformat(),
                "source": "middleware",
                "version": "1.0"
            }
        }

    return data


def format_broadcast(data: Dict[str, Any], destination: str) -> Dict[str, Any]:
    """
    Format pour broadcast vers toutes les APIs
    Adapte les données selon la destination

    Args:
        data: Données à broadcaster
        destination: ID de la destination

    Returns:
        Données formatées
    """
    base_data = {
        "message": data.get("message", "Test broadcast"),
        "timestamp": datetime.now(TZ_MARTINIQUE).isoformat()
    }

    if destination == "restful_api":
        return {
            "name": f"Broadcast {datetime.now(TZ_MARTINIQUE).strftime('%H:%M:%S')}",
            "data": {
                **base_data,
                "destination": "restful_api",
                "payload": data
            }
        }

    elif destination == "reqres":
        return {
            "name": data.get("message", "Broadcast"),
            "job": "broadcast_event"
        }

    elif destination == "httpbin":
        return {
            "event": "broadcast",
            **base_data,
            "original_data": data
        }

    elif destination == "jsonplaceholder":
        return {
            "title": data.get("message", "Broadcast"),
            "body": str(data),
            "userId": 1
        }

    return base_data


def format_debug(data: Dict[str, Any], destination: str) -> Dict[str, Any]:
    """
    Format pour debugging via HTTPBin
    Envoie toutes les données brutes avec métadonnées

    Args:
        data: Données à debugger
        destination: ID de la destination

    Returns:
        Données formatées avec métadonnées de debug
    """
    return {
        "debug_event": True,
        "timestamp": datetime.now(TZ_MARTINIQUE).isoformat(),
        "raw_data": data,
        "data_keys": list(data.keys()),
        "data_types": {k: type(v).__name__ for k, v in data.items()},
        "middleware_info": {
            "version": "1.0",
            "destination": destination
        }
    }


def format_generic(data: Dict[str, Any], destination: str) -> Dict[str, Any]:
    """
    Format générique pour JSONPlaceholder

    Args:
        data: Données à formater
        destination: ID de la destination

    Returns:
        Données formatées
    """
    if destination == "jsonplaceholder":
        return {
            "title": data.get("title", "Generic Event"),
            "body": data.get("body", str(data)),
            "userId": data.get("userId", 1)
        }

    return {
        "data": data,
        "metadata": {
            "processed_at": datetime.now(TZ_MARTINIQUE).isoformat(),
            "middleware_version": "1.0",
            "destination": destination
        }
    }


def format_order(data: Dict[str, Any], destination: str) -> Dict[str, Any]:
    """
    Formate les données de commande (exemple avancé)

    Args:
        data: Données de commande
        destination: ID de la destination

    Returns:
        Données formatées
    """
    if destination == "restful_api":
        return {
            "name": f"Order {data.get('order_id', 'N/A')}",
            "data": {
                "order_id": data.get("order_id"),
                "customer": data.get("customer_email"),
                "total": data.get("total", 0),
                "currency": data.get("currency", "EUR"),
                "items_count": len(data.get("items", [])),
                "created_at": datetime.now(TZ_MARTINIQUE).isoformat()
            }
        }

    elif destination == "httpbin":
        return {
            "event": "order_created",
            "order": data,
            "timestamp": datetime.now(TZ_MARTINIQUE).isoformat()
        }

    return data


# Export des formatters disponibles
FORMATTERS = {
    "format_product": format_product,
    "format_user": format_user,
    "format_broadcast": format_broadcast,
    "format_debug": format_debug,
    "format_generic": format_generic,
    "format_order": format_order
}
