"""
Implémentation de la destination APIs de test
"""
from typing import Dict, Any, Callable
from app.destinations.base import BaseDestination, APIConfig, RouteConfig, RouteType
from .formatters import FORMATTERS


class TestAPIsDestination(BaseDestination):
    """
    Destination pour les APIs de test publiques
    Utilisée pour le développement et les tests du middleware
    """

    def get_name(self) -> str:
        """Nom unique de la destination"""
        return "test_apis"

    def get_description(self) -> str:
        """Description de la destination"""
        return "APIs de test publiques gratuites pour développement"

    def get_version(self) -> str:
        """Version de l'implémentation"""
        return "1.0.0"

    def is_enabled(self) -> bool:
        """Les APIs de test sont toujours activées"""
        return True

    def get_api_configs(self) -> Dict[str, APIConfig]:
        """Configuration des APIs de test"""
        return {
            "restful_api": APIConfig(
                name="Restful-API.dev",
                base_url="https://api.restful-api.dev/objects",
                headers={"Content-Type": "application/json"},
                timeout=15,
                max_retries=3,
                description="API de test gratuite pour développement",
                enabled=True
            ),
            "reqres": APIConfig(
                name="Reqres.in",
                base_url="https://reqres.in/api/users",
                headers={"Content-Type": "application/json"},
                timeout=10,
                max_retries=3,
                description="API de test gratuite",
                enabled=True
            ),
            "httpbin": APIConfig(
                name="HTTPBin Echo",
                base_url="https://httpbin.org/post",
                headers={
                    "Content-Type": "application/json",
                    "User-Agent": "Middleware-API/1.0"
                },
                timeout=10,
                max_retries=2,
                description="API de test pour debugging HTTP",
                enabled=True
            ),
            "jsonplaceholder": APIConfig(
                name="JSONPlaceholder",
                base_url="https://jsonplaceholder.typicode.com/posts",
                headers={"Content-Type": "application/json"},
                timeout=10,
                max_retries=3,
                description="API de test gratuite",
                enabled=True
            )
        }

    def get_routes(self) -> Dict[str, RouteConfig]:
        """Configuration des routes de test"""
        return {
            "create_product": RouteConfig(
                event_type="create_product",
                route_type=RouteType.SIMPLE,
                destinations=["restful_api"],
                formatter="format_product",
                description="Test - création d'un produit",
                enabled=True
            ),
            "create_user": RouteConfig(
                event_type="create_user",
                route_type=RouteType.MULTI,
                destinations=["reqres", "httpbin"],
                formatter="format_user",
                description="Test - création utilisateur multi-destinations",
                enabled=True
            ),
            "test_broadcast": RouteConfig(
                event_type="test_broadcast",
                route_type=RouteType.MULTI,
                destinations=["restful_api", "reqres", "httpbin", "jsonplaceholder"],
                formatter="format_broadcast",
                description="Test - envoi vers toutes les APIs",
                enabled=True
            ),
            "expensive_product": RouteConfig(
                event_type="expensive_product",
                route_type=RouteType.CONDITIONAL,
                destinations=["restful_api", "httpbin"],
                formatter="format_product",
                condition=lambda data: data.get("price", 0) > 100,
                description="Test - envoi conditionnel si prix > 100",
                enabled=True
            ),
            "debug_event": RouteConfig(
                event_type="debug_event",
                route_type=RouteType.SIMPLE,
                destinations=["httpbin"],
                formatter="format_debug",
                description="Debug - inspection des données via HTTPBin",
                enabled=True
            ),
            "generic_event": RouteConfig(
                event_type="generic_event",
                route_type=RouteType.SIMPLE,
                destinations=["jsonplaceholder"],
                formatter="format_generic",
                description="Test - événement générique",
                enabled=True
            )
        }

    def get_formatters(self) -> Dict[str, Callable[[Dict[str, Any], str], Dict[str, Any]]]:
        """Formatters pour les APIs de test"""
        return FORMATTERS

    def get_validators(self) -> Dict[str, Callable[[Dict[str, Any]], Any]]:
        """Pas de validateurs pour les APIs de test"""
        return {}
