"""
Validateurs pour la destination FNE-CDI
Réexporte les validateurs depuis les modules internes
"""
from .base import ValidationResult, InvoiceItem, CustomTax
from .fne_invoice import FNEInvoiceValidator, FNEInvoiceData, FNECancelInvoiceData
from .gcv_invoice import validate_gcvx_invoice

__all__ = [
    'ValidationResult',
    'InvoiceItem',
    'CustomTax',
    'FNEInvoiceValidator',
    'FNEInvoiceData',
    'FNECancelInvoiceData',
    'validate_fne_invoice',
    'validate_gcvx_invoice',
    'validate_fne_cancel',
    'validate_fne_quote'
]


def validate_fne_invoice(data: dict) -> ValidationResult:
    """
    Valide une facture FNE (format de sortie vers l'API)

    Args:
        data: Données de la facture au format FNE à valider

    Returns:
        ValidationResult avec les erreurs éventuelles
    """
    return FNEInvoiceValidator.validate(data)


def validate_fne_cancel(data: dict) -> ValidationResult:
    """
    Valide une demande d'annulation de facture FNE

    Args:
        data: Données d'annulation à valider

    Returns:
        ValidationResult avec les erreurs éventuelles
    """
    return FNEInvoiceValidator.validate_cancel(data)


def validate_fne_quote(data: dict) -> ValidationResult:
    """
    Valide un devis FNE (même schéma qu'une facture)

    Args:
        data: Données du devis à valider

    Returns:
        ValidationResult avec les erreurs éventuelles
    """
    return FNEInvoiceValidator.validate(data)


# Export du registre des validateurs
VALIDATORS = {
    "validate_gcvx_invoice": validate_gcvx_invoice,  # Validateur des données entrantes GCVX
    "validate_fne_invoice": validate_fne_invoice,    # Validateur des données sortantes FNE
    "validate_fne_cancel": validate_fne_cancel,
    "validate_fne_quote": validate_fne_quote
}
