"""
Formatters pour la destination FNE-CDI
Convertit les données GCVX en format FNE via le mapper
"""
from typing import Dict, Any
from app.logger import get_logger
from .mapper import format_gcvx_to_fne
from .config import FNE_CDI_ESTABLISHMENT, FNE_CDI_POS

logger = get_logger(__name__)


def format_fne_invoice(data: Dict[str, Any], destination: str) -> Dict[str, Any]:
    """
    Formate les données de facture GCVX pour l'API FNE-CDI

    Cette fonction reçoit des données au format GCVX et les convertit en format FNE

    Args:
        data: Données de la facture au format GCVX (déjà validées)
        destination: ID de la destination (ex: "fne_cdi_dev")

    Returns:
        Données formatées pour l'API FNE
    """
    # Convertir les données GCVX en format FNE via le mapper
    logger.info("GCVX data to FNE-CDI", destination=destination, fac_num=data.get("fac_num"))

    fne_data = format_gcvx_to_fne(data, destination)

    # Injecter establishment et pointOfSale depuis la configuration
    fne_data["establishment"] = FNE_CDI_ESTABLISHMENT
    fne_data["pointOfSale"] = FNE_CDI_POS

    logger.info("Données FNE mappées", fne_data=fne_data)

    return fne_data
    #return None  # FIXME: temporaire pour debugger


def format_fne_cancel(data: Dict[str, Any], destination: str) -> Dict[str, Any]:
    """
    Formate les données d'annulation de facture pour l'API FNE-CDI

    Args:
        data: Données d'annulation validées
        destination: ID de la destination

    Returns:
        Données formatées pour l'API FNE
    """
    # L'API FNE attend les données brutes sans enveloppe
    return data


def format_fne_quote(data: Dict[str, Any], destination: str) -> Dict[str, Any]:
    """
    Formate les données de devis (proforma) pour l'API FNE-CDI

    Args:
        data: Données du devis validées
        destination: ID de la destination

    Returns:
        Données formatées pour l'API FNE
    """
    # L'API FNE attend les données brutes sans enveloppe
    return data


def format_fne_credit_note(data: Dict[str, Any], destination: str) -> Dict[str, Any]:
    """
    Formate les données d'avoir (note de crédit) pour l'API FNE-CDI

    Args:
        data: Données de l'avoir validées
        destination: ID de la destination

    Returns:
        Données formatées pour l'API FNE
    """
    # L'API FNE attend les données brutes sans enveloppe
    return data


# Export des formatters disponibles
FORMATTERS = {
    "format_fne_invoice": format_fne_invoice,
    "format_fne_cancel": format_fne_cancel,
    "format_fne_quote": format_fne_quote,
    "format_fne_credit_note": format_fne_credit_note
}
