"""
Implémentation de la destination FNE-CDI
"""
__VERSION__="1.0.0"

from typing import Dict, Any, Callable
from app.destinations.base import BaseDestination, APIConfig, RouteConfig, RouteType
from .config import (
    FNE_CDI_TOKEN,
    FNE_CDI_ESTABLISHMENT,
    FNE_CDI_POS,
    FNE_CDI_BASE_URL,
    FNE_CDI_TIMEOUT,
    FNE_CDI_MAX_RETRIES,
    is_configured
)
from .formatters import FORMATTERS
from .validators import VALIDATORS


class FNECDIDestination(BaseDestination):
    """
    Destination pour l'API FNE-CDI (Côte d'Ivoire)
    Gestion de la facturation électronique conforme à la norme FNE
    """

    def get_name(self) -> str:
        """Nom unique de la destination"""
        return "fne_cdi"

    def get_description(self) -> str:
        """Description de la destination"""
        return "API de signature électronique des factures FNE-CDI (Côte d'Ivoire)"

    def get_version(self) -> str:
        """Version de l'implémentation"""
        return __VERSION__

    def is_enabled(self) -> bool:
        """Vérifie si la destination est correctement configurée"""
        return is_configured()

    def get_api_configs(self) -> Dict[str, APIConfig]:
        """Configuration des APIs FNE"""
        return {
            "fne_cdi_dev": APIConfig(
                name="FNE-CDI DEV",
                base_url=FNE_CDI_BASE_URL,
                headers={
                    "Content-Type": "application/json",
                    "Authorization": f"Bearer {FNE_CDI_TOKEN}"
                },
                timeout=FNE_CDI_TIMEOUT,
                max_retries=FNE_CDI_MAX_RETRIES,
                description="API de signature électronique des factures (depuis config.toml)",
                enabled=is_configured(),
                extra={
                    "establishment": FNE_CDI_ESTABLISHMENT,
                    "pos": FNE_CDI_POS
                }
            ),
            # Prêt pour ajouter la prod plus tard
            # "fne_cdi_prod": APIConfig(...)
        }

    def get_routes(self) -> Dict[str, RouteConfig]:
        """Configuration des routes FNE"""
        return {
            "fnedev_invoice_sign": RouteConfig(
                event_type="fnedev_invoice_sign",
                route_type=RouteType.SIMPLE,
                destinations=["fne_cdi_dev"],
                formatter="format_fne_invoice",
                validator="validate_gcvx_invoice",  # Valide les données GCVX entrantes
                description="Signature électronique d'une facture GCVX via FNE-CDI",
                enabled=is_configured()
            ),
            # "fnedev_invoice_cancel": RouteConfig(
            #     event_type="fnedev_invoice_cancel",
            #     route_type=RouteType.SIMPLE,
            #     destinations=["fne_cdi_dev"],
            #     formatter="format_fne_cancel",
            #     validator="validate_fne_cancel",
            #     description="Annulation d'une facture via FNE-CDI",
            #     enabled=is_configured()
            # ),
            # Routes commentées en attente d'implémentation
            # "create_invoice": RouteConfig(...),
            # "create_quote": RouteConfig(...),
            # "create_credit_note": RouteConfig(...),
        }

    def get_formatters(self) -> Dict[str, Callable[[Dict[str, Any], str], Dict[str, Any]]]:
        """Formatters FNE"""
        return FORMATTERS

    def get_validators(self) -> Dict[str, Callable[[Dict[str, Any]], Any]]:
        """Validateurs FNE"""
        return VALIDATORS
