"""
Configuration pour la destination FNE-CDI

Ce module utilise un chargement paresseux (lazy-loading) de la configuration
pour permettre à la variable d'environnement MIDDLEWARE_CONFIG_PATH d'être
définie avant le chargement de la config.
"""
import os
import tomllib
from pathlib import Path
from typing import Dict, Any, Optional


# Cache pour la configuration lazy-loaded
_fne_config_cache: Optional[Dict[str, Any]] = None


def load_fne_config() -> Dict[str, Any]:
    """
    Charge la configuration FNE depuis le fichier config.toml

    Returns:
        Configuration FNE ou config par défaut
    """
    # Récupérer le chemin depuis la variable d'environnement
    config_path_str = os.environ.get('MIDDLEWARE_CONFIG_PATH')

    if config_path_str:
        config_path = Path(config_path_str)
    else:
        # Fallback: remonter de app/destinations/fne_cdi/ vers la racine du projet
        config_path = Path(__file__).parent.parent.parent.parent / 'config.toml'

    if not config_path.exists():
        return {
            "token": "",
            "establishment": "",
            "pos": ""
        }

    with open(config_path, 'rb') as f:
        config = tomllib.load(f)
        return config.get('fne_cdi', {})


def get_fne_config() -> Dict[str, Any]:
    """
    Retourne la configuration FNE avec lazy-loading.
    La config est chargée à la première demande puis mise en cache.

    Returns:
        Configuration FNE complète
    """
    global _fne_config_cache

    if _fne_config_cache is None:
        _fne_config_cache = load_fne_config()

    return _fne_config_cache


class _LazyFNEConfig:
    """
    Classe pour accéder aux variables de configuration FNE avec lazy-loading.
    Les valeurs sont chargées à la première utilisation.
    """

    @property
    def token(self) -> str:
        return get_fne_config().get('token', '')

    @property
    def establishment(self) -> str:
        return get_fne_config().get('establishment', '')

    @property
    def pos(self) -> str:
        return get_fne_config().get('pos', '')

    @property
    def base_url(self) -> str:
        api_config = get_fne_config().get('api', {})
        return api_config.get('base_url', 'http://54.247.95.108/ws/external/invoices/sign')

    @property
    def timeout(self) -> int:
        api_config = get_fne_config().get('api', {})
        return api_config.get('timeout', 30)

    @property
    def max_retries(self) -> int:
        api_config = get_fne_config().get('api', {})
        return api_config.get('max_retries', 5)


# Instance lazy
_lazy_config = _LazyFNEConfig()


def __getattr__(name: str):
    """
    Module-level __getattr__ pour implémenter le lazy-loading des variables.
    Permet d'importer FNE_CDI_TOKEN etc. de manière transparente.
    """
    if name == 'FNE_CDI_TOKEN':
        return _lazy_config.token
    elif name == 'FNE_CDI_ESTABLISHMENT':
        return _lazy_config.establishment
    elif name == 'FNE_CDI_POS':
        return _lazy_config.pos
    elif name == 'FNE_CDI_BASE_URL':
        return _lazy_config.base_url
    elif name == 'FNE_CDI_TIMEOUT':
        return _lazy_config.timeout
    elif name == 'FNE_CDI_MAX_RETRIES':
        return _lazy_config.max_retries
    raise AttributeError(f"module '{__name__}' has no attribute '{name}'")


def is_configured() -> bool:
    """Vérifie si la configuration FNE est complète"""
    return bool(
        _lazy_config.token and
        _lazy_config.establishment and
        _lazy_config.pos and
        _lazy_config.base_url
    )
