"""
Classes de base pour les destinations
Définit l'interface que chaque destination doit implémenter
"""
from typing import Dict, Any, List, Optional, Callable
from abc import ABC, abstractmethod
from dataclasses import dataclass, field
from enum import Enum


class RouteType(str, Enum):
    """Types de routing disponibles"""
    SIMPLE = "simple"
    MULTI = "multi"
    CONDITIONAL = "conditional"


@dataclass
class APIConfig:
    """Configuration d'une API destination"""
    name: str
    base_url: str
    headers: Dict[str, str]
    timeout: int = 30
    max_retries: int = 3
    description: str = ""
    enabled: bool = True
    extra: Dict[str, Any] = field(default_factory=dict)


@dataclass
class RouteConfig:
    """Configuration d'une route"""
    event_type: str
    route_type: RouteType
    destinations: List[str]
    formatter: str
    validator: Optional[str] = None
    condition: Optional[Callable[[Dict[str, Any]], bool]] = None
    description: str = ""
    enabled: bool = True


class BaseDestination(ABC):
    """
    Classe abstraite de base pour toutes les destinations

    Chaque destination doit hériter de cette classe et implémenter:
    - get_name(): nom unique de la destination
    - get_api_configs(): configurations des APIs
    - get_routes(): configurations des routes
    - get_formatters(): dictionnaire des formatters
    - get_validators(): dictionnaire des validateurs (optionnel)
    """

    @abstractmethod
    def get_name(self) -> str:
        """
        Retourne le nom unique de la destination

        Returns:
            Nom de la destination (ex: "fne_cdi", "test_apis")
        """
        pass

    @abstractmethod
    def get_api_configs(self) -> Dict[str, APIConfig]:
        """
        Retourne les configurations des APIs de cette destination

        Returns:
            Dictionnaire {api_id: APIConfig}
        """
        pass

    @abstractmethod
    def get_routes(self) -> Dict[str, RouteConfig]:
        """
        Retourne les routes gérées par cette destination

        Returns:
            Dictionnaire {event_type: RouteConfig}
        """
        pass

    @abstractmethod
    def get_formatters(self) -> Dict[str, Callable[[Dict[str, Any], str], Dict[str, Any]]]:
        """
        Retourne les formatters de cette destination

        Returns:
            Dictionnaire {formatter_name: fonction_formatter}
        """
        pass

    def get_validators(self) -> Dict[str, Callable[[Dict[str, Any]], Any]]:
        """
        Retourne les validateurs de cette destination (optionnel)

        Returns:
            Dictionnaire {validator_name: fonction_validator}
        """
        return {}

    def is_enabled(self) -> bool:
        """
        Vérifie si la destination est activée

        Returns:
            True si la destination est activée
        """
        return True

    def get_description(self) -> str:
        """
        Retourne une description de la destination

        Returns:
            Description de la destination
        """
        return ""

    def get_version(self) -> str:
        """
        Retourne la version de l'implémentation de la destination

        Returns:
            Version (ex: "1.0.0")
        """
        return "1.0.0"

    def validate_config(self) -> bool:
        """
        Valide la configuration de la destination

        Returns:
            True si la configuration est valide

        Raises:
            ValueError: Si la configuration est invalide
        """
        # Validation basique
        if not self.get_name():
            raise ValueError("Le nom de la destination ne peut pas être vide")

        if not self.get_api_configs():
            raise ValueError(f"La destination {self.get_name()} doit définir au moins une API")

        if not self.get_routes():
            raise ValueError(f"La destination {self.get_name()} doit définir au moins une route")

        if not self.get_formatters():
            raise ValueError(f"La destination {self.get_name()} doit définir au moins un formatter")

        return True

    def __repr__(self) -> str:
        return f"<{self.__class__.__name__} name='{self.get_name()}' version='{self.get_version()}'>"
