"""
Système de gestion des destinations
Découverte automatique et enregistrement des destinations disponibles
"""
from typing import Dict, List, Type, Optional
import importlib
import pkgutil
from pathlib import Path

from app.destinations.base import BaseDestination, APIConfig, RouteConfig, RouteType

# Registre global des destinations
_DESTINATIONS: Dict[str, BaseDestination] = {}


def discover_destinations() -> None:
    """
    Découvre automatiquement toutes les destinations disponibles
    Parcourt tous les sous-modules et charge les classes Destination
    """
    global _DESTINATIONS

    # Réinitialiser le registre
    _DESTINATIONS.clear()

    # Chemin du package destinations
    package_path = Path(__file__).parent

    # Parcourir tous les sous-modules
    for importer, module_name, is_pkg in pkgutil.iter_modules([str(package_path)]):
        # Ignorer les modules spéciaux
        if module_name in ('base', 'template', '__pycache__'):
            continue

        # Ignorer les fichiers
        if not is_pkg:
            continue

        try:
            # Importer le module
            module = importlib.import_module(f'.{module_name}', package='app.destinations')

            # Chercher une classe qui hérite de BaseDestination
            for attr_name in dir(module):
                attr = getattr(module, attr_name)

                # Vérifier que c'est une classe et qu'elle hérite de BaseDestination
                if (isinstance(attr, type) and
                    issubclass(attr, BaseDestination) and
                    attr is not BaseDestination):

                    # Instancier la destination
                    destination = attr()

                    # Valider la configuration
                    destination.validate_config()

                    # Enregistrer la destination
                    _DESTINATIONS[destination.get_name()] = destination

                    print(f"Destination chargée: {destination.get_name()} v{destination.get_version()}")

        except Exception as e:
            print(f"Erreur lors du chargement de la destination '{module_name}': {e}")


def get_destination(name: str) -> Optional[BaseDestination]:
    """
    Récupère une destination par son nom

    Args:
        name: Nom de la destination

    Returns:
        Instance de la destination ou None si non trouvée
    """
    return _DESTINATIONS.get(name)


def get_all_destinations() -> Dict[str, BaseDestination]:
    """
    Retourne toutes les destinations enregistrées

    Returns:
        Dictionnaire {nom: destination}
    """
    return _DESTINATIONS.copy()


def get_enabled_destinations() -> Dict[str, BaseDestination]:
    """
    Retourne uniquement les destinations activées

    Returns:
        Dictionnaire {nom: destination} des destinations activées
    """
    return {
        name: dest
        for name, dest in _DESTINATIONS.items()
        if dest.is_enabled()
    }


def get_all_api_configs() -> Dict[str, APIConfig]:
    """
    Collecte toutes les configurations d'API de toutes les destinations

    Returns:
        Dictionnaire {api_id: APIConfig}
    """
    all_configs = {}

    for destination in _DESTINATIONS.values():
        if destination.is_enabled():
            all_configs.update(destination.get_api_configs())

    return all_configs


def get_all_routes() -> Dict[str, RouteConfig]:
    """
    Collecte toutes les routes de toutes les destinations

    Returns:
        Dictionnaire {event_type: RouteConfig}
    """
    all_routes = {}

    for destination in _DESTINATIONS.values():
        if destination.is_enabled():
            all_routes.update(destination.get_routes())

    return all_routes


def get_all_formatters() -> Dict[str, callable]:
    """
    Collecte tous les formatters de toutes les destinations

    Returns:
        Dictionnaire {formatter_name: fonction}
    """
    all_formatters = {}

    for destination in _DESTINATIONS.values():
        if destination.is_enabled():
            all_formatters.update(destination.get_formatters())

    return all_formatters


def get_all_validators() -> Dict[str, callable]:
    """
    Collecte tous les validateurs de toutes les destinations

    Returns:
        Dictionnaire {validator_name: fonction}
    """
    all_validators = {}

    for destination in _DESTINATIONS.values():
        if destination.is_enabled():
            all_validators.update(destination.get_validators())

    return all_validators


def list_destinations() -> List[Dict[str, str]]:
    """
    Liste toutes les destinations avec leurs informations

    Returns:
        Liste de dictionnaires avec les infos des destinations
    """
    return [
        {
            "name": dest.get_name(),
            "description": dest.get_description(),
            "version": dest.get_version(),
            "enabled": dest.is_enabled(),
            "apis_count": len(dest.get_api_configs()),
            "routes_count": len(dest.get_routes())
        }
        for dest in _DESTINATIONS.values()
    ]


# Découvrir automatiquement les destinations au chargement du module
discover_destinations()


__all__ = [
    'BaseDestination',
    'APIConfig',
    'RouteConfig',
    'RouteType',
    'discover_destinations',
    'get_destination',
    'get_all_destinations',
    'get_enabled_destinations',
    'get_all_api_configs',
    'get_all_routes',
    'get_all_formatters',
    'get_all_validators',
    'list_destinations'
]
