# Système de Destinations Modulaires

Ce dossier contient le nouveau système de destinations modulaires du middleware, permettant d'ajouter facilement de nouvelles intégrations API.

## Architecture

Chaque destination est un module Python autonome qui encapsule :
- **Configuration** : URLs, tokens, paramètres spécifiques
- **Formatters** : Transformation des données pour l'API cible
- **Validateurs** : Validation Pydantic des données entrantes
- **Routes** : Mapping des event_type vers les APIs

## Structure

```
destinations/
├── base.py                    # Classes abstraites de base
├── __init__.py               # Système de découverte automatique
│
├── fne_cdi/                  # Destination FNE Côte d'Ivoire
│   ├── __init__.py
│   ├── destination.py        # Implémentation BaseDestination
│   ├── config.py            # Configuration (tokens, etc.)
│   ├── formatters.py        # Formatters FNE
│   ├── validators.py        # Validateurs Pydantic
│   ├── base.py             # Modèles de base
│   ├── compat.py           # Compatibilité Pydantic v1/v2
│   └── fne_invoice.py      # Modèles factures FNE
│
├── test_apis/               # APIs de test
│   ├── __init__.py
│   ├── destination.py
│   └── formatters.py
│
└── template/                # Template pour nouvelles destinations
    ├── README.md           # Guide de création
    ├── __init__.py.template
    ├── destination.py.template
    ├── formatters.py.template
    ├── config.py.template
    └── validators.py.template
```

## Destinations Disponibles

### FNE-CDI
- **Description** : Facturation électronique pour la Côte d'Ivoire
- **APIs** : fne_cdi_dev
- **Routes** : fnedev_invoice_sign, fnedev_invoice_cancel
- **Validation** : Pydantic avec compatibilité v1/v2

### Test APIs
- **Description** : APIs publiques gratuites pour tests
- **APIs** : restful_api, reqres, httpbin, jsonplaceholder
- **Routes** : create_product, create_user, test_broadcast, etc.

## Créer une Nouvelle Destination

### 1. Créer le dossier

```bash
cd destinations/
mkdir ma_destination
cd ma_destination
```

### 2. Copier les templates

```bash
cp ../template/__init__.py.template __init__.py
cp ../template/destination.py.template destination.py
cp ../template/formatters.py.template formatters.py

# Optionnel
cp ../template/config.py.template config.py
cp ../template/validators.py.template validators.py
```

### 3. Éditer les fichiers

#### `destination.py`

```python
from typing import Dict, Any, Callable
from ..base import BaseDestination, APIConfig, RouteConfig, RouteType
from .formatters import FORMATTERS

class MaDestination(BaseDestination):
    def get_name(self) -> str:
        return "ma_destination"

    def get_api_configs(self) -> Dict[str, APIConfig]:
        return {
            "mon_api": APIConfig(
                name="Mon API",
                base_url="https://api.example.com",
                headers={"Content-Type": "application/json"},
                timeout=30,
                max_retries=3,
                description="Mon API exemple"
            )
        }

    def get_routes(self) -> Dict[str, RouteConfig]:
        return {
            "mon_event": RouteConfig(
                event_type="mon_event",
                route_type=RouteType.SIMPLE,
                destinations=["mon_api"],
                formatter="format_mon_event",
                description="Mon event exemple"
            )
        }

    def get_formatters(self) -> Dict[str, Callable]:
        return FORMATTERS
```

#### `formatters.py`

```python
from typing import Dict, Any

def format_mon_event(data: Dict[str, Any], destination: str) -> Dict[str, Any]:
    """Formate les données pour mon API"""
    return {
        "formatted_data": data,
        "destination": destination
    }

FORMATTERS = {
    "format_mon_event": format_mon_event
}
```

### 4. Tester

```bash
# Retour à la racine
cd ../..

# Activer l'environnement
source venv/bin/activate

# Tester la découverte
python3 -c "from destinations import list_destinations; import json; print(json.dumps(list_destinations(), indent=2))"

# Démarrer le serveur
uvicorn main:app --reload
```

### 5. Vérifier

```bash
# Lister les destinations
curl http://localhost:8000/destinations

# Lister les routes
curl http://localhost:8000/routing-rules

# Tester votre route
curl -X POST http://localhost:8000/submit \
  -H "Content-Type: application/json" \
  -d '{"event_type": "mon_event", "data": {"test": "data"}}'
```

## Fonctionnalités

### Découverte Automatique
Le système découvre automatiquement toutes les destinations au démarrage. Aucune configuration manuelle requise.

### Activation/Désactivation
Implémentez `is_enabled()` pour contrôler dynamiquement l'activation :

```python
def is_enabled(self) -> bool:
    return bool(self.api_token)  # Désactivé si pas de token
```

### Validation des Données
Utilisez Pydantic pour valider les données entrantes :

```python
from pydantic import BaseModel, Field

class MonSchema(BaseModel):
    field1: str = Field(..., min_length=1)
    field2: int = Field(..., ge=0)

def validate_mon_event(data: dict) -> ValidationResult:
    result = ValidationResult(valid=True)
    try:
        validated = MonSchema(**data)
        result.data = validated.model_dump()
    except Exception as e:
        result.add_error(str(e))
    return result
```

### Configuration Externe
Stockez vos secrets dans `config.toml` :

```toml
[ma_destination]
api_key = "votre_clé"
api_secret = "votre_secret"
```

Puis chargez-les dans `config.py` :

```python
import tomllib
from pathlib import Path

def load_config():
    config_path = Path(__file__).parent.parent.parent / 'config.toml'
    with open(config_path, 'rb') as f:
        config = tomllib.load(f)
        return config.get('ma_destination', {})
```

## Bonnes Pratiques

1. **Nommage** : Utilisez snake_case pour les identifiants
2. **Documentation** : Documentez chaque formatter et validator
3. **Tests** : Créez des tests unitaires
4. **Secrets** : Ne committez jamais de secrets dans le code
5. **Versioning** : Incrémentez la version à chaque modification
6. **Validation** : Validez toujours les données entrantes
7. **Logging** : Utilisez le logger du middleware pour la traçabilité

## API de Base

### BaseDestination

Méthodes abstraites à implémenter :
- `get_name()` → nom unique
- `get_api_configs()` → configuration des APIs
- `get_routes()` → routes disponibles
- `get_formatters()` → fonctions de formatage

Méthodes optionnelles :
- `get_validators()` → validateurs Pydantic
- `get_description()` → description de la destination
- `get_version()` → version de l'implémentation
- `is_enabled()` → activation dynamique

### APIConfig

Configuration d'une API :
- `name` : nom affiché
- `base_url` : URL de base de l'API
- `headers` : headers HTTP par défaut
- `timeout` : timeout en secondes
- `max_retries` : nombre de tentatives
- `description` : description
- `enabled` : activation
- `extra` : données supplémentaires

### RouteConfig

Configuration d'une route :
- `event_type` : type d'événement
- `route_type` : SIMPLE, MULTI ou CONDITIONAL
- `destinations` : liste d'APIs cibles
- `formatter` : nom du formatter à utiliser
- `validator` : nom du validator (optionnel)
- `condition` : fonction de condition (optionnel)
- `description` : description
- `enabled` : activation

## Support

Pour toute question, consultez :
- La documentation complète dans `../README.md`
- Les exemples dans `fne_cdi/` et `test_apis/`
- Le template dans `template/README.md`
