"""
Modèles SQLAlchemy pour la base de données
"""
from datetime import datetime
from typing import Optional
from sqlalchemy import (
    Column, Integer, String, Text, Float, DateTime, ForeignKey, Index, Enum
)
from sqlalchemy.orm import declarative_base, relationship
from models import RequestStatus

Base = declarative_base()


class QueueModel(Base):
    """Modèle pour la table queue"""
    __tablename__ = "queue"

    id = Column(Integer, primary_key=True, autoincrement=True)
    event_type = Column(String(255), nullable=False)
    data = Column(Text, nullable=False)
    priority = Column(Integer, default=5)
    request_metadata = Column("metadata", Text, nullable=True)  # Renommé pour éviter conflit avec Base.metadata
    status = Column(
        Enum(RequestStatus),
        default=RequestStatus.PENDING,
        nullable=False
    )
    retry_count = Column(Integer, default=0)
    max_retries = Column(Integer, default=3)
    created_at = Column(DateTime, default=datetime.utcnow, nullable=False)
    processed_at = Column(DateTime, nullable=True)
    next_retry_at = Column(DateTime, nullable=True)
    error_details = Column(Text, nullable=True)  # Stocke les erreurs de validation/traitement en JSON

    # Relations
    api_responses = relationship(
        "APIResponseModel",
        back_populates="queue_item",
        cascade="all, delete-orphan"
    )
    logs = relationship(
        "LogModel",
        back_populates="queue_item",
        cascade="all, delete-orphan"
    )

    # Index
    __table_args__ = (
        Index("idx_status", "status"),
        Index("idx_next_retry", "next_retry_at"),
        Index("idx_priority_created", "priority", "created_at"),
    )

    def __repr__(self):
        return f"<QueueModel(id={self.id}, event_type='{self.event_type}', status='{self.status}')>"


class APIResponseModel(Base):
    """Modèle pour la table api_responses"""
    __tablename__ = "api_responses"

    id = Column(Integer, primary_key=True, autoincrement=True)
    queue_id = Column(Integer, ForeignKey("queue.id"), nullable=False)
    destination = Column(String(255), nullable=False)
    request_payload = Column(Text, nullable=True)  # Payload formaté envoyé à l'API
    status_code = Column(Integer, nullable=True)
    response_data = Column(Text, nullable=True)
    error = Column(Text, nullable=True)
    duration_ms = Column(Float, nullable=True)
    timestamp = Column(DateTime, default=datetime.utcnow, nullable=False)

    # Relations
    queue_item = relationship("QueueModel", back_populates="api_responses")

    # Index
    __table_args__ = (
        Index("idx_queue_id", "queue_id"),
        Index("idx_destination", "destination"),
        Index("idx_timestamp", "timestamp"),
    )

    def __repr__(self):
        return f"<APIResponseModel(id={self.id}, destination='{self.destination}', status={self.status_code})>"


class LogModel(Base):
    """Modèle pour la table logs"""
    __tablename__ = "logs"

    id = Column(Integer, primary_key=True, autoincrement=True)
    queue_id = Column(Integer, ForeignKey("queue.id"), nullable=True)
    level = Column(String(50), nullable=False)
    message = Column(Text, nullable=False)
    timestamp = Column(DateTime, default=datetime.utcnow, nullable=False)

    # Relations
    queue_item = relationship("QueueModel", back_populates="logs")

    # Index
    __table_args__ = (
        Index("idx_queue_id_log", "queue_id"),
        Index("idx_level", "level"),
        Index("idx_timestamp_log", "timestamp"),
    )

    def __repr__(self):
        return f"<LogModel(id={self.id}, level='{self.level}', queue_id={self.queue_id})>"
