"""
Configuration globale du middleware
Chargement automatique depuis le système de destinations modulaires
"""
from typing import Dict, List, Any, Optional
from pathlib import Path
from copy import deepcopy
import yaml
import os
import tomllib

# Import du système de destinations
from app.destinations import (
    get_all_api_configs,
    get_all_routes,
    list_destinations,
    RouteType
)

# Charger automatiquement depuis les destinations
_api_configs = get_all_api_configs()
_routes = get_all_routes()

# Convertir en dictionnaires simples
API_DESTINATIONS = {
    api_id: {
        "name": api.name,
        "base_url": api.base_url,
        "headers": api.headers,
        "timeout": api.timeout,
        "max_retries": api.max_retries,
        "description": api.description
    }
    for api_id, api in _api_configs.items()
}

ROUTING_RULES = {
    event_type: {
        "type": route.route_type,
        "destinations": route.destinations,
        "formatter": route.formatter,
        "validator": route.validator,
        "condition": route.condition,
        "description": route.description
    }
    for event_type, route in _routes.items()
}

# Configuration de la queue
QUEUE_CONFIG = {
    "max_queue_size": 10000,
    "batch_size": 10,
    "worker_count": 1,
    "retry_delays": [5, 30, 120, 300, 900]  # 5s, 30s, 2min, 5min, 15min
}

# Configuration du stockage
STORAGE_CONFIG = {
    "keep_responses_days": 30,
    "keep_failed_days": 90
}


def load_middleware_config() -> Dict[str, Any]:
    """
    Charge la configuration complète du middleware depuis config.toml

    Returns:
        Configuration complète ou dict vide si non trouvé
    """
    # Récupérer le chemin depuis la variable d'environnement
    config_path_str = os.environ.get('MIDDLEWARE_CONFIG_PATH') # Rempli par run.py

    config_path = Path(config_path_str)

    if not config_path.exists():
        return {}

    with open(config_path, 'rb') as f:
        return tomllib.load(f)


# Cache pour la configuration lazy-loaded
_middleware_config_cache: Optional[Dict[str, Any]] = None


def get_middleware_config() -> Dict[str, Any]:
    """
    Retourne la configuration du middleware avec lazy-loading.
    La config est chargée à la première demande (quand MIDDLEWARE_CONFIG_PATH est défini)
    puis mise en cache.

    Returns:
        Configuration complète ou dict vide si non trouvé
    """
    global _middleware_config_cache

    if _middleware_config_cache is None:
        _middleware_config_cache = load_middleware_config()

    return _middleware_config_cache


# Pour compatibilité avec le code existant, on crée un proxy
# qui charge la config à la demande
class _LazyConfigDict(dict):
    """
    Dict qui charge la configuration de manière paresseuse.
    Permet d'utiliser MIDDLEWARE_CONFIG comme avant tout en chargeant
    la config seulement quand elle est accédée.
    """
    _loaded = False

    def _ensure_loaded(self):
        if not self._loaded:
            self.update(get_middleware_config())
            self._loaded = True

    def __getitem__(self, key):
        self._ensure_loaded()
        return super().__getitem__(key)

    def __contains__(self, key):
        self._ensure_loaded()
        return super().__contains__(key)

    def get(self, key, default=None):
        self._ensure_loaded()
        return super().get(key, default)

    def items(self):
        self._ensure_loaded()
        return super().items()

    def keys(self):
        self._ensure_loaded()
        return super().keys()

    def values(self):
        self._ensure_loaded()
        return super().values()

    def __iter__(self):
        self._ensure_loaded()
        return super().__iter__()

    def __len__(self):
        self._ensure_loaded()
        return super().__len__()

    def __repr__(self):
        self._ensure_loaded()
        return super().__repr__()

    def __str__(self):
        self._ensure_loaded()
        return super().__str__()


# Configuration lazy-loaded - compatible avec l'ancien code
MIDDLEWARE_CONFIG = _LazyConfigDict()


def _deep_merge(base: Dict, override: Dict) -> Dict:
    """Merge récursif de deux dictionnaires"""
    result = deepcopy(base)

    for key, value in override.items():
        if key in result and isinstance(result[key], dict) and isinstance(value, dict):
            result[key] = _deep_merge(result[key], value)
        else:
            result[key] = deepcopy(value)

    return result


def load_client_config(config_path: str = None) -> None:
    """
    Charge une configuration client optionnelle depuis un fichier YAML
    pour surcharger les configurations par défaut

    Args:
        config_path: Chemin vers le fichier YAML (optionnel)
    """
    global API_DESTINATIONS, ROUTING_RULES, QUEUE_CONFIG, STORAGE_CONFIG

    if config_path is None:
        module_dir = Path(__file__).parent
        config_path = module_dir / "client_config.yaml"

        if not config_path.exists():
            config_path = Path.cwd() / "client_config.yaml"
    else:
        config_path = Path(config_path)

    if not config_path.exists():
        return

    with open(config_path, 'r', encoding='utf-8') as f:
        client_config = yaml.safe_load(f)

    if not client_config:
        return

    # Merge les configurations
    if "api_destinations" in client_config:
        API_DESTINATIONS = _deep_merge(API_DESTINATIONS, client_config["api_destinations"])

    if "routing_rules" in client_config:
        ROUTING_RULES = _deep_merge(ROUTING_RULES, client_config["routing_rules"])

    if "queue" in client_config:
        QUEUE_CONFIG = _deep_merge(QUEUE_CONFIG, client_config["queue"])

    if "storage" in client_config:
        STORAGE_CONFIG = _deep_merge(STORAGE_CONFIG, client_config["storage"])


def get_config_value(section: str, key: str = None, default: Any = None) -> Any:
    """
    Récupère une valeur de configuration

    Args:
        section: Section ('api_destinations', 'routing_rules', 'queue', 'storage')
        key: Clé spécifique (optionnel)
        default: Valeur par défaut

    Returns:
        Valeur de configuration
    """
    configs = {
        "api_destinations": API_DESTINATIONS,
        "routing_rules": ROUTING_RULES,
        "queue": QUEUE_CONFIG,
        "storage": STORAGE_CONFIG
    }

    if section not in configs:
        return default

    config = configs[section]

    if key is None:
        return config

    return config.get(key, default)


def get_destinations_info() -> List[Dict[str, Any]]:
    """Retourne les informations sur toutes les destinations chargées"""
    return list_destinations()


# Charge automatiquement la configuration client au démarrage
load_client_config()
