#!/usr/bin/env python3
"""
Script de test du système de logging email_sender

Ce script permet de valider la configuration du logging structlog
et de générer des exemples de logs pour chaque type d'événement.
"""

import os
import sys
import time
from pathlib import Path

# Configurer le logging AVANT d'importer les modules
os.environ['EMAIL_LOG_LEVEL'] = 'DEBUG'
os.environ['EMAIL_LOG_TO_SCREEN'] = '1'
os.environ['EMAIL_LOG_DIR'] = '/tmp/email_sender_test'

from logger import (
    get_logger,
    log_email_sent,
    log_email_error,
    log_api_call,
    get_log_config
)


def test_basic_logging():
    """Test du logging basique"""
    print("\n" + "="*60)
    print("TEST 1: Logging basique")
    print("="*60)

    logger = get_logger("test")

    logger.debug("test_debug", test_id=1, message="Ceci est un debug")
    logger.info("test_info", test_id=2, message="Ceci est une info")
    logger.warning("test_warning", test_id=3, message="Ceci est un warning")
    logger.error("test_error", test_id=4, message="Ceci est une erreur")

    print("✓ Logs basiques générés")


def test_email_logging():
    """Test des fonctions de logging d'emails"""
    print("\n" + "="*60)
    print("TEST 2: Logging d'envoi d'emails")
    print("="*60)

    # Simuler un envoi réussi
    log_email_sent(
        provider="brevo",
        to_email="test@example.com",
        subject="Test Email",
        message_id="msg_12345",
        duration_ms=245.6
    )
    print("✓ Email envoyé loggé")

    # Simuler une erreur
    log_email_error(
        provider="brevo",
        to_email="error@example.com",
        error="API rate limit exceeded",
        error_code="429",
        subject="Failed Email"
    )
    print("✓ Erreur d'email loggée")


def test_api_logging():
    """Test du logging d'appels API"""
    print("\n" + "="*60)
    print("TEST 3: Logging d'appels API")
    print("="*60)

    # Simuler plusieurs appels API
    for i in range(3):
        start = time.time()
        time.sleep(0.1)  # Simuler un appel API
        duration = (time.time() - start) * 1000

        log_api_call(
            provider="brevo",
            endpoint="/smtp/email",
            method="POST",
            status_code=201 if i % 2 == 0 else 400,
            duration_ms=duration
        )

    print("✓ 3 appels API loggés")


def test_template_logging():
    """Test du logging de templates"""
    print("\n" + "="*60)
    print("TEST 4: Logging de templates")
    print("="*60)

    logger = get_logger("email_sender.brevo")

    logger.debug(
        "brevo_send_template_request",
        to_email="template@example.com",
        template_id=2,
        params_count=8,
        has_cc=False,
        has_bcc=False
    )

    log_email_sent(
        provider="brevo",
        to_email="template@example.com",
        template_id=2,
        message_id="tpl_98765",
        duration_ms=189.3
    )

    print("✓ Template loggé")


def test_configuration():
    """Affiche la configuration actuelle"""
    print("\n" + "="*60)
    print("TEST 5: Configuration")
    print("="*60)

    config = get_log_config()

    print(f"Niveau de log: {config['level']}")
    print(f"Répertoire: {config['log_dir']}")
    print(f"Fichier: {config['log_file']}")
    print(f"Sortie console: {'Oui' if config['log_to_screen'] else 'Non'}")
    print(f"Taille max: {config['max_bytes'] / 1024 / 1024:.1f} MB")
    print(f"Nombre de backups: {config['backup_count']}")

    # Vérifier que le fichier existe
    log_path = Path(config['log_dir']) / config['log_file']
    if log_path.exists():
        size = log_path.stat().st_size
        print(f"\n✓ Fichier de log créé: {log_path}")
        print(f"  Taille: {size} bytes")
    else:
        print(f"\n✗ Fichier de log non trouvé: {log_path}")


def test_performance():
    """Test de performance du logging"""
    print("\n" + "="*60)
    print("TEST 6: Performance")
    print("="*60)

    logger = get_logger("performance_test")

    iterations = 1000
    start = time.time()

    for i in range(iterations):
        logger.info(
            "performance_test",
            iteration=i,
            timestamp=time.time(),
            data={"key": "value"}
        )

    duration = time.time() - start
    rate = iterations / duration

    print(f"✓ {iterations} logs générés en {duration:.3f}s")
    print(f"  Débit: {rate:.0f} logs/seconde")


def main():
    """Exécute tous les tests"""
    print("\n" + "#"*60)
    print("#" + " "*20 + "TEST DU LOGGING" + " "*23 + "#")
    print("#"*60)

    try:
        test_configuration()
        test_basic_logging()
        test_email_logging()
        test_api_logging()
        test_template_logging()
        test_performance()

        print("\n" + "="*60)
        print("RÉSULTAT: ✓ Tous les tests réussis")
        print("="*60)

        # Afficher les dernières lignes du fichier de log
        config = get_log_config()
        log_path = Path(config['log_dir']) / config['log_file']

        if log_path.exists():
            print(f"\nDernières lignes de {log_path}:")
            print("-"*60)
            with open(log_path) as f:
                lines = f.readlines()
                for line in lines[-5:]:
                    print(line.rstrip())

    except Exception as e:
        print(f"\n✗ ERREUR: {e}")
        import traceback
        traceback.print_exc()
        return 1

    return 0


if __name__ == "__main__":
    sys.exit(main())
