#!/usr/bin/env python3
"""
Script autonome pour rejouer une requête Brevo à partir des logs

Usage:
    # Copier le payload depuis les logs, puis:
    python test_brevo_request.py --api-key "VOTRE_CLE_API" --payload payload.json

    # Ou directement avec le payload en argument:
    python test_brevo_request.py --api-key "VOTRE_CLE_API" --payload-inline '{"to":[...]}'
"""
import argparse
import json
import requests
import sys


def replay_brevo_request(api_key: str, payload: dict, base_url: str = "https://api.brevo.com/v3"):
    """
    Rejoue une requête Brevo avec les paramètres fournis

    Args:
        api_key: Clé API Brevo
        payload: Payload JSON de la requête
        base_url: URL de base de l'API Brevo

    Returns:
        Réponse de l'API
    """
    url = f"{base_url}/smtp/email"
    headers = {
        "api-key": api_key,
        "Content-Type": "application/json",
        "Accept": "application/json"
    }

    print(f"🔄 Envoi de la requête à {url}")
    print(f"📦 Payload size: {len(json.dumps(payload))} bytes")

    # Afficher un résumé du payload (sans les données volumineuses)
    summary = {
        "to": payload.get("to"),
        "subject": payload.get("subject"),
        "htmlContent_length": len(payload.get("htmlContent", "")) if "htmlContent" in payload else 0,
        "has_attachments": "attachment" in payload,
        "attachment_count": len(payload.get("attachment", [])) if "attachment" in payload else 0
    }
    print(f"📝 Résumé: {json.dumps(summary, indent=2)}")
    print()

    try:
        response = requests.post(
            url,
            headers=headers,
            json=payload,
            timeout=30
        )

        print(f"✅ Status Code: {response.status_code}")
        print(f"📨 Response Headers: {dict(response.headers)}")
        print()

        if response.ok:
            print("✅ SUCCESS!")
            print(f"Response: {json.dumps(response.json(), indent=2)}")
            return response.json()
        else:
            print("❌ ERROR!")
            print(f"Status: {response.status_code} {response.reason}")
            print(f"Body: {response.text}")
            try:
                error_json = response.json()
                print(f"Error JSON: {json.dumps(error_json, indent=2)}")
            except:
                pass
            return None

    except Exception as e:
        print(f"❌ Exception: {e}")
        import traceback
        traceback.print_exc()
        return None


def main():
    parser = argparse.ArgumentParser(
        description="Rejoue une requête Brevo à partir des logs pour debugging"
    )
    parser.add_argument(
        "--api-key",
        required=True,
        help="Clé API Brevo"
    )
    parser.add_argument(
        "--payload",
        help="Chemin vers un fichier JSON contenant le payload"
    )
    parser.add_argument(
        "--payload-inline",
        help="Payload JSON directement en argument"
    )
    parser.add_argument(
        "--base-url",
        default="https://api.brevo.com/v3",
        help="URL de base de l'API Brevo (défaut: production)"
    )

    args = parser.parse_args()

    # Charger le payload
    if args.payload:
        with open(args.payload, 'r', encoding='utf-8') as f:
            payload = json.load(f)
    elif args.payload_inline:
        payload = json.loads(args.payload_inline)
    else:
        print("❌ Erreur: --payload ou --payload-inline requis", file=sys.stderr)
        sys.exit(1)

    # Rejouer la requête
    result = replay_brevo_request(args.api_key, payload, args.base_url)

    if result:
        sys.exit(0)
    else:
        sys.exit(1)


if __name__ == "__main__":
    main()
