"""
Configuration du système de logging avec structlog pour email_sender

Ce module configure structlog pour :
- Logs structurés au format JSON
- Sauvegarde dans un fichier avec rotation adaptative
- Contexte automatique (timestamp, niveau, module)
- Support des variables d'environnement pour la configuration

Rotation des logs :
- Mode dev (DBPATH non défini) : RotatingFileHandler avec rotation automatique
- Mode prod (DBPATH défini) : FileHandler simple, rotation gérée par logrotate
"""

import os
import sys
import logging
from pathlib import Path
from logging.handlers import RotatingFileHandler
from typing import Any, Dict

import structlog


# Configuration par défaut
DEFAULT_LOG_LEVEL = "DEBUG"
DBPATH = os.getenv("DBPATH", "")
if not Path(DBPATH).is_dir():
    DBPATH = None
if DBPATH:
    DEFAULT_LOG_DIR = f"/home/log/{Path(DBPATH).name}/daily"
else:
    DEFAULT_LOG_DIR = "/tmp"
DEFAULT_LOG_FILE = "email_sender.log"
DEFAULT_MAX_BYTES = 10 * 1024 * 1024  # 10 MB
DEFAULT_BACKUP_COUNT = 5


def get_log_config() -> Dict[str, Any]:
    """
    Récupère la configuration du logging depuis les variables d'environnement

    Variables d'environnement:
        EMAIL_LOG_LEVEL: Niveau de log (DEBUG, INFO, WARNING, ERROR) - défaut: INFO
        EMAIL_LOG_DIR: Répertoire des logs - défaut: /home/log/email_sender
        EMAIL_LOG_FILE: Nom du fichier de log - défaut: email_sender.log
        EMAIL_LOG_TO_SCREEN: Afficher aussi sur stdout (1/0) - défaut: 0

    Returns:
        Dictionnaire de configuration
    """
    return {
        "level": os.getenv("EMAIL_LOG_LEVEL", DEFAULT_LOG_LEVEL).upper(),
        "log_dir": os.getenv("EMAIL_LOG_DIR", DEFAULT_LOG_DIR),
        "log_file": os.getenv("EMAIL_LOG_FILE", DEFAULT_LOG_FILE),
        "log_to_screen": os.getenv("EMAIL_LOG_TO_SCREEN", "0") == "1"
    }


def setup_logging() -> structlog.BoundLogger:
    """
    Configure et initialise le système de logging avec structlog

    Returns:
        Logger structlog configuré
    """
    config = get_log_config()
    if not DBPATH:
        config |= {
            "max_bytes": DEFAULT_MAX_BYTES,
            "backup_count": DEFAULT_BACKUP_COUNT,
        }

    log_dir = Path(config["log_dir"])
    log_dir.mkdir(parents=True, exist_ok=True)

    log_path = log_dir / config["log_file"]

    # Convertir le niveau de log string en niveau logging
    log_level = getattr(logging, config["level"], logging.INFO)

    # Configuration du logging standard
    logging.basicConfig(
        format="%(message)s",
        level=log_level,
        handlers=[]
    )

    # Choisir le handler selon l'environnement
    if DBPATH is None:
        # Mode dev/test : utiliser RotatingFileHandler
        file_handler = RotatingFileHandler(
            log_path,
            maxBytes=config["max_bytes"],
            backupCount=config["backup_count"],
            encoding="utf-8"
        )
    else:
        # Mode production : utiliser FileHandler simple (rotation gérée par logrotate)
        file_handler = logging.FileHandler(
            log_path,
            encoding="utf-8"
        )

    file_handler.setLevel(log_level)

    # Ajouter le handler au logger racine
    root_logger = logging.getLogger()
    root_logger.addHandler(file_handler)

    # Si demandé, ajouter aussi la sortie console
    if config["log_to_screen"]:
        console_handler = logging.StreamHandler(sys.stdout)
        console_handler.setLevel(log_level)
        root_logger.addHandler(console_handler)

    # Configuration de structlog
    processors = [
        structlog.stdlib.filter_by_level,
        structlog.stdlib.add_logger_name,
        structlog.stdlib.add_log_level,
        structlog.stdlib.PositionalArgumentsFormatter(),
        structlog.processors.TimeStamper(fmt="iso"),
        structlog.processors.StackInfoRenderer(),
        structlog.processors.format_exc_info,
        structlog.processors.UnicodeDecoder(),
    ]

    # En mode dev (DEBUG), ajouter les infos de call site (fichier:ligne)
    if log_level == logging.DEBUG:
        processors.insert(3, structlog.processors.CallsiteParameterAdder(
            [
                structlog.processors.CallsiteParameter.FILENAME,
                structlog.processors.CallsiteParameter.LINENO,
            ]
        ))

    processors.append(structlog.processors.JSONRenderer())

    structlog.configure(
        processors=processors,
        context_class=dict,
        logger_factory=structlog.stdlib.LoggerFactory(),
        cache_logger_on_first_use=True,
    )

    # Retourner un logger structlog
    return structlog.get_logger("email_sender")


def get_logger(name: str = None) -> structlog.BoundLogger:
    """
    Obtient un logger structlog

    Args:
        name: Nom du logger (optionnel, par défaut "email_sender")

    Returns:
        Logger structlog configuré
    """
    if name:
        return structlog.get_logger(name)
    return structlog.get_logger("email_sender")


# Logger global pour le module
logger = setup_logging()


# Fonctions de convenance pour le logging
def log_email_sent(
    provider: str,
    to_email: str,
    subject: str = None,
    template_id: int = None,
    message_id: str = None,
    **kwargs
):
    """
    Log l'envoi d'un email avec toutes les informations pertinentes

    Args:
        provider: Fournisseur utilisé (brevo, mailgun)
        to_email: Destinataire
        subject: Sujet de l'email (si applicable)
        template_id: ID du template (si applicable)
        message_id: ID du message retourné par le provider
        **kwargs: Informations supplémentaires
    """
    logger.info(
        "email_sent",
        provider=provider,
        to_email=to_email,
        subject=subject,
        template_id=template_id,
        message_id=message_id,
        **kwargs
    )


def log_email_error(
    provider: str,
    to_email: str,
    error: str,
    error_code: str = None,
    **kwargs
):
    """
    Log une erreur lors de l'envoi d'un email

    Args:
        provider: Fournisseur utilisé (brevo, mailgun)
        to_email: Destinataire
        error: Message d'erreur
        error_code: Code d'erreur (si disponible)
        **kwargs: Informations supplémentaires
    """
    logger.error(
        "email_error",
        provider=provider,
        to_email=to_email,
        error=error,
        error_code=error_code,
        **kwargs
    )


def log_api_call(
    provider: str,
    endpoint: str,
    method: str,
    status_code: int = None,
    duration_ms: float = None,
    **kwargs
):
    """
    Log un appel API vers un fournisseur d'email

    Args:
        provider: Fournisseur (brevo, mailgun)
        endpoint: Endpoint appelé
        method: Méthode HTTP
        status_code: Code de statut HTTP retourné
        duration_ms: Durée de l'appel en millisecondes
        **kwargs: Informations supplémentaires
    """
    logger.debug(
        "api_call",
        provider=provider,
        endpoint=endpoint,
        method=method,
        status_code=status_code,
        duration_ms=duration_ms,
        **kwargs
    )
