"""
Client Brevo (ex-SendinBlue) pour l'envoi d'emails
"""
import requests
import time
from typing import Dict, List, Optional, Tuple
from dataclasses import dataclass
import base64
import json

from .logger import get_logger, log_email_sent, log_email_error, log_api_call

logger = get_logger("email_sender.brevo")


@dataclass
class BrevoConfig:
    """Configuration Brevo"""
    api_key: str
    from_email: str
    from_name: str = "GCV Gestion"
    subject: Optional[str] = None
    tags: Optional[List[str]] = None


# https://developers.brevo.com/docs/batch-send-transactional-emails#creating-the-api-call-with-a-template-id
class BrevoClient:
    """Client pour l'API Brevo (SendinBlue)"""

    def __init__(self, config: BrevoConfig):
        """
        Initialise le client Brevo

        Args:
            config: Configuration Brevo
        """
        self.config = config
        self.base_url = "https://api.brevo.com/v3"
        self.headers = {
            "api-key": config.api_key,
            "Content-Type": "application/json",
            "Accept": "application/json"
        }

        logger.debug("brevo_client_initialized")

    def send_email(
        self,
        to_email: str,
        subject: str,
        html_body: str,
        text_body: Optional[str] = None,
        attachments: Optional[List[Tuple[str, bytes, str]]] = None,
        inline_images: Optional[Dict[str, bytes]] = None,
        cc: Optional[List[str]] = None,
        bcc: Optional[List[str]] = None,
    ) -> Dict:
        """
        Envoie un email via Brevo

        Args:
            to_email: Adresse du destinataire
            subject: Sujet de l'email
            html_body: Corps HTML de l'email
            text_body: Corps texte brut (optionnel)
            attachments: Liste de tuples (filename, content, mime_type)
            inline_images: Dict {cid: image_bytes} pour les images inline
            cc: Liste d'adresses en copie
            bcc: Liste d'adresses en copie cachée

        Returns:
            Réponse de l'API Brevo

        Raises:
            requests.exceptions.RequestException: En cas d'erreur API
        """
        # Construire le payload
        payload = {
            "sender": {
                "name": self.config.from_name,
                "email": self.config.from_email
            },
            "to": [{"email": to_email}],
            "subject": subject,
            "htmlContent": html_body,
        }

        logger.debug("brevo_client_send_email")

        if text_body:
            payload["textContent"] = text_body

        if cc:
            payload["cc"] = [{"email": addr} for addr in cc]

        if bcc:
            payload["bcc"] = [{"email": addr} for addr in bcc]

        # Ajouter les pièces jointes
        if attachments:
            payload["attachment"] = []
            for filename, content, _mime_type in attachments:
                payload["attachment"].append({
                    "name": filename,
                    "content": base64.b64encode(content).decode('utf-8')
                })

        # Ajouter les images inline
        if inline_images:
            # Brevo utilise un format spécial pour les images inline
            # Il faut les mettre dans le HTML en base64 data URI au lieu de CID
            # OU les ajouter comme attachments normaux et modifier le HTML

            # Solution: Remplacer les cid: dans le HTML par des data URIs
            for cid, image_bytes in inline_images.items():
                # Encoder l'image en base64
                img_base64 = base64.b64encode(image_bytes).decode('utf-8')
                # Remplacer cid:name par data URI dans le HTML
                payload["htmlContent"] = payload["htmlContent"].replace(
                    f'cid:{cid}',
                    f'data:image/png;base64,{img_base64}'
                )

        # Logger la requête
        logger.debug(
            "brevo_send_email_request",
            to_email=to_email,
            subject=subject,
            has_attachments=bool(attachments),
            has_inline_images=bool(inline_images),
            has_cc=bool(cc),
            has_bcc=bool(bcc)
        )

        # Envoyer la requête avec mesure du temps
        start_time = time.time()
        try:
            response = requests.post(
                f"{self.base_url}/smtp/email",
                headers=self.headers,
                json=payload,
                timeout=30
            )
            duration_ms = (time.time() - start_time) * 1000

            # Logger l'appel API
            log_api_call(
                provider="brevo",
                endpoint="/smtp/email",
                method="POST",
                status_code=response.status_code,
                duration_ms=duration_ms
            )

            # En cas d'erreur, logger et lever l'exception
            if not response.ok:
                error_detail = {
                    "status_code": response.status_code,
                    "reason": response.reason,
                    "body": response.text
                }
                try:
                    error_detail["json"] = response.json()
                except:
                    pass

                log_email_error(
                    provider="brevo",
                    to_email=to_email,
                    error=f"{response.status_code} {response.reason}",
                    error_code=str(response.status_code),
                    subject=subject
                )

                raise requests.exceptions.HTTPError(
                    f"{response.status_code} {response.reason} for url: {response.url}. "
                    f"Detail: {error_detail}",
                    response=response
                )

            result = response.json()

            # Logger le succès
            log_email_sent(
                provider="brevo",
                to_email=to_email,
                subject=subject,
                message_id=result.get("messageId"),
                duration_ms=duration_ms
            )

            return result

        except requests.exceptions.RequestException as e:
            duration_ms = (time.time() - start_time) * 1000
            log_email_error(
                provider="brevo",
                to_email=to_email,
                error=str(e),
                subject=subject,
                duration_ms=duration_ms
            )
            raise

    def send_email_with_template(
        self,
        to_email: str,
        template_id: int,
        template_params: Dict[str, str],
        cc: Optional[List[str]] = None,
        bcc: Optional[List[str]] = None,
    ) -> Dict:
        """
        Envoie un email en utilisant un template Brevo

        Args:
            to_email: Adresse du destinataire
            template_id: ID du template Brevo à utiliser
            template_params: Dictionnaire des paramètres pour le template
                           (ex: {"FIRSTNAME": "Jean", "ORDER_NUMBER": "FA001"})
            cc: Liste d'adresses en copie
            bcc: Liste d'adresses en copie cachée
            
        Returns:
            Réponse de l'API Brevo

        Raises:
            requests.exceptions.RequestException: En cas d'erreur API

        See Also:
            https://developers.brevo.com/docs/batch-send-transactional-emails#creating-a-transactional-template
        """
        # Construire le payload
        payload = {
            "sender": {
                "name": self.config.from_name,
                "email": self.config.from_email
                },
            "to": [{"email": to_email}],
            "templateId": template_id,
            "params": template_params
        }

        logger.debug("brevo_send_template_payload", payload=payload)

        if cc:
            payload["cc"] = [{"email": addr} for addr in cc]

        if bcc:
            payload["bcc"] = [{"email": addr} for addr in bcc]

        # Logger la requête
        logger.debug(
            "brevo_send_template_request",
            to_email=to_email,
            template_id=template_id,
            params_count=len(template_params),
            has_cc=bool(cc),
            has_bcc=bool(bcc)
        )


        # Envoyer la requête avec mesure du temps
        start_time = time.time()
        try:
            response = requests.post(
                f"{self.base_url}/smtp/email",
                headers=self.headers,
                json=payload,
                timeout=30
            )
            duration_ms = (time.time() - start_time) * 1000

            # Logger l'appel API
            log_api_call(
                provider="brevo",
                endpoint="/smtp/email",
                method="POST",
                status_code=response.status_code,
                duration_ms=duration_ms,
                template_id=template_id
            )

            # En cas d'erreur, logger et lever l'exception
            if not response.ok:
                error_detail = {
                    "status_code": response.status_code,
                    "reason": response.reason,
                    "body": response.text
                }
                try:
                    error_detail["json"] = response.json()
                except:
                    pass

                log_email_error(
                    provider="brevo",
                    to_email=to_email,
                    error=f"{response.status_code} {response.reason}",
                    error_code=str(response.status_code),
                    template_id=template_id
                )

                raise requests.exceptions.HTTPError(
                    f"{response.status_code} {response.reason} for url: {response.url}. "
                    f"Detail: {error_detail}",
                    response=response
                )

            result = response.json()

            # Logger le succès
            log_email_sent(
                provider="brevo",
                to_email=to_email,
                template_id=template_id,
                message_id=result.get("messageId"),
                duration_ms=duration_ms
            )

            return result

        except requests.exceptions.RequestException as e:
            duration_ms = (time.time() - start_time) * 1000
            log_email_error(
                provider="brevo",
                to_email=to_email,
                error=str(e),
                template_id=template_id,
                duration_ms=duration_ms
            )
            raise

    def verify_api_key(self) -> bool:
        """
        Vérifie que la clé API est valide

        Returns:
            True si la clé API est valide
        """
        try:
            logger.debug("brevo_verify_api_key")
            response = requests.get(
                f"{self.base_url}/account",
                headers=self.headers,
                timeout=10
            )
            is_valid = response.status_code == 200
            logger.info(
                "brevo_api_key_verified",
                is_valid=is_valid,
                status_code=response.status_code
            )
            return is_valid
        except requests.exceptions.RequestException as e:
            logger.error("brevo_verify_api_key_error", error=str(e))
            return False

    def get_account_info(self) -> Dict:
        """
        Récupère les informations du compte

        Returns:
            Informations du compte Brevo
        """
        response = requests.get(
            f"{self.base_url}/account",
            headers=self.headers,
            timeout=10
        )
        response.raise_for_status()
        return response.json()
