# Module d'envoi d'emails

Module Python générique pour l'envoi d'emails via Mailgun ou Brevo (SendinBlue).

## Caractéristiques

- **Générique**: Envoi d'emails texte ou HTML
- **Multi-provider**: Support de Mailgun et Brevo
- **Templates Brevo**: Support des templates transactionnels Brevo
- **Flexible**: Pièces jointes, images inline, CC, BCC
- **Logging structuré**: Logs JSON avec structlog et rotation automatique
- **Standalone**: Utilisable indépendamment de tout projet
- **Simple**: API intuitive et facile à utiliser

## Installation

```bash
pip install -r requirements.txt
```

Ou manuellement:
```bash
pip install requests structlog
```

Dépendances optionnelles:
```bash
pip install pillow qrcode  # Pour la génération de QR codes
```

## Configuration

### Via dictionnaire Python

```python
from email_sender import EmailSender

# Mailgun
sender = EmailSender(
    provider="mailgun",
    api_key="votre-cle-api",
    domain="mg.votre-domaine.com",
    from_email="noreply@votre-domaine.com",
    from_name="Mon Application"
)

# Brevo
sender = EmailSender(
    provider="brevo",
    api_key="votre-cle-api",
    from_email="noreply@votre-domaine.com",
    from_name="Mon Application"
)
```

### Via fichier TOML

```python
import tomli
from email_sender import EmailSender

with open("config.toml", "rb") as f:
    config = tomli.load(f)

sender = EmailSender.from_config(config)
```

Format du fichier TOML :

```toml
[email]
provider = "mailgun"  # ou "brevo"

[email.mailgun]
api_key = "votre-cle-api"
domain = "mg.votre-domaine.com"
from_email = "noreply@votre-domaine.com"
from_name = "Mon Application"

[email.brevo]
api_key = "votre-cle-api"
from_email = "noreply@votre-domaine.com"
from_name = "Mon Application"
```

## Usage

### Email HTML simple

```python
sender.send_html(
    to_email="user@example.com",
    subject="Bienvenue",
    html_body="<h1>Bonjour!</h1><p>Merci de votre inscription.</p>"
)
```

### Email texte simple

```python
sender.send_text(
    to_email="user@example.com",
    subject="Notification",
    text_body="Votre commande a été expédiée."
)
```

### Email avec pièce jointe

```python
with open("document.pdf", "rb") as f:
    pdf_bytes = f.read()

sender.send_file(
    to_email="user@example.com",
    subject="Votre document",
    body="Veuillez trouver ci-joint votre document.",
    filename="document.pdf",
    file_content=pdf_bytes,
    mime_type="application/pdf",
    body_type="text"
)
```

### Email HTML avec image inline

```python
from email_sender.qrcode_generator import generate_qrcode

# Générer un QR code
qr_bytes = generate_qrcode("https://example.com", size=300)

# Envoyer avec image inline
sender.send_html(
    to_email="user@example.com",
    subject="Votre QR code",
    html_body='<h1>Scanner ce code:</h1><img src="cid:qrcode">',
    inline_images={"qrcode": qr_bytes}
)
```

### Email avec CC et BCC

```python
sender.send(
    to_email="user@example.com",
    subject="Réunion",
    body="<p>Invitation à la réunion</p>",
    body_type="html",
    cc=["manager@example.com"],
    bcc=["admin@example.com"]
)
```

## API complète

### EmailSender

**Constructeur:**
```python
EmailSender(
    provider: str,           # "mailgun" ou "brevo"
    api_key: str,           # Clé API du provider
    from_email: str,        # Email expéditeur
    from_name: str = "",    # Nom expéditeur (optionnel)
    domain: str = None      # Domaine (requis pour Mailgun)
)
```

**Méthode classmethod:**
```python
EmailSender.from_config(config: dict) -> EmailSender
```

**Méthodes d'envoi:**

```python
# Méthode générique
send(
    to_email: str,
    subject: str,
    body: str,
    body_type: str = "html",  # "html" ou "text"
    attachments: List[Tuple[str, bytes, str]] = None,
    inline_images: Dict[str, bytes] = None,
    cc: List[str] = None,
    bcc: List[str] = None
) -> Dict[str, Any]

# Raccourcis
send_html(to_email, subject, html_body, text_body=None, **kwargs)
send_text(to_email, subject, text_body, **kwargs)
send_file(to_email, subject, body, filename, file_content, mime_type, body_type="text")

# Vérification
verify_configuration() -> bool
```

### Générateur de QR code

```python
from email_sender.qrcode_generator import generate_qrcode, generate_qrcode_base64

# Générer un QR code en bytes PNG
qr_bytes = generate_qrcode("https://example.com", size=300)

# Générer un QR code en base64
qr_base64 = generate_qrcode_base64("https://example.com", size=300)
```

## Utilisation avec le middleware FNE

Le module `middleware/app/email_notifications.py` fournit une couche d'intégration spécifique pour le middleware :

```python
from app.email_notifications import MiddlewareEmailNotifications

# Initialiser avec la config du middleware
email_notif = MiddlewareEmailNotifications(config)

# Envoyer une facture au client
# Note: download_url est l'URL complète depuis response_data.token FNE
# (ex: "http://54.247.95.108/fr/verification/019a799e-4205-7331-8b6d-34be13a9dacb")
email_notif.send_invoice_to_client(
    client_email="client@example.com",
    client_name="Jean Dupont",
    invoice_number="FA25000100",
    invoice_date="12/01/2025",
    invoice_amount=50000,
    download_url="http://54.247.95.108/fr/verification/019a799e-4205-7331-8b6d-34be13a9dacb"
)

# Envoyer une alerte admin
email_notif.send_admin_alert(
    warnings=["Erreur de validation"],
    balance_funds=500,
    invoice_number="FA25000100",
    request_id=123,
    download_url="http://54.247.95.108/fr/verification/..."  # Optionnel
)
```

## Exemples complets

### Exemple 1: Email de bienvenue avec logo

```python
from email_sender import EmailSender

sender = EmailSender(
    provider="mailgun",
    api_key="key-xxx",
    domain="mg.example.com",
    from_email="noreply@example.com",
    from_name="Mon App"
)

with open("logo.png", "rb") as f:
    logo_bytes = f.read()

html = '''
<html>
  <body>
    <img src="cid:logo" width="200">
    <h1>Bienvenue !</h1>
    <p>Merci de votre inscription.</p>
  </body>
</html>
'''

sender.send_html(
    to_email="user@example.com",
    subject="Bienvenue sur Mon App",
    html_body=html,
    inline_images={"logo": logo_bytes}
)
```

### Exemple 2: Facture avec QR code

```python
from email_sender import EmailSender
from email_sender.qrcode_generator import generate_qrcode

sender = EmailSender.from_config(config)

# Générer le QR code
invoice_url = "https://factures.example.com/download/token123"
qr_code = generate_qrcode(invoice_url, size=300)

html = f'''
<html>
  <body>
    <h1>Votre facture FA001</h1>
    <p>Montant: 1000 FCFA</p>
    <h2>QR Code de téléchargement:</h2>
    <img src="cid:qrcode">
    <p><a href="{invoice_url}">Cliquez ici pour télécharger</a></p>
  </body>
</html>
'''

sender.send_html(
    to_email="client@example.com",
    subject="Votre facture FA001",
    html_body=html,
    inline_images={"qrcode": qr_code}
)
```

### Exemple 3: Email texte simple

```python
sender.send_text(
    to_email="user@example.com",
    subject="Rappel",
    text_body="Votre abonnement expire dans 7 jours."
)
```

## Gestion des erreurs

```python
try:
    result = sender.send_html(
        to_email="invalid-email",
        subject="Test",
        html_body="<p>Test</p>"
    )
    print(f"Email envoyé: {result.get('id')}")
except Exception as e:
    print(f"Erreur: {e}")
```

## Logging

Le module utilise **structlog** pour un logging structuré au format JSON avec rotation automatique.

### Configuration rapide

```bash
# Variables d'environnement
export EMAIL_LOG_LEVEL=INFO              # DEBUG, INFO, WARNING, ERROR
export EMAIL_LOG_DIR=/home/log/email_sender
export EMAIL_LOG_TO_SCREEN=1             # Afficher aussi sur stdout
```

### Événements loggés automatiquement

- **Envoi d'emails** - Destinataire, sujet, message_id, durée
- **Erreurs** - Code erreur, message, contexte complet
- **Appels API** - Endpoint, méthode, status code, temps de réponse
- **Templates** - ID template, paramètres

### Format des logs

```json
{
  "event": "email_sent",
  "level": "info",
  "timestamp": "2025-11-13T15:30:45.123456Z",
  "provider": "brevo",
  "to_email": "client@example.com",
  "subject": "Votre facture",
  "message_id": "abc123",
  "duration_ms": 245.6
}
```

### Documentation complète

Voir [LOGGING.md](LOGGING.md) pour:
- Configuration détaillée
- Liste complète des événements
- Analyse et monitoring
- Exemples d'utilisation

### Test du logging

```bash
cd /home/ccopol/Workspace/GCV/gcv/Gestion/python/email_sender
./test_logging.py
```

## Dépendances

- `requests` - Appels HTTP vers les APIs
- `structlog` - Logging structuré JSON
- `pillow` (optionnel) - Manipulation d'images (QR codes)
- `qrcode` (optionnel) - Génération de QR codes

## Providers supportés

### Mailgun

- Configuration requise: `api_key`, `domain`, `from_email`
- Documentation: https://documentation.mailgun.com/

### Brevo (SendinBlue)

- Configuration requise: `api_key`, `from_email`
- Documentation: https://developers.brevo.com/

## Limitations

- Mailgun: 300 emails/heure en plan gratuit
- Brevo: 300 emails/jour en plan gratuit
- Taille maximale des pièces jointes: Variable selon le provider

## Licence

Voir le fichier LICENSE du projet principal.
