#!/usr/bin/env python3
"""
Script pour envoyer une requête FNE au middleware à partir d'un fichier JSON.

Usage:
    send_json_facture_to_gcmiddleware.py <fichier.json> <event_type> [priority]

Arguments:
    fichier.json - Fichier JSON contenant les données de la facture
    event_type   - Type d'événement (fnedev_invoice_sign, fne_invoice_sign, etc.)
    priority     - Priorité de 1 (haute) à 10 (basse) (défaut: 3)

Variables d'environnement:
    BASE_URL - URL du middleware (défaut: http://localhost:8001)

Codes de sortie:
    0 - Succès
    1 - Erreur (fichier introuvable, JSON invalide, erreur réseau, etc.)
"""

import argparse
import json
import os
import sys
from pathlib import Path

import requests


def main() -> int:
    parser = argparse.ArgumentParser(
        description="Envoyer une requête FNE au middleware à partir d'un fichier JSON"
    )
    parser.add_argument("json_file", help="Fichier JSON contenant les données de la facture")
    parser.add_argument("event_type", help="Type d'événement (fnedev_invoice_sign, etc.)")
    parser.add_argument(
        "--priority",
        nargs="?",
        type=int,
        default=3,
        help="Priorité de 1 (haute) à 10 (basse) (défaut: 3)",
    )
    parser.add_argument("--url", help="URL du middleware (défaut: http://localhost:8001)",
                        default="http://localhost:8001")
    args = parser.parse_args()

    # Configuration
    json_file = Path(args.json_file)

    # Vérifier que le fichier existe
    if not json_file.is_file():
        print(f"Erreur: Fichier introuvable: {json_file}", file=sys.stderr)
        return 1

    # Lire et valider le JSON
    try:
        invoice_data = json.loads(json_file.read_text(encoding="utf-8"))
    except json.JSONDecodeError as e:
        print(f"Erreur: Le fichier n'est pas un JSON valide: {json_file}", file=sys.stderr)
        print(f"Détail: {e}", file=sys.stderr)
        return 1

    print(f"INFO: Envoi de {json_file} vers {args.url} (event={args.event_type}, priority={args.priority})")

    # Construire le payload
    payload = {
        "event_type": args.event_type,
        "data": invoice_data,
        "priority": args.priority,
    }

    # Envoyer la requête
    try:
        response = requests.post(
            f"{args.url}/submit",
            json=payload,
            headers={"Content-Type": "application/json"},
            timeout=30,
        )
    except requests.RequestException as e:
        print(f"Erreur: Échec de la connexion au middleware: {e}", file=sys.stderr)
        return 1

    # Vérifier le code de réponse
    if response.status_code in (200, 201, 202):
        try:
            body = response.json()
            request_id = body.get("request_id", "")
            print(f"SUCCESS: Requête acceptée (HTTP {response.status_code}, request_id={request_id})")
            print(json.dumps(body, indent=2, ensure_ascii=False))
        except json.JSONDecodeError:
            print(f"SUCCESS: Requête acceptée (HTTP {response.status_code})")
            print(response.text)
        return 0
    else:
        print(f"Erreur: Requête rejetée (HTTP {response.status_code})", file=sys.stderr)
        try:
            print(json.dumps(response.json(), indent=2, ensure_ascii=False), file=sys.stderr)
        except json.JSONDecodeError:
            print(response.text, file=sys.stderr)
        return 1


if __name__ == "__main__":
    sys.exit(main())