import argparse
from datetime import datetime
import json
from pathlib import Path
import sqlite3
import subprocess
import sys
import tempfile
import os

import pandas as pd
import jinja2

EDITIONS_VALTYPE_TEXT = 1
EDITIONS_VALTYPE_INT = 2
EDITIONS_VALTYPE_FLOAT = 3
EDITIONS_VALTYPE_BOOL = 4

TEMPLATES_ROOTDIR = Path(__file__).parent / "templates"
print("TEMPLATES_rOOTDIR", TEMPLATES_ROOTDIR)
#TEMPLATES_ROOTDIR= Path("/home/ccopol/Workspace/GCV/gcv.worktrees/feat/editions-v2/extensions/editions/templates")

parser = argparse.ArgumentParser(
    description="Creer des éditions (html, excel, pdf) à partir des données stockées dans sqlite3"
)
parser.add_argument("doc_id", type=int, help="Identifiant du document à éditer")
parser.add_argument("--output", type=Path, help="Nom des fichiers de sortie")
parser.add_argument(
    "--json", "-j", type=Path, help="Fichier json de configuration du document"
)
parser.add_argument(
    "--dbpath",
    type=Path,
    default=Path().home() / "extractions/editions.sqlite3",
    help="Chemin de la base de données",
)
parser.add_argument(
    "--only-html", action="store_true", help="Créer uniquement le fichier html"
)
parser.add_argument(
    "--only-pdf", action="store_true", help="Créer uniquement le fichier pdf"
)
parser.add_argument(
    "--only-excel", action="store_true", help="Créer uniquement le fichier excel"
)
parser.add_argument(
    "--force", action="store_true", help="Ecraser le fichier de sortie s'il existe déjà"
)

# Create a group of mutually exclusive arguments
group = parser.add_mutually_exclusive_group()
group.add_argument(
    "--lualatex", action="store_true", help="Utiliser lualatex pour générer le pdf"
)
group.add_argument(
    "--latexmk", action="store_true", help="Utiliser latexmk pour générer le pdf"
)
group.add_argument(
    "--arara", action="store_true", help="Utiliser arara pour générer le pdf"
)


def convert_data_line(row):
    if row["type"] == EDITIONS_VALTYPE_TEXT:
        return row["valeur"].strip()
    if row["type"] == EDITIONS_VALTYPE_INT:
        return int(row["valeur"])
    if row["type"] == EDITIONS_VALTYPE_FLOAT:
        return float(int(row["valeur"]) / (10 ** row["arrondi"]))
    if row["type"] == EDITIONS_VALTYPE_BOOL:
        return row["valeur"] == "T"
    return row["valeur"]


def reconstituer_lignes(df):
    # Grouper par rowidx puuis transformer en ligne
    new_df = pd.DataFrame()
    for name, group in df.groupby("rowidx"):
        line = group[["valeur"]]
        line.index = group["colidx"].to_list()
        line.columns = [name]
        new_df = pd.concat([new_df, line.transpose()])
    return new_df


def to_html(document, filename, index=True):
    df = document["data"]
    bootstrap_html = """<!doctype html>
    <html lang="fr">
    <head>
        <meta charset="utf-8">
        <meta name="viewport" content="width=device-width, initial-scale=1">
        <title>%%%title%%%</title>
        <meta name="description" content="%%%description%%%">
        <meta name="author" content="EMC2 GCV6 x.y.z">
        <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet" integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH" crossorigin="anonymous">
    </head>
    <body>
        <div class="container">
            <h1>%%%title%%%</h1>
            <div class="d-flex">
                <p class="author p-2">%%%created_by%%%</p>
                <p class="date p-2">%%%created_at%%%</p>
            </div>
            <p>%%%description%%%</p>
                %%%body_content%%%
        </div>
        <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js" integrity="sha384-YvpcrYf0tY3lHB60NNkmXc5s9fDVZLESaAA55NDzOxhy9GkcIdslK1eN7N6jIeHz" crossorigin="anonymous"></script>
    </body>
    </html>
    """
    with open(filename, "w") as f:
        f.write(
            bootstrap_html.replace(
                "%%%body_content%%%",
                df.to_html(
                    index=index, classes="table table-striped table-hover table-sm"
                ),
            )
            .replace("%%%title%%%", "Liste des Familles")
            .replace("%%%created_by%%%", document["created_by"])
            .replace("%%%created_at%%%", document["created_at"])
            .replace("%%%description%%%", document["description"])
        )


def build_pdf(document, args):
    output = args.output

    if not output:
        output = tempfile.NamedTemporaryFile(mode="w", delete=False, suffix=".tex")
        output = Path(output.name)
    if output.suffix == ".pdf":
       output = output.with_suffix(".tex")
    texfile = output.parent / f"{output.stem}.2.tex"

    columns = document["columns"]
    df = document["data"]
    tmpfile = texfile.with_suffix(".tmp")
    df.to_latex(
        tmpfile,
        columns=columns,
        index=False,
        float_format="%.2f",
    )
    # self.df.to_excel(self.texfile.with_suffix(".xlsx"), index=False)

    # print(tmpfile)

    tableau_content = []
    with open(tmpfile) as tmp:
        # On supprime les premières lignes jusqu'à l'entête des colonnes
        # sans garder la ligne \midrule
        while line := tmp.readline():
            if line.strip() == "\\midrule":
                break

        # On supprime tout après \bottomrule, y compris cette ligne
        while line := tmp.readline():
            if line.strip() == "\\bottomrule":
                break
            tableau_content.append(line)

    # # Supprime les lignes vides en fin de tableau
    # idxfam = -1
    # print(tableau_content[idxfam].split())
    # while tableau_content[idxfam].split() == "":
    #     tableau_content.pop(idxfam)
    #     idxfam -=1

    content = "".join(tableau_content)
    # print(content)
    # tmpfile.unlink()

    from analyse_donnees.etats import chore

    couleurs = chore.Couleurs.from_string("gris")
    template = chore.liste_jinja_env.get_template("Liste.tex.jinja")
    largeur = 0.7  # proportion de textwidth
    header = [columns]  # liste de listes d'en-tête de colonnes
    # colspec = "Q[l,1cm]Q[l,2cm]Q[l,3cm]Q[l,4cm]"
    colspec = "l" * len(columns)

    tableau = {
        "tabularray": "longtblr",
        "colspec": colspec,
        "header": header,
        "largeur": f"{largeur}\\textwidth",
        "couleurs": couleurs,
        # texte_avant="",
        "tableau_content": content,
    }

    with open(texfile, "w") as fp:
        print(template.render(**tableau), file=fp)
    # # texfile is a file object and it's already opened
    # print(template.render(**tableau), file=texfile)
    # texfile.close()

    latex_jinja_env = jinja2.Environment(
        block_start_string="\BLOCK{",
        block_end_string="}",
        variable_start_string="\VAR{",
        variable_end_string="}",
        comment_start_string="\#{",
        comment_end_string="}",
        line_statement_prefix="%%",
        line_comment_prefix="%#",
        trim_blocks=True,
        autoescape=False,
        loader=jinja2.FileSystemLoader(
            str( (TEMPLATES_ROOTDIR / "documents").absolute())
        ),
    )

    document = {
        "fontsize": 8,
        "todaytime": f"{datetime.now():%a %d/%m/%Y %Hh%M}",
        "template": "simple.tex",
    } | tableau

    template = latex_jinja_env.get_template(document["template"])
    with open(output, "w") as fp:
        print(template.render(files=[str(texfile)], **document), file=fp)
    args.latexmk=False
    args.arara = True
    if args.latexmk:
        # command = f"latexmk -lualatex --quiet {output}"
        command = f"latexmk -cd -lualatex {output}"
        # if args.quiet:
        #     command += " --quiet"
        x = subprocess.call(command.split(), env=os.environ)
    elif args.lualatex:
        for _ in range(2):
            command = f"lualatex --output-dir={output.parent} {output}"
            x = subprocess.call(command.split(), env=os.environ)
    elif args.arara:
        options = ["-v", "--log"]
        # options.append("-v" if verbose else "-s")
        # options.append("--log" if debug > 0 else "")
        command = f"arara {' '.join(options)} --working-directory={output.parent} {output}"
        x = subprocess.call(command.split(), env=os.environ)

    else:
        print("Aucune action spécifiée")
        sys.exit(1)
    texfile.unlink()
    print(f"{output} créé")



def read_sqlite(doc_id, dbpath):
    conn = sqlite3.connect(str(dbpath))

    c = conn.cursor()
    c.execute(
        "SELECT descr_id, created_at, created_by, description FROM document where id = ?;",
        (doc_id,),
    )
    result = c.fetchone()
    if result is None:
        print(f"Document {doc_id} not found")
        sys.exit(1)
    descr_id, created_at, created_by, description = result

    document = {
        "created_by": created_by,
        "created_at": created_at,
        "description": description,
    }

    document["columns"] = pd.read_sql(
        "SELECT * FROM dcolonne where descr_id = ? ORDER BY idx;",
        conn,
        params=(descr_id,),
    )["dname"].to_list()
    document["data"] = pd.read_sql(
        "SELECT * FROM donnee where doc_id = ?;", conn, params=(doc_id,)
    )
    conn.close()

    document["data"]["colidx"] = document["data"]["colidx"].apply(
        lambda x: document["columns"][x]
    )
    document["data"].drop(columns=["id", "doc_id"], inplace=True)
    # transformer les données str en fonction du type
    document["data"]["value"] = document["data"].apply(convert_data_line, axis=1)
    document["data"] = reconstituer_lignes(document["data"])

    return document

def write_loading_html(output):

    with open(output, "w") as html:
        print("""
        <div class="spinner"></div>

<style>
.spinner {
   width: 56px;
   height: 56px;
   display: grid;
   border: 4.5px solid #0000;
   border-radius: 50%;
   border-color: #dbdcef #0000;
   animation: spinner-e04l1k 1s infinite linear;
}

.spinner::before,
.spinner::after {
   content: "";
   grid-area: 1/1;
   margin: 2.2px;
   border: inherit;
   border-radius: 50%;
}

.spinner::before {
   border-color: #474bff #0000;
   animation: inherit;
   animation-duration: 0.5s;
   animation-direction: reverse;
}

.spinner::after {
   margin: 8.9px;
}

@keyframes spinner-e04l1k {
   100% {
      transform: rotate(1turn);
   }
}
</style>""",file=html)


def build_html(document, args):
    output = args.output
    if not output:
        output = tempfile.NamedTemporaryFile(mode="w", delete=False, suffix=".html")
        output = Path(output.name)
    if output.suffix != ".html":
       output = output.with_suffix(".html")
    to_html(document, str(output), index=False)
    print(f"Fichier {output} créé")

    # if output.exists():
    #     # from webbrowser import open
    #     # open(output)

    #     from xhtml2pdf import pisa

    #     with open(output, "r") as f:
    #         html = f.read()
    #     with open(output.with_suffix(".html.pdf"), "w+b") as f:
    #         pisa.CreatePDF(html, dest=f)
    #     print(f"Fichier {output.with_suffix('.html.pdf')} créé")b




def build_exel(document, args):
    document["data"].to_excel(str(args.output) + ".xlsx", index=False)


if __name__ == "__main__":
    args = parser.parse_args()

    doc_id = args.doc_id
    dbpath = args.dbpath

    # if not args.output.is_file() or args.force:
    document = read_sqlite(doc_id, args.dbpath)

    col_indexes = list(range(len(document["columns"])))
    # if columns is None:
    #     columns = colonne_names
    #     colonne_index = list(range(len(columns)))
    # else:
    #     colonne_index = [colonne_names.index(colname) for colname in columns]

    pd.set_option("colheader_justify", "center")  # FOR TABLE <th>
    if args.only_html:
        build_html(document, args)
        sys.exit(0)
    if args.only_excel:
        build_exel(document, args)
        sys.exit(0)

    if args.only_pdf:
        build_pdf(document, args)
        sys.exit(0)

    build_html(document, args)
    build_exel(document, args)
    build_pdf(document, args)

    print("Fichier %s créé", args.output)
