__all__ = ["Liste", "LongListe"]
available_types = __all__

from pathlib import Path
import tempfile
import sqlite3

import numpy as np
import pandas as pd

from . import chore

class Liste:

    def __init__(self, tableau=None, tabularray="tblr", *args, **kwargs):
        self._tableau = tableau.copy() or None
        self.tableau = tableau or None
        self.tabularray = tabularray or None

        assert tabularray in ["longtblr", "tblr"]

        # self.select_latex_template()
        self.set_largeur()
        self.set_style()
        if "sqlite" in tableau:
            self.sqlite = Path(tableau["sqlite"])
            self.csv = None
        elif "csv" in tableau:
            tableau["csv"] = Path(tableau["csv"])
            self.csvfile = tableau["csv"]
            self.sqlite = None

        #self.texfile = Path(self.csvfile.stem + ".tex")
        self.texfile = Path(tempfile.mkstemp(prefix="tableau", suffix=".tex")[1])

        if "alignement" not in tableau:
            self.tableau["alignement"] = "center"

        # self._sanity_check(tableau)

        if self.csv is not None:
            self.read_csv()
        elif self.sqlite is not None:
            self.read_sqlite( tableau )
        else:
            self.sqlite = Path().home() / "extractions/editions.sqlite3"
            self.read_sqlite( tableau )

        self.variables_to_template = {"couleurs":self.couleurs}


    def _sanity_check(self, tableau):
        if "colonnes" not in self.tableau:
            self.tableau["colonnes"] = None

        if "header" not in self.tableau:
            raise AttributeError("Le paramètre header est manquant",tableau)

    def read_csv(self, sep=";"):
        self.df = pd.read_csv(self.csvfile, sep=sep)
        self.df = self.df.replace({np.nan: ""})
        if self.tableau["colonnes"] is None:
            self.tableau["colonnes"] = self.df.columns.to_list()

    def read_sqlite(self, tableau):
        doc_id = tableau.get("doc_id")
        if doc_id is None:
            raise AttributeError("Le paramètre doc_id est manquant",tableau)

        # Read sqlite query results into a pandas DataFrame
        con = sqlite3.connect("data/portal_mammals.sqlite")
        df = pd.read_sql_query(f"SELECT * from donnee where doc_id = {doc_id}", con)

        # Verify that result of SQL query is stored in the dataframe
        print(df.head())

        con.close()
        # sql = f"SELECT * FROM {doc_id}"
        # self.df = pd.read_sql_query(tableau["sqlite"], tableau["sql"])
        # self.df = self.df.replace({np.nan: ""})
        # if self.tableau["colonnes"] is None:
        #     self.tableau["colonnes"] = self.df.columns.to_list()


    def to_latex(self):
        tmpfile = self.texfile.with_suffix(".tmp")
        self.df.to_latex(
            tmpfile,
            columns= self.tableau["colonnes"],
            index=False,
            float_format="%.2f",
        )
        self.df.to_excel(self.texfile.with_suffix(".xlsx"), index=False)

        tableau_content = []
        with open(tmpfile) as tmp:
            # On supprime les premières lignes jusqu'à l'entête des colonnes
            # sans garder la ligne \midrule
            while line:=tmp.readline():
                if line.strip()=="\\midrule":
                    break

            # On supprime tout après \bottomrule, y compris cette ligne
            while line:=tmp.readline():
                if line.strip()=="\\bottomrule":
                    break
                tableau_content.append(line)


        # # Supprime les lignes vides en fin de tableau
        # idxfam = -1
        # print(tableau_content[idxfam].split())
        # while tableau_content[idxfam].split() == "":
        #     tableau_content.pop(idxfam)
        #     idxfam -=1


        self.tableau["tableau_content"] = "".join(tableau_content)
        tmpfile.unlink()
        # print(self.tableau["tableau_content"])
        # print(tmpfile)
        # import sys
        # sys.exit(0)

        template = chore.liste_jinja_env.get_template("Liste.tex.jinja")
        with open(self.texfile, "w") as fp:
            print(template.render(tabularray=self.tabularray, **self.variables_to_template, **self.tableau), file=fp)


    def set_largeur(self):
        if "largeur" not in self.tableau or self.tableau["largeur"] in [0, 1]:
            # Prendre toute la largeur de la page
            self.tableau["largeur"] = "\\textwidth"
        elif 0<self.tableau["largeur"]<1:
            self.tableau["largeur"] = f'{self.tableau["largeur"]:.1f}'+"\\textwidth"

    def set_style(self):
        if "style" in self.tableau:
            self.couleurs = chore.Couleurs.from_string(self.tableau["style"])
        else:
            self.couleurs = chore.Couleurs(None, None)



class LongListe(Liste):

    def __init__(self, tableau=None, tabularray="longtblr", *args, **kwargs):
        Liste.__init__(self, tableau, tabularray, *args, **kwargs)
