#!/bin/bash

# Couleurs
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
CYAN='\033[0;36m'
NC='\033[0m' # No Color

# Configuration
DBNAME="gc"
TABLE="gc_mvt2"
INDEXNAME1="ix_mvt2_01"
ISQL="isql"
BATCH_SIZE=40  # Nombre de ROWID à traiter par lot

SCRIPT_NAME=$(basename $0 .sh)

if [ -z "$INFORMIXDIR" ] || [ -z "$INFORMIXSERVER" ] || [ -z "$DBPATH" ]; then
    echo -e "${RED}Erreur: Les variables d'environnement INFORMIXDIR, INFORMIXSERVER et DBPATH doivent etre definies.${NC}"
    exit 1
fi

if [ ! -d "$DBPATH" ]; then
    echo -e "${RED}Erreur: Le chemin de la base de donnees DBPATH=$DBPATH n'existe pas.${NC}"
    exit 1
fi

DBPATHNAME=$(basename $DBPATH)
# Fichiers temporaires
ERRORLOG=/home/log/${DBPATHNAME}/${SCRIPT_NAME}.log
SQLFILE=/home/log/${DBPATHNAME}/${SCRIPT_NAME}.sql
ROWID_FILE=/home/log/${DBPATHNAME}/${SCRIPT_NAME}.rowids
SCHEMA_BEFORE=/home/log/${DBPATHNAME}/${SCRIPT_NAME}_schema_before.sql
SCHEMA_AFTER=/home/log/${DBPATHNAME}/${SCRIPT_NAME}_schema_after.sql

echo -e "${CYAN}>> Etape 1 : Sauvegarde du schema avant modification...${NC}"

# Sauvegarder le schéma de la table avant modification
dbschema -d $DBNAME -t $TABLE -ss > "$SCHEMA_BEFORE" 2>/dev/null
echo -e "${GREEN}  ✓ Schema sauvegarde : $SCHEMA_BEFORE${NC}"

echo -e "${CYAN}>> Etape 2 : Repair de la table $TABLE et capture des erreurs...${NC}"

# Execution du REPAIR TABLE et redirection de stderr vers ERRORLOG
echo "repair table $TABLE" | $ISQL gc > "$ERRORLOG" 2>&1

echo -e "${CYAN}>> Etape 3 : Extraction des ROWID des doublons...${NC}"

# Extraire les rowid à partir de lignes comme : ERROR: duplicate key value, record 722893
grep -E "ERROR: duplicate key value, record [0-9]+" "$ERRORLOG" \
    | grep -oE "[0-9]+" > "$ROWID_FILE"

if [ ! -s "$ROWID_FILE" ]; then
    echo -e "${GREEN}  ✓ Aucun doublon trouve.${NC}"
    rm -f "$ERRORLOG" "$SQLFILE" "$ROWID_FILE"
    exit 0
fi

ROWID_COUNT=$(wc -l < "$ROWID_FILE")
echo -e "${YELLOW}  ⚠ Nombre de doublons detectes : $ROWID_COUNT${NC}"

# Generer le fichier SQL pour supprimer les index
cat <<EOF > "$SQLFILE"
-- Supprimer les index existants
DROP INDEX $INDEXNAME1;
EOF

echo -e "${CYAN}>> Etape 4 : Suppression des index...${NC}"
$ISQL $DBNAME < "$SQLFILE"

# Supprimer les doublons par lots
echo -e "${CYAN}>> Etape 5 : Suppression des doublons (par lots de $BATCH_SIZE)...${NC}"

BATCH_NUM=0
TOTAL_DELETED=0
ROWID_BATCH=()

while IFS= read -r line; do
    ROWID_BATCH+=("$line")

    # Si le batch est plein
    if [ ${#ROWID_BATCH[@]} -eq $BATCH_SIZE ]; then
        BATCH_NUM=$((BATCH_NUM + 1))
        ROWID_LIST=$(IFS=,; echo "${ROWID_BATCH[*]}")

        echo -e "${BLUE}  Batch $BATCH_NUM : suppression de ${#ROWID_BATCH[@]} lignes...${NC}"

        cat <<EOFDELETE > "$SQLFILE"
DELETE FROM $TABLE WHERE rowid IN ($ROWID_LIST);
EOFDELETE
        $ISQL $DBNAME < "$SQLFILE"

        TOTAL_DELETED=$((TOTAL_DELETED + ${#ROWID_BATCH[@]}))
        ROWID_BATCH=()
    fi
done < "$ROWID_FILE"

# Traiter le dernier batch s'il reste des éléments
if [ ${#ROWID_BATCH[@]} -gt 0 ]; then
    BATCH_NUM=$((BATCH_NUM + 1))
    ROWID_LIST=$(IFS=,; echo "${ROWID_BATCH[*]}")

    echo -e "${BLUE}  Batch $BATCH_NUM (final) : suppression de ${#ROWID_BATCH[@]} lignes...${NC}"

    cat <<EOFDELETE > "$SQLFILE"
DELETE FROM $TABLE WHERE rowid IN ($ROWID_LIST);
EOFDELETE
    $ISQL $DBNAME < "$SQLFILE"

    TOTAL_DELETED=$((TOTAL_DELETED + ${#ROWID_BATCH[@]}))
fi

echo -e "${GREEN}  ✓ Total supprime : $TOTAL_DELETED lignes${NC}"

# Recreer les index
echo -e "${CYAN}>> Etape 6 : Recreation des index...${NC}"
cat <<EOF > "$SQLFILE"
-- Recreer les index
create unique index ix_mvt2_01 on gc_mvt2 (nsk);
EOF

$ISQL $DBNAME < "$SQLFILE"

echo -e "${CYAN}>> Etape 7 : Sauvegarde du schema après modification...${NC}"

# Sauvegarder le schéma de la table après modification
dbschema -d $DBNAME -t $TABLE -ss > "$SCHEMA_AFTER" 2>/dev/null
echo -e "${GREEN}  ✓ Schema sauvegarde : $SCHEMA_AFTER${NC}"

# Comparer les schémas
echo -e "${CYAN}>> Etape 8 : Verification des index...${NC}"

# Extraire uniquement les lignes CREATE INDEX des deux fichiers
INDEXES_BEFORE=$(grep -i "create.*index" "$SCHEMA_BEFORE" | sort)
INDEXES_AFTER=$(grep -i "create.*index" "$SCHEMA_AFTER" | sort)

if [ "$INDEXES_BEFORE" == "$INDEXES_AFTER" ]; then
    echo -e "${GREEN}  ✓ Les index ont ete correctement recrees${NC}"
else
    echo -e "${RED}  ⚠ ATTENTION : Les index ne correspondent pas exactement${NC}"
    echo -e "${YELLOW}  Utilisez 'diff $SCHEMA_BEFORE $SCHEMA_AFTER' pour voir les differences${NC}"
fi

# Nettoyage
echo -e "${CYAN}>> Etape 9 : Recapitulatif...${NC}"
echo -e "  Erreurs repair    : ${YELLOW}${ERRORLOG}${NC}"
echo -e "  Requetes SQL      : ${YELLOW}${SQLFILE}${NC}"
echo -e "  ROWIDs doublons   : ${YELLOW}${ROWID_FILE}${NC}"
echo -e "  Schema avant      : ${YELLOW}${SCHEMA_BEFORE}${NC}"
echo -e "  Schema apres      : ${YELLOW}${SCHEMA_AFTER}${NC}"
echo ""
echo -e "${GREEN}✓ Script termine avec succes.${NC}"
