#!/home/local/venv/bin/python3.11
# -*- coding: utf-8 -*-

"""
Script d'envoi quotidien des logs vers Loki
Compatible Python 3.11.10
Usage:
  python3 send-logs-daily.py              # Envoie uniquement les logs de la veille
  python3 send-logs-daily.py --all        # Envoie tous les fichiers .log-YYYYMMDD
  python3 send-logs-daily.py --date YYYYMMDD  # Envoie les logs d'une date spécifique

Attention ! sur la centos6 veuillez utiliser `send-logs-wrapper.sh`
Cron: 0 2 * * * /home/local/venv/bin/python3 /home/local/loki/scripts/send-logs-daily.py
"""

import json
import gzip
import bz2
import re
import time
import socket
import ssl
import tomllib
import logging
from pathlib import Path
from datetime import datetime, timedelta
from typing import List, Dict, Tuple, Optional
from urllib.request import Request, urlopen
from urllib.error import URLError, HTTPError
import base64

import typer
import structlog

# Créer un contexte SSL qui accepte les certificats auto-signés
ssl_context = ssl.create_default_context()
ssl_context.check_hostname = False
ssl_context.verify_mode = ssl.CERT_NONE

# ==================== CONFIGURATION ====================

def load_config(config_file: Path):
    """Charge la configuration depuis le fichier TOML"""
    if not config_file.exists():
        raise FileNotFoundError(f"Fichier de configuration non trouve: {config_file}")

    with open(config_file, 'rb') as f:
        return tomllib.load(f)

# ==================== FONCTIONS ====================

def get_yesterday_date() -> str:
    """Retourne la date de la veille au format YYYY-MM-DD"""
    yesterday = datetime.now() - timedelta(days=1)
    return yesterday.strftime('%Y-%m-%d')


def get_yesterday_date_compact() -> str:
    """Retourne la date de la veille au format YYYYMMDD"""
    yesterday = datetime.now() - timedelta(days=1)
    return yesterday.strftime('%Y%m%d')


def extract_date_from_filename(filename: str) -> str | None:
    """
    Extrait la date du nom de fichier selon le format:
    - *.log-YYYYMMDD -> YYYYMMDD (date du log)
    Retourne None si pas de correspondance
    """
    # Format: *.log-YYYYMMDD
    match = re.search(r'\.log-(\d{8})$', filename)
    if match:
        return match.group(1)

    return None


def extract_module_from_filename(filename: str) -> str | None:
    """
    Extrait le module du nom de fichier selon le format:
    user-module-nunsoc-numcais-numpost...
    Retourne le module ou None si extraction impossible
    """
    # Retirer l'extension .log-YYYYMMDD si présente
    base_name = re.sub(r'\.log-\d{8}$', '', filename)
    # Retirer autres extensions (.log, .gz, .bz2)
    base_name = re.sub(r'\.(log|gz|bz2)$', '', base_name)

    parts = base_name.split('-')
    if len(parts) >= 2:
        return parts[1]  # Le module est en position 1

    return None


def should_process_file(file_path: Path, target_date: Optional[str] = None, process_all: bool = False) -> bool:
    """
    Détermine si le fichier doit être traité
    Basé sur le format *.log-YYYYMMDD

    Args:
        file_path: Chemin du fichier
        target_date: Date spécifique au format YYYYMMDD (None = veille)
        process_all: Si True, traite tous les fichiers *.log-YYYYMMDD
    """
    filename = file_path.name

    # Extraire la date du fichier
    file_date = extract_date_from_filename(filename)
    if not file_date:
        # Fichiers sans date dans le nom ne sont pas traités
        return False

    # Mode: traiter tous les fichiers
    if process_all:
        return True

    # Mode: date spécifique
    if target_date:
        return file_date == target_date

    # Mode par défaut: veille
    yesterday_compact = get_yesterday_date_compact()
    return file_date == yesterday_compact


def sanitize_line(line: str) -> str:
    """
    Supprime tous les caractères non-ASCII (UTF-8 étendu)
    Garde uniquement les caractères ASCII (0-127)
    """
    # Convertir en ASCII, ignorer les caractères non-ASCII
    return line.encode('ascii', errors='ignore').decode('ascii')


def read_log_file(file_path: Path) -> List[str]:
    """
    Lit un fichier de log (normal, .gz, ou .bz2)
    Pour les fichiers *.log-YYYYMMDD, retourne tout le contenu (pas de filtre par date)
    Supprime les caractères UTF-8 non-ASCII
    """
    logger = structlog.get_logger()
    lines = []
    encoding = "cp850"
    try:
        if file_path.suffix == '.gz':
            with gzip.open(file_path, 'rt', encoding=encoding, errors='ignore') as f:
                lines = [line.strip() for line in f if line.strip()]
        elif file_path.suffix == '.bz2':
            with bz2.open(file_path, 'rt', encoding=encoding, errors='ignore') as f:
                lines = [line.strip() for line in f if line.strip()]
        else:
            with open(file_path, 'r', encoding=encoding, errors='ignore') as f:
                lines = [line.strip() for line in f if line.strip()]
    except Exception as e:
        logger.error("erreur_lecture_fichier", fichier=str(file_path), erreur=str(e))

    return lines


def build_loki_payload(logs: List[str], filename: str, filepath: str, job: str, hostname: str) -> Dict:
    """
    Construit le payload JSON pour l'API Loki
    """
    # Convertir les logs en format Loki (timestamp, message)
    values = []
    for log_line in logs:
        # Timestamp en nanosecondes
        timestamp_ns = str(int(time.time() * 1_000_000_000))
        values.append([timestamp_ns, log_line])

    # Extraire le module du nom de fichier
    stream_labels = {
        "job": job,
        "host": hostname,
        "filename": filename,
        "path": filepath,
    }

    module = extract_module_from_filename(filename)
    if module:
        stream_labels["module"] = module

    payload = {
        "streams": [
            {
                "stream": stream_labels,
                "values": values
            }
        ]
    }

    return payload


def send_to_loki(payload: Dict, loki_push_api: str, loki_user: Optional[str], loki_password: Optional[str]) -> Tuple[bool, str]:
    """
    Envoie le payload à Loki via l'API HTTP
    Retourne (success, message)
    """
    try:
        json_data = json.dumps(payload).encode('utf-8')

        request = Request(
            loki_push_api,
            data=json_data,
            headers={'Content-Type': 'application/json'}
        )

        # Authentification si configurée
        if loki_user and loki_password:
            credentials = f"{loki_user}:{loki_password}"
            encoded_credentials = base64.b64encode(credentials.encode()).decode()
            request.add_header('Authorization', f'Basic {encoded_credentials}')

        with urlopen(request, timeout=30, context=ssl_context) as response:
            response_data = response.read().decode('utf-8')
            if response.status == 204 or response.status == 200:
                return True, "OK"
            else:
                return False, f"HTTP {response.status}: {response_data}"

    except HTTPError as e:
        return False, f"HTTP Error {e.code}: {e.read().decode('utf-8', errors='ignore')}"
    except URLError as e:
        return False, f"URL Error: {e.reason}"
    except Exception as e:
        return False, f"Exception: {str(e)}"


def process_log_file(
    file_path: Path,
    target_date: Optional[str],
    process_all: bool,
    loki_push_api: str,
    loki_user: Optional[str],
    loki_password: Optional[str],
    job: str,
    hostname: str
):
    """
    Traite un fichier de log et l'envoie à Loki

    Args:
        file_path: Chemin du fichier
        target_date: Date spécifique au format YYYYMMDD (None = veille)
        process_all: Si True, traite tous les fichiers *.log-YYYYMMDD
        loki_push_api: URL de l'API Loki
        loki_user: Utilisateur Loki (optionnel)
        loki_password: Mot de passe Loki (optionnel)
        job: Label job pour Loki
        hostname: Hostname pour les labels
    """
    logger = structlog.get_logger()

    # Vérifier si le fichier doit être traité
    if not should_process_file(file_path, target_date, process_all):
        logger.debug("fichier_ignore", fichier=str(file_path))
        return

    logger.info("traitement_fichier", fichier=str(file_path))

    # Lire les logs
    logs = read_log_file(file_path)

    if not logs:
        logger.warning("aucun_log_dans_fichier", fichier=str(file_path))
        return

    logger.info("lignes_a_envoyer", count=len(logs), fichier=str(file_path))

    # Construire le payload
    filename = file_path.name
    filepath = str(file_path.parent)
    payload = build_loki_payload(logs, filename, filepath, job, hostname)

    # Envoyer à Loki
    success, message = send_to_loki(payload, loki_push_api, loki_user, loki_password)

    if success:
        logger.info("envoi_succes", fichier=str(file_path))
    else:
        logger.error("erreur_envoi", fichier=str(file_path), reponse=message)


def expand_patterns(patterns: List[str]) -> List[Path]:
    """
    Expand les patterns glob et retourne la liste des fichiers
    """
    logger = structlog.get_logger()
    files = []
    for pattern in patterns:
        # Si c'est un chemin absolu avec wildcards
        if '*' in pattern:
            # Trouver le premier niveau sans wildcard
            parts = Path(pattern).parts
            for i, part in enumerate(parts):
                if '*' in part:
                    # Construire le parent sans double slash
                    if i > 0:
                        parent = Path(*parts[:i])
                    else:
                        parent = Path('/')
                    glob_pattern = str(Path(*parts[i:]))

                    try:
                        found_files = list(parent.glob(glob_pattern))
                        files.extend([f for f in found_files if f.is_file()])
                    except Exception as e:
                        logger.warning("erreur_glob", pattern=pattern, erreur=str(e))
                    break
        else:
            # Chemin direct
            p = Path(pattern)
            if p.is_file():
                files.append(p)

    return files


def cleanup_old_logs(retention_days: int, log_dir: Path):
    """
    Supprime les anciens logs d'envoi
    """
    logger = structlog.get_logger()
    try:
        cutoff_time = time.time() - (retention_days * 86400)
        for log_file in log_dir.glob("send-*.log"):
            if log_file.stat().st_mtime < cutoff_time:
                log_file.unlink()
                logger.info("suppression_ancien_log", fichier=str(log_file))
    except Exception as e:
        logger.error("erreur_nettoyage_logs", erreur=str(e))


# ==================== MAIN ====================

app = typer.Typer(help="Script d'envoi quotidien des logs vers Loki")

@app.command()
def main(
    date: Optional[str] = typer.Option(
        None,
        "--date",
        help="Date spécifique au format YYYYMMDD"
    ),
    all: bool = typer.Option(
        False,
        "--all",
        help="Envoyer tous les fichiers .log-YYYYMMDD"
    ),
    config: Path = typer.Option(
        Path(__file__).parent / "config.toml",
        "--config",
        "-c",
        help="Chemin vers le fichier de configuration TOML"
    )
):
    """
    Envoie les logs vers Loki.

    Par défaut, envoie uniquement les logs de la veille.
    """
    # Charger la configuration
    config_data = load_config(config)

    # Configuration Loki
    LOKI_URL = config_data['loki']['url']
    LOKI_PUSH_API = f"{LOKI_URL}/loki/api/v1/push"
    LOKI_USER = config_data['loki'].get('user')
    LOKI_PASSWORD = config_data['loki'].get('password')

    # Patterns de logs
    LOGS_PATTERNS = config_data['logs']['patterns']

    # Configuration output
    LOG_DIR = Path(config_data['output']['log_dir'])
    LOG_FILE = LOG_DIR / f"send-{datetime.now().strftime('%Y%m%d')}.log"
    RETENTION_DAYS = config_data['output'].get('retention_days', 30)

    # Labels
    JOB = config_data['labels']['job']
    HOSTNAME = config_data['labels'].get('hostname', socket.gethostname())

    # Configurer le logging
    LOG_DIR.mkdir(parents=True, exist_ok=True)

    structlog.configure(
        processors=[
            structlog.contextvars.merge_contextvars,
            structlog.processors.add_log_level,
            structlog.processors.TimeStamper(fmt="iso"),
            structlog.processors.JSONRenderer()
        ],
        wrapper_class=structlog.make_filtering_bound_logger(logging.INFO),
        context_class=dict,
        logger_factory=structlog.PrintLoggerFactory(file=open(LOG_FILE, 'a')),
        cache_logger_on_first_use=False,
    )

    logger = structlog.get_logger()

    logger.info("demarrage_envoi_logs", config_file=str(config), separator="=" * 60)
    logger.info("envoi_logs_loki", timestamp=datetime.now().isoformat())

    # Validation de la date si fournie
    if date:
        if not re.match(r'^\d{8}$', date):
            logger.error("format_date_invalide", date=date, format_attendu="YYYYMMDD")
            raise typer.BadParameter(f"Format de date invalide: {date}. Attendu: YYYYMMDD")
        logger.info("mode_date_specifique", date=date)
    elif all:
        logger.info("mode_tous_fichiers")
    else:
        logger.info("mode_veille", date=get_yesterday_date_compact())

    # Expansion des patterns
    log_files = expand_patterns(LOGS_PATTERNS)

    if not log_files:
        logger.warning("aucun_fichier_trouve")
        return

    logger.info("fichiers_trouves", count=len(log_files))

    # Traiter chaque fichier
    processed_count = 0
    for log_file in log_files:
        try:
            # Vérifier si le fichier doit être traité avant
            if should_process_file(log_file, date, all):
                process_log_file(
                    log_file,
                    date,
                    all,
                    LOKI_PUSH_API,
                    LOKI_USER,
                    LOKI_PASSWORD,
                    JOB,
                    HOSTNAME
                )
                processed_count += 1
        except Exception as e:
            logger.error("erreur_traitement_fichier", fichier=str(log_file), erreur=str(e))

    logger.info("envoi_termine",
                fichiers_traites=processed_count,
                timestamp=datetime.now().isoformat(),
                separator="=" * 60)

    # Nettoyage
    cleanup_old_logs(RETENTION_DAYS, LOG_DIR)


if __name__ == "__main__":
    app()

