#!/usr/bin/env python3
"""
Convertisseur CSV vers SQLite3 pour les données de déploiement GCV6
Structure CSV: version;date;heure
Exemple: GCV6-6.06.009am;20250718;052026
"""

import sqlite3
import csv
import sys
import os

def create_deployment_db(db_path):
    """Crée la base de données SQLite avec la table de déploiement"""
    conn = sqlite3.connect(db_path)
    cursor = conn.cursor()
    
    cursor.execute("""
        CREATE TABLE IF NOT EXISTS deployments (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            version TEXT NOT NULL,
            date TEXT NOT NULL,
            time TEXT NOT NULL,
            created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
        )
    """)
    
    conn.commit()
    return conn

def convert_csv_to_sqlite(csv_path, db_path):
    """Convertit le fichier CSV en base SQLite"""
    if not os.path.exists(csv_path):
        print(f"Erreur: Le fichier CSV {csv_path} n'existe pas")
        return False
    
    conn = create_deployment_db(db_path)
    cursor = conn.cursor()
    
    # Vider la table avant insertion
    cursor.execute("DELETE FROM deployments")
    
    try:
        with open(csv_path, 'r', encoding='utf-8') as csvfile:
            reader = csv.reader(csvfile, delimiter=';')
            for row_num, row in enumerate(reader, 1):
                if len(row) >= 3:
                    version, date, time = row[0], row[1], row[2]
                    cursor.execute(
                        "INSERT INTO deployments (version, date, time) VALUES (?, ?, ?)",
                        (version, date, time)
                    )
                else:
                    print(f"Attention: Ligne {row_num} ignorée (format incorrect): {row}")
        
        conn.commit()
        print(f"Conversion terminée: {cursor.rowcount} enregistrements importés")
        return True
        
    except Exception as e:
        print(f"Erreur lors de la conversion: {e}")
        return False
    finally:
        conn.close()

def main():
    if len(sys.argv) != 3:
        print("Usage: python3 csv_to_sqlite.py <fichier.csv> <base.db>")
        print("Exemple: python3 csv_to_sqlite.py deploiement.csv deployments.db")
        sys.exit(1)
    
    csv_path = sys.argv[1]
    db_path = sys.argv[2]
    
    if convert_csv_to_sqlite(csv_path, db_path):
        print(f"Base de données créée avec succès: {db_path}")
    else:
        sys.exit(1)

if __name__ == "__main__":
    main()