#!/bin/bash

# =======================================================
# gcmiddleware    Script de demarrage du middleware GCV
# Genere automatiquement par Ansible
# Ne pas modifier manuellement - vos changements seront ecrases
# =======================================================
#
#
# chkconfig: 2345 95 05
# description: Middleware GCV - Routeur d'evenements vers APIs externes (FNE-CDI, etc.)
# processname: gcmiddleware
# config: /etc/gcmiddleware/config.toml
# pidfile: /var/run/gcmiddleware/gcmiddleware.pid
#
### BEGIN INIT INFO
# Provides:          gcmiddleware
# Required-Start:    $local_fs $network
# Required-Stop:     $local_fs $network
# Default-Start:     2 3 4 5
# Default-Stop:      0 1 6
# Short-Description: GCV Middleware
# Description:       Middleware GCV pour routage d'evenements vers APIs externes
### END INIT INFO

# Source function library
. /etc/init.d/functions

# Configuration par defaut (peut etre surchargee dans /etc/sysconfig/gcmiddleware)
MIDDLEWARE_NAME="gcmiddleware"
MIDDLEWARE_USER="gestcom"
MIDDLEWARE_GROUP="gestcom"
MIDDLEWARE_DIR="/home/local/bin/GCV6/python/site-packages/gcmiddleware"
MIDDLEWARE_CONFIG=""
MIDDLEWARE_PORT="8001"
MIDDLEWARE_HOST="0.0.0.0"
MIDDLEWARE_VENV="/home/local/venv"
MIDDLEWARE_PIDDIR="/var/run/gcmiddleware"
# MIDDLEWARE_PIDFILE est calcule apres chargement du sysconfig (voir plus bas)
MIDDLEWARE_PIDFILE=""
# MIDDLEWARE_LOGDIR est calcule automatiquement depuis DBPATH (voir plus bas)
# Format: /home/log/{dbname}/daily/ ou dbname = basename(DBPATH)
MIDDLEWARE_LOGDIR=""
MIDDLEWARE_LOGFILE=""
MIDDLEWARE_DEV_MODE="false"

# Timeout pour l'arret gracieux (secondes)
STOP_TIMEOUT=30

# Detecter le nom du service (pour support multi-instances via liens symboliques)
SERVICE_NAME=$(basename "$0")

# Charger la configuration personnalisee si elle existe
# Priorite: /etc/sysconfig/{nom-du-service} puis /etc/sysconfig/gcmiddleware
if [ -f "/etc/sysconfig/${SERVICE_NAME}" ]; then
    . "/etc/sysconfig/${SERVICE_NAME}"
elif [ -f /etc/sysconfig/gcmiddleware ]; then
    . /etc/sysconfig/gcmiddleware
fi

# Variables derivees (calculees apres chargement du sysconfig)
PYTHON="${MIDDLEWARE_VENV}/bin/python"
RETVAL=0
LOCKFILE="/var/lock/subsys/${MIDDLEWARE_NAME}"

# Calculer MIDDLEWARE_PIDFILE si non defini
if [ -z "${MIDDLEWARE_PIDFILE}" ]; then
    MIDDLEWARE_PIDFILE="${MIDDLEWARE_PIDDIR}/${MIDDLEWARE_NAME}.pid"
fi

# Calculer MIDDLEWARE_LOGDIR depuis DBPATH si non defini
# Format: /home/log/{dbname}/daily/ ou dbname = basename(DBPATH)
if [ -z "${MIDDLEWARE_LOGDIR}" ] && [ -n "${DBPATH}" ]; then
    DBPATHNAME=$(basename "${DBPATH}")
    MIDDLEWARE_LOGDIR="/home/log/${DBPATHNAME}/daily"
fi
# Fallback si DBPATH n'est pas defini
if [ -z "${MIDDLEWARE_LOGDIR}" ]; then
    MIDDLEWARE_LOGDIR="/var/log/gcmiddleware"
fi
# Calculer MIDDLEWARE_LOGFILE si non defini
if [ -z "${MIDDLEWARE_LOGFILE}" ]; then
    MIDDLEWARE_LOGFILE="${MIDDLEWARE_LOGDIR}/${MIDDLEWARE_NAME}.log"
fi

# Creer les repertoires necessaires
create_dirs() {
    if [ ! -d "${MIDDLEWARE_PIDDIR}" ]; then
        mkdir -p "${MIDDLEWARE_PIDDIR}"
        chown ${MIDDLEWARE_USER}:${MIDDLEWARE_GROUP} "${MIDDLEWARE_PIDDIR}"
    fi
    if [ ! -d "${MIDDLEWARE_LOGDIR}" ]; then
        mkdir -p "${MIDDLEWARE_LOGDIR}"
        chown ${MIDDLEWARE_USER}:${MIDDLEWARE_GROUP} "${MIDDLEWARE_LOGDIR}"
    fi
}

# Verifier les prerequis
check_prerequisites() {
    # Verifier que Python existe
    if [ ! -x "${PYTHON}" ]; then
        echo "ERREUR: Python non trouve: ${PYTHON}"
        echo "Installez le virtualenv: python3 -m venv ${MIDDLEWARE_VENV}"
        return 1
    fi

    # Verifier que le fichier de config existe
    if [ ! -f "${MIDDLEWARE_CONFIG}" ]; then
        echo "ERREUR: Configuration non trouvee: ${MIDDLEWARE_CONFIG}"
        echo "Copiez config.toml.example vers ${MIDDLEWARE_CONFIG}"
        return 1
    fi

    # Verifier que run.py existe
    if [ ! -f "${MIDDLEWARE_DIR}/run.py" ]; then
        echo "ERREUR: run.py non trouve: ${MIDDLEWARE_DIR}/run.py"
        return 1
    fi

    return 0
}

# Obtenir le PID du processus depuis le fichier PID
get_pid() {
    if [ -f "${MIDDLEWARE_PIDFILE}" ]; then
        cat "${MIDDLEWARE_PIDFILE}"
    fi
}

# Obtenir le PID du processus en cherchant par port (fallback)
get_pid_by_port() {
    # Chercher le processus run.py qui ecoute sur le port configure
    pgrep -f "run.py.*--port=${MIDDLEWARE_PORT}" 2>/dev/null | head -1
}

# Obtenir le PID reel (fichier PID ou recherche par port)
get_real_pid() {
    local pid=$(get_pid)
    # Verifier que le PID du fichier est valide
    if [ -n "${pid}" ] && kill -0 "${pid}" 2>/dev/null; then
        echo "${pid}"
        return
    fi
    # Fallback: chercher par port
    get_pid_by_port
}

# Verifier si le processus est en cours d'execution
is_running() {
    local pid=$(get_real_pid)
    if [ -n "${pid}" ] && kill -0 "${pid}" 2>/dev/null; then
        return 0
    fi
    return 1
}

# Demarrer le service
start() {
    echo -n "Demarrage de ${MIDDLEWARE_NAME}: "

    if is_running; then
        echo "deja en cours d'execution (PID: $(get_pid))"
        return 0
    fi

    check_prerequisites || return 1
    create_dirs

    # Preparer les options
    local dev_opt=""
    if [ "${MIDDLEWARE_DEV_MODE}" = "true" ]; then
        dev_opt="--dev"
    fi

    # Lancer le middleware en arriere-plan
    cd "${MIDDLEWARE_DIR}"

    # Export des variables d'environnement
    export MIDDLEWARE_CONFIG_PATH="${MIDDLEWARE_CONFIG}"
    export DBPATH="${DBPATH}"
    export LANG="fr_FR.UTF8"
    
    # Configuration OpenSSL et SQLite3 (requis pour Python 3.11+)
    export SQLITE3DIR=/opt/sqlite3
    export SSLDIR=/opt/openssl-3.2.1
    export PATH="${SQLITE3DIR}/bin:${SSLDIR}/bin${PATH:+:$PATH}"
    export LD_LIBRARY_PATH="${SQLITE3DIR}/lib:${SSLDIR}/lib64${LD_LIBRARY_PATH:+:$LD_LIBRARY_PATH}"

    daemon --user=${MIDDLEWARE_USER} --pidfile=${MIDDLEWARE_PIDFILE} \
        "cd ${MIDDLEWARE_DIR} && ${PYTHON} run.py \
            --config=${MIDDLEWARE_CONFIG} \
            --port=${MIDDLEWARE_PORT} \
            --host=${MIDDLEWARE_HOST} \
            ${dev_opt} \
            >> ${MIDDLEWARE_LOGFILE} 2>&1 & echo \$! > ${MIDDLEWARE_PIDFILE}"

    RETVAL=$?

    # Attendre un peu et verifier que le processus a bien demarre
    sleep 2

    if is_running; then
        echo_success
        echo
        touch "${LOCKFILE}"
    else
        echo_failure
        echo
        rm -f "${MIDDLEWARE_PIDFILE}"
        RETVAL=1
    fi

    return ${RETVAL}
}

# Arreter le service
stop() {
    echo -n "Arret de ${MIDDLEWARE_NAME}: "

    if ! is_running; then
        echo "non en cours d'execution"
        rm -f "${MIDDLEWARE_PIDFILE}" "${LOCKFILE}"
        return 0
    fi

    local pid=$(get_real_pid)

    # Tentative d'arret gracieux (SIGTERM)
    kill -TERM "${pid}" 2>/dev/null

    # Attendre l'arret gracieux
    local count=0
    while [ ${count} -lt ${STOP_TIMEOUT} ] && is_running; do
        sleep 1
        count=$((count + 1))
    done

    if is_running; then
        echo -n "(forcage) "
        # Forcer l'arret (SIGKILL)
        pid=$(get_real_pid)
        kill -9 "${pid}" 2>/dev/null
        # Tuer aussi les processus run.py sur ce port
        pkill -9 -f "run.py.*--port=${MIDDLEWARE_PORT}" 2>/dev/null
        # Tuer par le port (dernier recours)
        fuser -k ${MIDDLEWARE_PORT}/tcp 2>/dev/null
        sleep 1
    fi

    if ! is_running; then
        echo_success
        echo
        rm -f "${MIDDLEWARE_PIDFILE}" "${LOCKFILE}"
        RETVAL=0
    else
        echo_failure
        echo
        RETVAL=1
    fi

    return ${RETVAL}
}

# Forcer l'arret immediat
forcestop() {
    echo -n "Arret force de ${MIDDLEWARE_NAME}: "

    if ! is_running; then
        echo "non en cours d'execution"
        rm -f "${MIDDLEWARE_PIDFILE}" "${LOCKFILE}"
        return 0
    fi

    local pid=$(get_real_pid)

    # Arret force immediat (SIGKILL)
    kill -9 "${pid}" 2>/dev/null
    sleep 1

    # Tuer aussi les processus run.py et uvicorn sur ce port
    pkill -9 -f "run.py.*--port=${MIDDLEWARE_PORT}" 2>/dev/null
    pkill -9 -f "uvicorn.*${MIDDLEWARE_PORT}" 2>/dev/null
    # Tuer par le port (dernier recours)
    fuser -k ${MIDDLEWARE_PORT}/tcp 2>/dev/null
    sleep 1

    if ! is_running; then
        echo_success
        echo
        rm -f "${MIDDLEWARE_PIDFILE}" "${LOCKFILE}"
        RETVAL=0
    else
        echo_failure
        echo
        RETVAL=1
    fi

    return ${RETVAL}
}

# Redemarrer le service
restart() {
    stop
    sleep 2
    start
}

# Afficher le statut
status() {
    echo "=== Statut de ${MIDDLEWARE_NAME} ==="
    echo ""

    if is_running; then
        local pid=$(get_real_pid)
        local pid_source="detecte"
        if [ -f "${MIDDLEWARE_PIDFILE}" ] && [ "$(cat "${MIDDLEWARE_PIDFILE}")" = "${pid}" ]; then
            pid_source="fichier PID"
        else
            pid_source="detecte par port"
        fi
        echo "Etat:        EN COURS D'EXECUTION"
        echo "PID:         ${pid} (${pid_source})"
        echo "Port:        ${MIDDLEWARE_PORT}"
        echo "Host:        ${MIDDLEWARE_HOST}"
        echo "Config:      ${MIDDLEWARE_CONFIG}"
        echo "Log:         ${MIDDLEWARE_LOGFILE}"
        echo "DBPATH:      ${DBPATH}"
        echo "Mode dev:    ${MIDDLEWARE_DEV_MODE}"
        echo ""

        # Afficher les connexions reseau
        echo "Connexions reseau:"
        netstat -tlnp 2>/dev/null | grep ":${MIDDLEWARE_PORT}" || echo "  (aucune connexion trouvee)"
        echo ""

        # Afficher l'utilisation memoire/CPU
        echo "Ressources:"
        ps -p ${pid} -o pid,user,%cpu,%mem,vsz,rss,stat,start,time,cmd --no-headers 2>/dev/null | head -1
        echo ""

        # Verifier que l'API repond
        echo "Test API:"
        if curl -s -o /dev/null -w "%{http_code}" "http://127.0.0.1:${MIDDLEWARE_PORT}/health" 2>/dev/null | grep -q "200"; then
            echo "  /health: OK (200)"
        else
            echo "  /health: NON DISPONIBLE"
        fi

        return 0
    else
        echo "Etat:        ARRETE"
        echo "Config:      ${MIDDLEWARE_CONFIG}"
        echo ""

        # Verifier s'il y a un fichier PID orphelin
        if [ -f "${MIDDLEWARE_PIDFILE}" ]; then
            echo "ATTENTION: Fichier PID orphelin trouve: ${MIDDLEWARE_PIDFILE}"
            echo "Executez 'service ${MIDDLEWARE_NAME} forcestop' pour nettoyer"
        fi

        return 3
    fi
}

# Afficher les logs
showlog() {
    if [ -f "${MIDDLEWARE_LOGFILE}" ]; then
        tail -f "${MIDDLEWARE_LOGFILE}"
    else
        echo "Fichier log non trouve: ${MIDDLEWARE_LOGFILE}"
        return 1
    fi
}

# Afficher la configuration
showconfig() {
    echo "=== Configuration de ${MIDDLEWARE_NAME} ==="
    echo ""
    echo "Variables de service:"
    echo "  MIDDLEWARE_NAME:     ${MIDDLEWARE_NAME}"
    echo "  MIDDLEWARE_DIR:      ${MIDDLEWARE_DIR}"
    echo "  MIDDLEWARE_CONFIG:   ${MIDDLEWARE_CONFIG}"
    echo "  MIDDLEWARE_PORT:     ${MIDDLEWARE_PORT}"
    echo "  MIDDLEWARE_HOST:     ${MIDDLEWARE_HOST}"
    echo "  MIDDLEWARE_USER:     ${MIDDLEWARE_USER}"
    echo "  MIDDLEWARE_VENV:     ${MIDDLEWARE_VENV}"
    echo "  MIDDLEWARE_PIDFILE:  ${MIDDLEWARE_PIDFILE}"
    echo "  MIDDLEWARE_LOGFILE:  ${MIDDLEWARE_LOGFILE}"
    echo "  DBPATH:              ${DBPATH}"
    echo "  MIDDLEWARE_DEV_MODE: ${MIDDLEWARE_DEV_MODE}"
    echo ""

    if [ -f "/etc/sysconfig/gcmiddleware" ]; then
        echo "Fichier sysconfig: /etc/sysconfig/gcmiddleware"
        echo "Contenu:"
        cat /etc/sysconfig/gcmiddleware | sed 's/^/  /'
    else
        echo "Fichier sysconfig: (non defini)"
    fi
    echo ""

    if [ -f "${MIDDLEWARE_CONFIG}" ]; then
        echo "Fichier config.toml: ${MIDDLEWARE_CONFIG}"
        echo "Contenu (extrait):"
        head -30 "${MIDDLEWARE_CONFIG}" | sed 's/^/  /'
        echo "  ..."
    fi
}

# Point d'entree principal
case "$1" in
    start)
        start
        ;;
    stop)
        stop
        ;;
    forcestop)
        forcestop
        ;;
    restart)
        restart
        ;;
    status)
        status
        ;;
    log)
        showlog
        ;;
    config)
        showconfig
        ;;
    *)
        echo "Usage: $0 {start|stop|forcestop|restart|status|log|config}"
        echo ""
        echo "Commandes:"
        echo "  start      Demarrer le middleware"
        echo "  stop       Arreter le middleware (gracieux puis force apres ${STOP_TIMEOUT}s)"
        echo "  forcestop  Forcer l'arret immediat (SIGKILL)"
        echo "  restart    Redemarrer le middleware"
        echo "  status     Afficher l'etat detaille (PID, port, connexions, API)"
        echo "  log        Afficher les logs en temps reel (tail -f)"
        echo "  config     Afficher la configuration"
        RETVAL=2
        ;;
esac

exit ${RETVAL}
