# Guide d'utilisation de db_paths

## Vue d'ensemble

Le système Ansible pour GCV supporte désormais la gestion de **plusieurs chemins de bases de données** via la variable `db_paths` (liste), tout en maintenant la **compatibilité** avec l'ancienne variable `db_path` (chaîne unique).

## Compatibilité et migration

### Rétrocompatibilité

✅ **Votre configuration existante continue de fonctionner** sans modification.

```yaml
# Ancienne configuration (toujours supportée)
db_path: "/home/gestcom/dbs"
```

Cette configuration sera automatiquement convertie en :
```yaml
db_paths:
  - "/home/gestcom/dbs"
```

### Nouvelle configuration (multi-bases)

```yaml
# Nouvelle configuration (plusieurs bases)
db_paths:
  - "/home/gestcom/dbs"
  - "/home/bases/prod"
  - "/home/bases/test"
```

## Utilisation

### Configuration simple (une base)

**Option 1 : Utiliser db_path (ancien style)**
```yaml
# host_vars/server1.yml
db_path: "/home/gestcom/dbs"
```

**Option 2 : Utiliser db_paths (nouveau style)**
```yaml
# host_vars/server1.yml
db_paths:
  - "/home/gestcom/dbs"
```

### Configuration multi-bases

```yaml
# host_vars/server-multidb.yml
db_paths:
  - "/home/gestcom/dbs"
  - "/home/bases/production"
  - "/home/bases/development"
  - "/home/backup/databases"
```

## Fonctionnalités affectées

### 1. Création des dossiers annexes

Les dossiers `logos`, `csv`, `mod`, et `dyn` seront créés pour **chaque** base de données.

**Exemple avec une base :**
```
/home/gestcom/dbs/logos
/home/gestcom/dbs/csv
/home/gestcom/dbs/mod
/home/gestcom/dbs/dyn
```

**Exemple avec plusieurs bases :**
```
/home/gestcom/dbs/logos
/home/gestcom/dbs/csv
/home/gestcom/dbs/mod
/home/gestcom/dbs/dyn

/home/bases/prod/logos
/home/bases/prod/csv
/home/bases/prod/mod
/home/bases/prod/dyn

/home/bases/test/logos
/home/bases/test/csv
/home/bases/test/mod
/home/bases/test/dyn
```

### 2. Configuration logrotate

Les répertoires de logs seront créés pour **chaque** base de données.

**Structure générée :**
```
/home/log/dbs/hourly/
/home/log/dbs/daily/
/home/log/dbs/weekly/
/home/log/dbs/monthly/
/home/log/dbs/yearly/

/home/log/prod/hourly/
/home/log/prod/daily/
...
```

### 3. Nettoyage automatique (nettoyage_fichier.sh)

**Toutes** les bases seront exclues du nettoyage PDF/SLK.

**Script généré avec db_paths :**
```bash
find /home \
  -path /home/gestcom/dbs -prune -o \
  -path /home/bases/prod -prune -o \
  -path /home/bases/test -prune -o \
  -type f \( -name "*.pdf" -o -name "*.slk" \) -mtime +7 -exec rm -f {} \;
```

## Exemples par cas d'usage

### Cas 1 : Serveur standard (une base)

```yaml
# host_vars/prod-server.yml
db_path: "/home/gestcom/dbs"
```

### Cas 2 : Serveur multi-environnements

```yaml
# host_vars/multi-env-server.yml
db_paths:
  - "/home/bases/production"
  - "/home/bases/staging"
  - "/home/bases/development"
```

### Cas 3 : Serveur avec bases et archives

```yaml
# host_vars/archive-server.yml
db_paths:
  - "/home/gestcom/dbs"
  - "/home/gestcom/archives/2024"
  - "/home/gestcom/archives/2023"
  - "/home/backup/databases"
```

### Cas 4 : Migration progressive

```yaml
# Phase 1 : Utiliser l'ancienne variable
db_path: "/home/gestcom/dbs"

# Phase 2 : Ajouter une deuxième base (migrer vers db_paths)
db_paths:
  - "/home/gestcom/dbs"
  - "/home/bases/new_db"

# Phase 3 : Continuer à ajouter des bases
db_paths:
  - "/home/gestcom/dbs"
  - "/home/bases/new_db"
  - "/home/bases/another_db"
```

## Configuration par groupe

### Groupe production

```yaml
# group_vars/production.yml
db_paths:
  - "/home/gestcom/dbs"
  - "/home/backup/databases"
```

### Groupe développement

```yaml
# group_vars/development.yml
db_paths:
  - "/home/bases/dev"
  - "/home/bases/test"
  - "/home/bases/integration"
```

## Configuration dans l'inventaire

### Inventaire INI

```ini
[production]
server1.example.com db_path="/home/gestcom/dbs"
server2.example.com db_paths=["/home/bases/prod","/home/bases/backup"]

[development]
dev1.example.com db_paths=["/home/bases/dev","/home/bases/test"]
```

### Inventaire YAML

```yaml
all:
  children:
    production:
      hosts:
        server1:
          db_path: "/home/gestcom/dbs"
        server2:
          db_paths:
            - "/home/bases/prod"
            - "/home/bases/backup"
    development:
      hosts:
        dev1:
          db_paths:
            - "/home/bases/dev"
            - "/home/bases/test"
```

## Ligne de commande

### Avec une seule base

```bash
ansible-playbook -i inventory.ini deploy_gestion_centos6.yml -K \
  -e db_path="/home/gestcom/dbs"
```

### Avec plusieurs bases

```bash
ansible-playbook -i inventory.ini deploy_gestion_centos6.yml -K \
  -e 'db_paths=["/home/gestcom/dbs","/home/bases/prod"]'
```

## Vérification

### Vérifier la variable normalisée

```bash
# Vérifier que db_paths est bien défini
ansible -i inventory.ini hostname -m debug -a "var=db_paths"

# Vérifier db_path (compatibilité - premier élément)
ansible -i inventory.ini hostname -m debug -a "var=db_path"
```

### Vérifier les dossiers créés

```bash
# Lister tous les dossiers logos créés
ansible -i inventory.ini hostname -m shell -a "find /home -type d -name logos" -b

# Vérifier les logs
ansible -i inventory.ini hostname -m shell -a "ls -la /home/log/" -b
```

### Vérifier le script de nettoyage

```bash
# Voir les chemins exclus
ansible -i inventory.ini hostname -m shell -a "grep -A 2 'Exclusion' /sos/nettoyage_fichier.sh" -b
```

## Comportement de normalisation

Le système applique automatiquement la logique suivante dans tous les playbooks et rôles :

```yaml
# Logique de normalisation
db_paths: "{{ [db_path] if db_path is defined and db_path is string else (db_paths if db_paths is defined else []) }}"
```

**Résultats :**

| Configuration d'entrée | Résultat normalisé |
|------------------------|-------------------|
| `db_path: "/home/dbs"` | `db_paths: ["/home/dbs"]` |
| `db_paths: ["/home/dbs"]` | `db_paths: ["/home/dbs"]` |
| `db_paths: ["/home/dbs", "/home/prod"]` | `db_paths: ["/home/dbs", "/home/prod"]` |
| Aucune variable | **Erreur** - Au moins une base requise |

**Base principale (compatibilité) :**

Pour les parties du code qui ont besoin d'une seule base, `db_path` est automatiquement défini comme le **premier élément** de `db_paths` :

```yaml
db_path: "{{ db_paths[0] }}"
```

## Playbooks affectés

Les playbooks suivants supportent maintenant `db_paths` :

1. ✅ `deploy_gestion_centos6.yml`
2. ✅ `prep_gestion_sans_installation_gestion.yml`

## Rôles affectés

Les rôles suivants supportent maintenant `db_paths` :

1. ✅ `install_gestion` - Création de dossiers pour chaque base
2. ✅ `logrotate_setup` - Logs pour chaque base
3. ✅ `system_maintenance` - Exclusion de toutes les bases du nettoyage

## Dépannage

### Problème : "La variable db_path doit être définie"

**Cause :** Ni `db_path` ni `db_paths` n'est défini.

**Solution :**
```yaml
# Définir au moins une base
db_path: "/home/gestcom/dbs"
# OU
db_paths:
  - "/home/gestcom/dbs"
```

### Problème : Les dossiers ne sont pas créés pour toutes les bases

**Vérification :**
```bash
# Afficher db_paths
ansible -i inventory.ini hostname -m debug -a "var=db_paths"

# Vérifier le playbook avec -vvv
ansible-playbook -i inventory.ini deploy_gestion_centos6.yml -K -vvv --limit hostname
```

### Problème : Syntaxe d'inventaire INI

**Incorrect :**
```ini
db_paths="/home/dbs","/home/prod"  # ❌
```

**Correct :**
```ini
db_paths=["/home/dbs","/home/prod"]  # ✅
```

## Migration depuis l'ancienne configuration

### Étape 1 : Identifier les serveurs avec db_path

```bash
# Rechercher dans les fichiers de variables
grep -r "db_path:" inventaire/ host_vars/ group_vars/
```

### Étape 2 : Décider de la migration

**Option A : Ne rien faire**
- Si vous avez une seule base par serveur
- La compatibilité est maintenue automatiquement

**Option B : Migrer vers db_paths**
- Si vous voulez ajouter d'autres bases
- Si vous voulez une configuration plus claire

### Étape 3 : Migrer progressivement

```bash
# 1. Tester sur un serveur de dev
ansible-playbook -i inventory.ini deploy_gestion_centos6.yml -K \
  --limit dev-server -e 'db_paths=["/home/bases/dev"]'

# 2. Mettre à jour les fichiers de variables
# host_vars/dev-server.yml
db_paths:
  - "/home/bases/dev"

# 3. Déployer
ansible-playbook -i inventory.ini deploy_gestion_centos6.yml -K --limit dev-server

# 4. Vérifier
ansible -i inventory.ini dev-server -m shell -a "ls -la /home/bases/dev" -b
```

## Bonnes pratiques

1. **Cohérence** : Choisissez un style (`db_path` ou `db_paths`) et gardez-le pour tous vos serveurs similaires

2. **Documentation** : Commentez vos fichiers de variables
   ```yaml
   # Base de production principale
   db_paths:
     - "/home/gestcom/dbs"
     # Base de backup (réplication)
     - "/home/backup/databases"
   ```

3. **Organisation** : Utilisez `group_vars` pour les configurations communes
   ```yaml
   # group_vars/production.yml
   db_paths:
     - "/home/gestcom/dbs"

   # host_vars/special-server.yml
   db_paths:
     - "/home/gestcom/dbs"
     - "/home/bases/special"  # Base supplémentaire
   ```

4. **Tests** : Toujours tester sur un environnement de développement d'abord

5. **Vérification** : Après chaque déploiement, vérifiez que tous les dossiers attendus sont créés

## Références

- [Documentation du rôle system_maintenance](../roles/system_maintenance/DB_PATHS_EXCLUSION.md)
- [Documentation du rôle install_gestion](../roles/install_gestion/README.md)
- [Documentation du rôle logrotate_setup](../roles/logrotate_setup/ReadMe.md)
- [Guide de migration vers download_file](MIGRATION_DOWNLOAD_FILE.md)
