from datetime import date, timedelta
from pathlib import Path
import os
import re
import subprocess
import sys


def sanity_check():
    """Vérifie que les prérequis sont respectés pour l'utilisation de ce module."""
    # if not os.path.exists("/home/informix/bin/isql"):
    #     raise Exception("isql not found in /home/informix/bin/isql")
    # if not os.access("/home/informix/bin/isql", os.X_OK):
    #     raise Exception("isql is not executable")
    # if "INFORMIXDIR" not in os.environ:
    #     raise Exception("INFORMIXDIR not found in environment")
    # if "INFORMIXSERVER" not in os.environ:
    #     raise Exception("INFORMIXSERVER not found in environment")
    assert sys.version_info > (3, 7), "Nécessite Python 3.7 ou supérieur"
    if "DBPATH" not in os.environ:
        raise Exception("DBPATH not found in environment")
    if not Path(os.environ["DBPATH"]).exists() and not Path(os.environ["DBPATH"]).is_dir():
        raise Exception(f"DBPATH {os.environ['DBPATH']} does not exist")


def odbc(cmd, colonnes, debug=False):
    return extract_from_isql(cmd, colonnes, debug)

def isql_nb_rows(stderr):
    """Retourne le nombre de lignes extraites par la commande isql"""

    for action in ["retrieved", "unloaded", "inserted", "updated", "deleted"]:
        nb_rows = re.search("\d+ row\(s\) {}".format(action), stderr)
        if nb_rows:
            break

    if nb_rows:
        nb_rows = int(nb_rows.group(0).split()[0])
    else:
        nb_rows = 0
    return nb_rows


def send_cmd_to_isql(sql:str) -> dict:
    """Execute une requete sql et retourne la sortie shell de isql.
    
    Retourne un dictionnaire contenant le nombre de lignes extraites, la sortie standard et la sortie d'erreur.
    Le dictionnaire est de la forme {"nbrows": int, "stderr": str, "stdout": str}
    """
    # ajouter bash -c pour que subprocess.run lance correctemement cette commande composee d'un pipe
    cmd = ["bash", "-c", 'echo "{0}" | /home/informix/bin/isql gc'.format(sql)]
    rslt = subprocess.run(
        cmd, env=os.environ, capture_output=True, text=True, encoding="cp850"
    )
    stderr = rslt.stderr
    stdout = rslt.stdout
    returncode = rslt.returncode

    stderr = remove_empty_lines(stderr)
    stdout = remove_empty_lines(stdout)
    nbrows = isql_nb_rows(stderr)

    return {"nbrows": nbrows, "stderr": stderr, "stdout": stdout}

def remove_empty_lines(stream):
    """Remove empty lines from a stream"""
    return "\n".join([s for s in stream.split("\n") if s.strip()])

def guess_arrangement(lines, columns):
    """Guess the arrangement of the data in the lines.

    Parameters
    ----------
    lines : list
        List of lines to analyze.

    columns : list
        List of columns to extract from the lines.

    Returns
    -------
    str
        'rows' or 'columns' depending on the arrangement of the data in the lines.
    """
    # Determine if the data is arranged in rows or columns
    # Example of a return from the shell command
    # code   numcrt
    fr = lines[0] # first row
    col_indexes = [fr.find(col) for col in columns] + [-1]
    colonnes = [fr[col_indexes[i] : col_indexes[i + 1]].strip() for i in range(len(columns))]
    return colonnes == columns, col_indexes

def __convert_columns_to_dict(line, columns, col_indexes):
    """Convert a line to a dictionary.

    Parameters
    ----------
    line : str
        Line to convert.

    columns : list
        List of columns to extract from the line.

    col_indexes : list
        List of indexes of the columns in the line.

    Returns
    -------
    dict
        Dictionary containing the values of the columns.
    """
    # return {col: line[col_indexes[i] : col_indexes[i + 1]].strip() for i, col in enumerate(columns)}

    if len(columns) == 1:
        return {columns[0]: line[col_indexes[0]:].strip()}
    else:
        data = {col: line[col_indexes[i] : col_indexes[i + 1]].strip() for i, col in enumerate(columns[:-1])}
        data[columns[-1]] = line[col_indexes[-2]:].strip()
        return data

def __convert_rows_to_dict(lines, columns):
    """Convert a list of lines to a dictionary.

    Parameters
    ----------
    lines : list
        List of lines to convert.

    columns : list
        List of columns to extract from the lines.

    Returns
    -------
    dict
        Dictionary containing the values of the columns.
    """
    data = {}
    for colname, line in zip(columns,lines):
        l = line.strip()
        if not line.startswith(colname):
            raise ValueError(f"line does not start with '{colname}': {line}")
        data[colname] = l.replace(colname, "").strip() # la valeur est tout ce qui reste apres le nom de la colonne

    return data

def extract_from_isql(cmd, colonnes, debug=False):
    """Execute une commande shell et retourne les donnees extraites dans un dictionnaire.

    Parameters
    ----------
    cmd : str
        Commande SQL a executer.

    view_as_rows : bool
        En fonction du nombre de donnees a retourner, isql affiche les donnees sur 2 lignes
        (view_as_rows=False) ou sur 2 colonnes (view_as_rows=True).
        Il appartient a l'utilisateur de savoir comment les donnees seront affichees, en fonction
        cmd.

    columns : list
        Liste des colonnes récupérer dans la commande SQL. Cette liste doit correspondre à la commande
        SQL (ordre, nombre et nom des colonnes). Elle constituera les clés du dictionnaire retourné.
    """

    rslt = send_cmd_to_isql(cmd)

    stderr = rslt["stderr"]
    stdout = rslt["stdout"]
    nbrows = rslt["nbrows"]

    if nbrows == 0:
        return 0, rslt

    columns = colonnes # add rowid to capture empty results.
    # Extraire les valeurs des colonnes du retour de la commande shell
    lines = stdout.split("\n")
    is_columns, col_indexes = guess_arrangement(lines, columns)
    if debug:
        # limiter le nombre de résultats à traiter pour le debug. 5 résultats maximum
        nbrows = 5  if nbrows > 5 else nbrows
        if is_columns:
            lines = lines[:nbrows+1] # +1 car la premi�re ligne contient les noms des colonnes
        else:
            lines = lines[:nbrows*len(columns)]
        for l in lines:
            print(l)

    data = []
    if is_columns:
        # exemple de retour de la commande shell
        # Retour de 3 résultat sous forme de colonnes
        # code   numcrt
        # 1PVWLA 103945150
        # 1PVWLA 103945150
        # 1PVWLA 103945150

        lines = lines[1:]
        for line in lines:
            data.append(__convert_columns_to_dict(line, columns, col_indexes))
    else:
        # exemple de retour de la commande shell
        # Retour de 3 résultats sous forme de lignes
        # code   1PVWLA
        # numcrt 103945150
        # code   1PVWLA
        # numcrt 103945150
        # code   1PVWLA
        # numcrt 103945150
        for i in range(0, len(lines), len(columns)):
            data.append(__convert_rows_to_dict(lines[i:i+len(columns)], columns))

    assert len(data) == nbrows, f"len(data) != nbrows: {len(data)} != {nbrows}"

    return len(data), data

# def cli_must_be_disabled_RGPD():
#     """Client qu'il faut desactiver à cause de la RGPD

#     Client sans achat depuis 3ans
#     """
#     # select count(*) from gc_cli where gc_cli.crtf = 1 and gc_cli.code in (select gc_fac.cli from gc_fac where gc_fac.dteinit >= '01/01/2021');

#     starting = date.today() - relativedelta(years=3)
#     cmd = f"select code from gc_cli where gc_cli.crtf = 1 and gc_cli.code in (select gc_fac.cli from gc_fac where gc_fac.dteinit >= '{starting:%d-%m-%Y}');"
#     print(cmd)
#     nbrows, clients = odbc(cmd, colonnes=["code"])
#     print(nbrows)
#     clients = [c["code"] for c in clients ]
#     print(clients)
#     create temp table t1 (
#         code char(6)
#     )
