"""Interface IfxPy pour l'accès à la base de données Informix.

Ce module fournit une interface simple et optimisée utilisant IfxPy,
le driver Python natif pour Informix.
"""

from pathlib import Path
import os
import sys
from typing import Tuple, List, Dict, Optional
import IfxPy


# Variable globale pour stocker la connexion
_connection = None


def set_environnement_variables():
    """Configure les variables d'environnement Informix."""
    os.environ['INFORMIXDIR'] = '/home/informix64'
    os.environ['LD_LIBRARY_PATH'] = (
        f"{os.environ['INFORMIXDIR']}/lib:"
        f"{os.environ['INFORMIXDIR']}/lib/esql:"
        f"{os.environ['INFORMIXDIR']}/lib/cli"
    )


def get_connection():
    """Récupère ou crée une connexion à la base de données Informix.

    La connexion est mise en cache et réutilisée pour toutes les requêtes.

    Returns:
        Connection object from IfxPy

    Raises:
        Exception: Si les variables d'environnement requises sont absentes
        ValueError: Si le DSN n'existe pas dans .odbc.ini
        Exception: Si la connexion échoue
    """
    global _connection

    if _connection is not None:
        return _connection

    set_environnement_variables()

    # Vérifier les variables d'environnement requises
    required_vars = ["INFORMIXSERVER", "DBPATH", "DBPATH_DSN"]
    missing_vars = [var for var in required_vars if var not in os.environ]

    if missing_vars:
        raise Exception(f"Variables d'environnement manquantes: {', '.join(missing_vars)}")

    dsn = os.environ.get("DBPATH_DSN")

    # Vérifier que le DSN est défini dans $HOME/.odbc.ini
    odbc_ini = Path.home() / ".odbc.ini"
    if not odbc_ini.exists():
        msg = f"Fichier {odbc_ini} n'existe pas"
        raise FileNotFoundError(msg)

    with open(odbc_ini, "r") as f:
        for line in f:
            if f"[{dsn}]" in line.strip():
                break
        else:
            raise ValueError(f"Le DSN [{dsn}] n'existe pas dans .odbc.ini")

    conn_str = f"DSN={dsn}"
    try:
        _connection = IfxPy.connect(conn_str, "", "")
        return _connection
    except Exception as e:
        raise Exception(f"Échec de connexion à Informix: {e}")


def close_connection():
    """Ferme la connexion à la base de données si elle existe."""
    global _connection

    if _connection is not None:
        try:
            IfxPy.close(_connection)
        except:
            pass
        finally:
            _connection = None


def sanity_check():
    """Vérifie que les prérequis sont respectés pour l'utilisation de ce module.

    Raises:
        AssertionError: Si Python < 3.7
        Exception: Si DBPATH n'existe pas
        ValueError: Si DBPATH_DSN n'est pas configuré ou n'existe pas
    """
    assert sys.version_info > (3, 7), "Nécessite Python 3.7 ou supérieur"

    if "DBPATH" not in os.environ:
        raise Exception("DBPATH not found in environment")

    dbpath = Path(os.environ["DBPATH"])
    if not dbpath.exists() or not dbpath.is_dir():
        raise Exception(f"DBPATH {os.environ['DBPATH']} does not exist")

    dsn = os.environ.get("DBPATH_DSN")
    if dsn is None:
        raise ValueError("DBPATH_DSN not found in environment")

    # Vérifier que le DSN est défini dans $HOME/.odbc.ini
    odbc_ini = Path.home() / ".odbc.ini"
    if not odbc_ini.exists():
        raise FileNotFoundError(f"Fichier {odbc_ini} n'existe pas")

    with open(odbc_ini, "r") as f:
        for line in f:
            if f"[{dsn}]" in line.strip():
                return
        raise ValueError(f"Le DSN [{dsn}] n'existe pas dans .odbc.ini")


def execute_query(sql: str, params: Optional[tuple] = None) -> Tuple[int, List[Dict[str, str]]]:
    """Exécute une requête SQL et retourne les résultats.

    Args:
        sql: Requête SQL à exécuter
        params: Paramètres optionnels pour requête préparée (non implémenté)

    Returns:
        Tuple (nbrows, data) où:
        - nbrows: nombre de lignes retournées
        - data: liste de dictionnaires avec les résultats

    Raises:
        Exception: Si la requête échoue
    """
    conn = get_connection()

    try:
        # Exécuter la requête
        stmt = IfxPy.exec_immediate(conn, sql)

        if stmt is False:
            error = IfxPy.stmt_error()
            raise Exception(f"Erreur SQL: {error}")

        # Récupérer tous les résultats
        data = []
        row = IfxPy.fetch_assoc(stmt)

        while row:
            # Convertir les valeurs en strings et enlever les espaces
            clean_row = {
                key.lower(): str(value).strip() if value is not None else ""
                for key, value in row.items()
            }
            data.append(clean_row)
            row = IfxPy.fetch_assoc(stmt)

        return len(data), data

    except Exception as e:
        raise Exception(f"Erreur lors de l'exécution de la requête: {e}")


def fetch_one(sql: str) -> Optional[Dict[str, str]]:
    """Exécute une requête et retourne un unique résultat.

    Args:
        sql: Requête SQL à exécuter

    Returns:
        Dictionnaire avec les résultats ou None si aucun résultat

    Raises:
        ValueError: Si plusieurs résultats sont trouvés
        Exception: Si la requête échoue
    """
    nbrows, data = execute_query(sql)

    if nbrows == 0:
        return None

    if nbrows > 1:
        raise ValueError(f"Plusieurs résultats trouvés ({nbrows}), un seul attendu")

    return data[0]


def fetch_all(sql: str) -> List[Dict[str, str]]:
    """Exécute une requête et retourne tous les résultats.

    Args:
        sql: Requête SQL à exécuter

    Returns:
        Liste de dictionnaires avec les résultats (peut être vide)

    Raises:
        Exception: Si la requête échoue
    """
    nbrows, data = execute_query(sql)
    return data


def fetch_value(sql: str, column: Optional[str] = None) -> Optional[str]:
    """Exécute une requête et retourne la première valeur.

    Args:
        sql: Requête SQL à exécuter
        column: Nom de la colonne (si None, prend la première colonne)

    Returns:
        La valeur ou None si aucun résultat

    Raises:
        ValueError: Si plusieurs résultats sont trouvés
        KeyError: Si la colonne n'existe pas
        Exception: Si la requête échoue
    """
    row = fetch_one(sql)

    if row is None:
        return None

    if column is None:
        # Prendre la première colonne
        return next(iter(row.values()))

    return row[column.lower()]


def execute_update(sql: str) -> int:
    """Exécute une requête UPDATE/INSERT/DELETE et retourne le nombre de lignes affectées.

    Args:
        sql: Requête SQL à exécuter

    Returns:
        Nombre de lignes affectées

    Raises:
        Exception: Si la requête échoue
    """
    conn = get_connection()

    try:
        stmt = IfxPy.exec_immediate(conn, sql)

        if stmt is False:
            error = IfxPy.stmt_error()
            raise Exception(f"Erreur SQL: {error}")

        # Récupérer le nombre de lignes affectées
        num_rows = IfxPy.num_rows(stmt)
        return num_rows if num_rows != -1 else 0

    except Exception as e:
        raise Exception(f"Erreur lors de l'exécution de la requête: {e}")


def unload_to_file(sql: str, output_file: str, delimiter: str = '|') -> int:
    """Exécute une requête UNLOAD TO et retourne le nombre de lignes exportées.

    Args:
        sql: Requête UNLOAD TO à exécuter
        output_file: Chemin du fichier de sortie
        delimiter: Délimiteur (défaut: '|')

    Returns:
        Nombre de lignes exportées

    Raises:
        Exception: Si la requête échoue
    """
    # Construire la requête UNLOAD
    unload_sql = f"UNLOAD TO {output_file} DELIMITER '{delimiter}' {sql}"
    return execute_update(unload_sql)
