"""
Tests unitaires pour converter.py

Teste les convertisseurs Excel et SQLite pour les invoices Kiabi.
"""

import pytest
from pathlib import Path

from gescokit.partners.kiabi.invoice_parser import KiabiInvoiceParser
from gescokit.partners.kiabi.converter import KiabiInvoiceConverter

# Import optionnels pour vérifier leur disponibilité
try:
    import pandas as pd
    PANDAS_AVAILABLE = True
except ImportError:
    PANDAS_AVAILABLE = False

try:
    from sqlalchemy import create_engine, inspect
    SQLALCHEMY_AVAILABLE = True
except ImportError:
    SQLALCHEMY_AVAILABLE = False


class TestKiabiInvoiceConverter:
    """Tests pour la classe KiabiInvoiceConverter."""

    def test_converter_initialization_default(self):
        """Test l'initialisation du convertisseur avec parser par défaut."""
        converter = KiabiInvoiceConverter()
        assert converter.parser is not None
        assert isinstance(converter.parser, KiabiInvoiceParser)

    def test_converter_initialization_custom(self):
        """Test l'initialisation du convertisseur avec parser personnalisé."""
        parser = KiabiInvoiceParser(encoding="cp850", delimiter=",")
        converter = KiabiInvoiceConverter(parser)
        assert converter.parser is parser
        assert converter.parser.encoding == "cp850"

    def test_lines_to_dict_list(self, sample_csv_file):
        """Test la conversion de lignes en liste de dictionnaires."""
        converter = KiabiInvoiceConverter()
        lines = converter.parser.parse_file(sample_csv_file)
        dict_list = converter._lines_to_dict_list(lines)

        # Vérifier le nombre d'éléments
        assert len(dict_list) == len(lines)

        # Vérifier que chaque élément est un dictionnaire
        for item in dict_list:
            assert isinstance(item, dict)

        # Vérifier les clés du premier élément
        first_item = dict_list[0]
        expected_keys = [
            "delivery_shop_id", "invoice_id", "order_id", "order_ref",
            "invoice_date", "invoice_date_year", "invoice_date_month",
            "invoice_date_day", "customer_lastname", "customer_firstname",
            "product_name", "product_ref", "voucher_id_1", "voucher_amount_1"
        ]
        for key in expected_keys:
            assert key in first_item

        # Vérifier les valeurs
        assert first_item["order_ref"] == "ORD001"
        assert first_item["customer_lastname"] == "DUPONT"
        assert first_item["invoice_date_year"] == 2025

    @pytest.mark.skipif(not PANDAS_AVAILABLE, reason="pandas not installed")
    def test_to_excel(self, sample_csv_file, tmp_path):
        """Test la conversion vers Excel."""
        converter = KiabiInvoiceConverter()
        output_file = tmp_path / "test_output.xlsx"

        result = converter.to_excel(sample_csv_file, output_file)

        # Vérifier que le fichier a été créé
        assert result.exists()
        assert result == output_file
        assert result.suffix == ".xlsx"

        # Vérifier le contenu avec pandas
        df = pd.read_excel(result, engine="openpyxl")
        assert len(df) == 4  # 4 lignes de données
        assert "order_ref" in df.columns
        assert "customer_lastname" in df.columns

    @pytest.mark.skipif(not PANDAS_AVAILABLE, reason="pandas not installed")
    def test_to_excel_auto_filename(self, sample_csv_file):
        """Test la conversion vers Excel avec nom de fichier automatique."""
        converter = KiabiInvoiceConverter()

        result = converter.to_excel(sample_csv_file)

        # Vérifier que le fichier a été créé avec le bon nom
        assert result.exists()
        assert result.stem == sample_csv_file.stem
        assert result.suffix == ".xlsx"

        # Nettoyer
        result.unlink()

    @pytest.mark.skipif(not PANDAS_AVAILABLE, reason="pandas not installed")
    def test_to_excel_custom_sheet_name(self, sample_csv_file, tmp_path):
        """Test la conversion vers Excel avec nom de feuille personnalisé."""
        converter = KiabiInvoiceConverter()
        output_file = tmp_path / "test_custom_sheet.xlsx"

        result = converter.to_excel(sample_csv_file, output_file, sheet_name="Factures")

        # Vérifier le nom de la feuille
        df = pd.read_excel(result, sheet_name="Factures", engine="openpyxl")
        assert len(df) == 4

    @pytest.mark.skipif(PANDAS_AVAILABLE, reason="Test requires pandas to be missing")
    def test_to_excel_without_pandas(self, sample_csv_file, tmp_path):
        """Test que la conversion Excel échoue sans pandas."""
        converter = KiabiInvoiceConverter()
        output_file = tmp_path / "test_output.xlsx"

        with pytest.raises(ImportError, match="pandas est requis"):
            converter.to_excel(sample_csv_file, output_file)

    @pytest.mark.skipif(not SQLALCHEMY_AVAILABLE, reason="SQLAlchemy not installed")
    def test_to_sqlite(self, sample_csv_file, tmp_path):
        """Test la conversion vers SQLite."""
        converter = KiabiInvoiceConverter()
        db_file = tmp_path / "test_output.db"

        result = converter.to_sqlite(sample_csv_file, db_file)

        # Vérifier que le fichier a été créé
        assert result.exists()
        assert result == db_file
        assert result.suffix == ".db"

        # Vérifier le contenu avec SQLAlchemy
        engine = create_engine(f"sqlite:///{db_file}")
        inspector = inspect(engine)

        # Vérifier que la table existe
        assert "invoices" in inspector.get_table_names()

        # Vérifier les colonnes
        columns = [col["name"] for col in inspector.get_columns("invoices")]
        assert "id" in columns
        assert "order_ref" in columns
        assert "customer_lastname" in columns
        assert "voucher_id_1" in columns

        # Vérifier les index
        indexes = inspector.get_indexes("invoices")
        index_columns = [idx["column_names"] for idx in indexes]
        assert ["order_ref"] in index_columns
        assert ["delivery_shop_id"] in index_columns

        # Vérifier le nombre de lignes
        with engine.connect() as conn:
            result = conn.execute("SELECT COUNT(*) FROM invoices")
            count = result.scalar()
            assert count == 4

    @pytest.mark.skipif(not SQLALCHEMY_AVAILABLE, reason="SQLAlchemy not installed")
    def test_to_sqlite_auto_filename(self, sample_csv_file):
        """Test la conversion vers SQLite avec nom de fichier automatique."""
        converter = KiabiInvoiceConverter()

        result = converter.to_sqlite(sample_csv_file)

        # Vérifier que le fichier a été créé avec le bon nom
        assert result.exists()
        assert result.stem == sample_csv_file.stem
        assert result.suffix == ".db"

        # Nettoyer
        result.unlink()

    @pytest.mark.skipif(not SQLALCHEMY_AVAILABLE, reason="SQLAlchemy not installed")
    def test_to_sqlite_overwrite(self, sample_csv_file, tmp_path):
        """Test l'écrasement d'une base SQLite existante."""
        converter = KiabiInvoiceConverter()
        db_file = tmp_path / "test_overwrite.db"

        # Créer la base une première fois
        converter.to_sqlite(sample_csv_file, db_file)

        # Vérifier le nombre de lignes
        engine = create_engine(f"sqlite:///{db_file}")
        with engine.connect() as conn:
            result = conn.execute("SELECT COUNT(*) FROM invoices")
            count1 = result.scalar()
            assert count1 == 4

        # Créer la base une deuxième fois avec overwrite=False (ajout)
        converter.to_sqlite(sample_csv_file, db_file, overwrite=False)

        with engine.connect() as conn:
            result = conn.execute("SELECT COUNT(*) FROM invoices")
            count2 = result.scalar()
            assert count2 == 8  # 4 + 4

        # Créer la base une troisième fois avec overwrite=True (écrasement)
        converter.to_sqlite(sample_csv_file, db_file, overwrite=True)

        with engine.connect() as conn:
            result = conn.execute("SELECT COUNT(*) FROM invoices")
            count3 = result.scalar()
            assert count3 == 4  # Réinitialisé à 4

    @pytest.mark.skipif(SQLALCHEMY_AVAILABLE, reason="Test requires SQLAlchemy to be missing")
    def test_to_sqlite_without_sqlalchemy(self, sample_csv_file, tmp_path):
        """Test que la conversion SQLite échoue sans SQLAlchemy."""
        converter = KiabiInvoiceConverter()
        db_file = tmp_path / "test_output.db"

        with pytest.raises(ImportError, match="SQLAlchemy est requis"):
            converter.to_sqlite(sample_csv_file, db_file)

    @pytest.mark.skipif(not SQLALCHEMY_AVAILABLE, reason="SQLAlchemy not installed")
    def test_to_sqlite_bulk(self, tmp_path, sample_csv_content):
        """Test la conversion en masse vers SQLite."""
        # Créer plusieurs fichiers CSV
        csv_file1 = tmp_path / "invoice1.csv"
        csv_file2 = tmp_path / "invoice2.csv"
        csv_file1.write_text(sample_csv_content, encoding="utf-8")
        csv_file2.write_text(sample_csv_content, encoding="utf-8")

        converter = KiabiInvoiceConverter()
        db_file = tmp_path / "test_bulk.db"

        result = converter.to_sqlite_bulk([csv_file1, csv_file2], db_file)

        # Vérifier que le fichier a été créé
        assert result.exists()
        assert result == db_file

        # Vérifier le nombre total de lignes (4 + 4 = 8)
        engine = create_engine(f"sqlite:///{db_file}")
        with engine.connect() as conn:
            result = conn.execute("SELECT COUNT(*) FROM invoices")
            count = result.scalar()
            assert count == 8

    @pytest.mark.skipif(not SQLALCHEMY_AVAILABLE, reason="SQLAlchemy not installed")
    def test_to_sqlite_bulk_overwrite(self, tmp_path, sample_csv_content):
        """Test l'écrasement en mode bulk."""
        csv_file1 = tmp_path / "invoice1.csv"
        csv_file1.write_text(sample_csv_content, encoding="utf-8")

        converter = KiabiInvoiceConverter()
        db_file = tmp_path / "test_bulk_overwrite.db"

        # Première conversion
        converter.to_sqlite_bulk([csv_file1], db_file)

        engine = create_engine(f"sqlite:///{db_file}")
        with engine.connect() as conn:
            result = conn.execute("SELECT COUNT(*) FROM invoices")
            count1 = result.scalar()
            assert count1 == 4

        # Deuxième conversion avec overwrite=True
        converter.to_sqlite_bulk([csv_file1], db_file, overwrite=True)

        with engine.connect() as conn:
            result = conn.execute("SELECT COUNT(*) FROM invoices")
            count2 = result.scalar()
            assert count2 == 4  # Toujours 4 (écrasé)

    @pytest.mark.skipif(not SQLALCHEMY_AVAILABLE, reason="SQLAlchemy not installed")
    def test_line_to_model(self, sample_csv_file):
        """Test la conversion d'une ligne en modèle SQLAlchemy."""
        converter = KiabiInvoiceConverter()
        lines = converter.parser.parse_file(sample_csv_file)

        # Convertir la première ligne
        model = converter._line_to_model(lines[0])

        # Vérifier les attributs
        assert model.order_ref == "ORD001"
        assert model.customer_lastname == "DUPONT"
        assert model.customer_firstname == "Jean"
        assert model.invoice_date.year == 2025
        assert model.product_name == "T-Shirt Bleu"
        assert model.currency == "EUR"
