"""
Tests unitaires pour cli.py

Teste l'interface CLI du plugin Kiabi.
"""

import pytest
from pathlib import Path
from typer.testing import CliRunner

from gescokit.partners.kiabi.cli import app

# Runner pour tester les commandes CLI
runner = CliRunner()


class TestCLI:
    """Tests pour l'interface CLI."""

    def test_cli_help(self):
        """Test l'affichage de l'aide."""
        result = runner.invoke(app, ["--help"])
        assert result.exit_code == 0
        assert "Commandes pour gérer les fichiers d'invoices Kiabi" in result.stdout

    def test_cli_parse_help(self):
        """Test l'aide de la commande parse."""
        result = runner.invoke(app, ["parse", "--help"])
        assert result.exit_code == 0
        assert "Parse un fichier CSV" in result.stdout

    def test_cli_parse_basic(self, sample_csv_file):
        """Test la commande parse sans options."""
        result = runner.invoke(app, ["parse", str(sample_csv_file)])
        assert result.exit_code == 0
        assert "Fichier parsé" in result.stdout
        assert "Nombre de lignes: 4" in result.stdout
        assert "Nombre de factures: 3" in result.stdout

    def test_cli_parse_with_show_lines(self, sample_csv_file):
        """Test la commande parse avec --show-lines."""
        result = runner.invoke(app, ["parse", str(sample_csv_file), "--show-lines"])
        assert result.exit_code == 0
        assert "ORD001" in result.stdout
        assert "DUPONT" in result.stdout
        assert "T-Shirt Bleu" in result.stdout

    def test_cli_parse_with_limit(self, sample_csv_file):
        """Test la commande parse avec --limit."""
        result = runner.invoke(app, ["parse", str(sample_csv_file), "--show-lines", "--limit", "2"])
        assert result.exit_code == 0
        # Devrait montrer au maximum 2 lignes

    def test_cli_parse_non_existent_file(self, tmp_path):
        """Test la commande parse avec un fichier inexistant."""
        non_existent = tmp_path / "non_existent.csv"
        result = runner.invoke(app, ["parse", str(non_existent)])
        assert result.exit_code == 1
        assert "Erreur" in result.stdout

    def test_cli_to_excel_help(self):
        """Test l'aide de la commande to-excel."""
        result = runner.invoke(app, ["to-excel", "--help"])
        assert result.exit_code == 0
        assert "Convertit un fichier CSV" in result.stdout
        assert "Excel" in result.stdout

    @pytest.mark.skipif(True, reason="Requires pandas")
    def test_cli_to_excel_basic(self, sample_csv_file, tmp_path):
        """Test la commande to-excel."""
        output_file = tmp_path / "output.xlsx"
        result = runner.invoke(app, ["to-excel", str(sample_csv_file), "-o", str(output_file)])

        if "pandas est requis" in result.stdout:
            pytest.skip("pandas not installed")

        assert result.exit_code == 0
        assert "Fichier Excel créé" in result.stdout
        assert output_file.exists()

    @pytest.mark.skipif(True, reason="Requires pandas")
    def test_cli_to_excel_auto_output(self, sample_csv_file):
        """Test la commande to-excel avec nom de fichier automatique."""
        result = runner.invoke(app, ["to-excel", str(sample_csv_file)])

        if "pandas est requis" in result.stdout:
            pytest.skip("pandas not installed")

        assert result.exit_code == 0
        assert "Fichier Excel créé" in result.stdout

        # Nettoyer
        output_file = sample_csv_file.with_suffix(".xlsx")
        if output_file.exists():
            output_file.unlink()

    def test_cli_to_sqlite_help(self):
        """Test l'aide de la commande to-sqlite."""
        result = runner.invoke(app, ["to-sqlite", "--help"])
        assert result.exit_code == 0
        assert "Convertit un fichier CSV" in result.stdout
        assert "SQLite" in result.stdout

    @pytest.mark.skipif(True, reason="Requires SQLAlchemy")
    def test_cli_to_sqlite_basic(self, sample_csv_file, tmp_path):
        """Test la commande to-sqlite."""
        db_file = tmp_path / "output.db"
        result = runner.invoke(app, ["to-sqlite", str(sample_csv_file), "-o", str(db_file)])

        if "SQLAlchemy est requis" in result.stdout:
            pytest.skip("SQLAlchemy not installed")

        assert result.exit_code == 0
        assert "Base SQLite créée" in result.stdout
        assert db_file.exists()

    @pytest.mark.skipif(True, reason="Requires SQLAlchemy")
    def test_cli_to_sqlite_overwrite(self, sample_csv_file, tmp_path):
        """Test la commande to-sqlite avec --overwrite."""
        db_file = tmp_path / "output.db"

        # Première création
        result1 = runner.invoke(app, ["to-sqlite", str(sample_csv_file), "-o", str(db_file)])
        if "SQLAlchemy est requis" in result1.stdout:
            pytest.skip("SQLAlchemy not installed")

        # Deuxième création avec overwrite
        result2 = runner.invoke(app, ["to-sqlite", str(sample_csv_file), "-o", str(db_file), "--overwrite"])
        assert result2.exit_code == 0

    def test_cli_to_sqlite_bulk_help(self):
        """Test l'aide de la commande to-sqlite-bulk."""
        result = runner.invoke(app, ["to-sqlite-bulk", "--help"])
        assert result.exit_code == 0
        assert "plusieurs fichiers CSV" in result.stdout

    @pytest.mark.skipif(True, reason="Requires SQLAlchemy")
    def test_cli_to_sqlite_bulk_basic(self, tmp_path, sample_csv_content):
        """Test la commande to-sqlite-bulk."""
        # Créer deux fichiers CSV
        csv1 = tmp_path / "invoice1.csv"
        csv2 = tmp_path / "invoice2.csv"
        csv1.write_text(sample_csv_content, encoding="utf-8")
        csv2.write_text(sample_csv_content, encoding="utf-8")

        db_file = tmp_path / "bulk.db"

        result = runner.invoke(app, [
            "to-sqlite-bulk",
            str(csv1), str(csv2),
            "-o", str(db_file)
        ])

        if "SQLAlchemy est requis" in result.stdout:
            pytest.skip("SQLAlchemy not installed")

        assert result.exit_code == 0
        assert "Base SQLite créée" in result.stdout
        assert "Nombre de fichiers traités: 2" in result.stdout
        assert db_file.exists()

    def test_cli_info_help(self):
        """Test l'aide de la commande info."""
        result = runner.invoke(app, ["info", "--help"])
        assert result.exit_code == 0
        assert "informations détaillées" in result.stdout

    def test_cli_info_basic(self, sample_csv_file):
        """Test la commande info."""
        result = runner.invoke(app, ["info", str(sample_csv_file)])
        assert result.exit_code == 0
        assert "Informations sur le fichier" in result.stdout
        assert "Nombre de lignes: 4" in result.stdout
        assert "Nombre de factures: 3" in result.stdout
        assert "Résumé des factures" in result.stdout

    def test_cli_doc(self):
        """Test la commande doc."""
        result = runner.invoke(app, ["doc"])
        assert result.exit_code == 0
        assert "Format CSV des invoices Kiabi" in result.stdout
        assert "Nomenclature des fichiers" in result.stdout
        assert "invoice-XXX-YYYYMMDD-AAAAAAAAA.csv" in result.stdout
        assert "Shop_id" in result.stdout
        assert "Voucher_id_1" in result.stdout
        assert "ITFWEBKIA_VENTES_decodekia" in result.stdout


class TestCLIEdgeCases:
    """Tests pour les cas limites du CLI."""

    def test_cli_parse_empty_file(self, tmp_path):
        """Test le parsing d'un fichier vide."""
        empty_file = tmp_path / "empty.csv"
        empty_file.write_text("", encoding="utf-8")

        result = runner.invoke(app, ["parse", str(empty_file)])
        # Le comportement peut varier, mais ne devrait pas crasher
        assert result.exit_code in [0, 1]

    def test_cli_parse_invalid_csv(self, tmp_path):
        """Test le parsing d'un CSV invalide."""
        invalid_file = tmp_path / "invalid.csv"
        invalid_file.write_text("invalid;csv;content\n", encoding="utf-8")

        result = runner.invoke(app, ["parse", str(invalid_file)])
        # Le comportement peut varier, mais ne devrait pas crasher
        assert result.exit_code in [0, 1]

    def test_cli_to_sqlite_bulk_empty_list(self, tmp_path):
        """Test to-sqlite-bulk avec une liste vide."""
        db_file = tmp_path / "empty.db"

        result = runner.invoke(app, ["to-sqlite-bulk", "-o", str(db_file)])
        # Devrait échouer car aucun fichier fourni
        assert result.exit_code != 0

    def test_cli_to_sqlite_bulk_non_existent_files(self, tmp_path):
        """Test to-sqlite-bulk avec des fichiers inexistants."""
        db_file = tmp_path / "output.db"
        non_existent = tmp_path / "non_existent.csv"

        result = runner.invoke(app, [
            "to-sqlite-bulk",
            str(non_existent),
            "-o", str(db_file)
        ])
        assert result.exit_code == 1
        assert "Fichier introuvable" in result.stdout
