"""
Parser pour les fichiers CSV d'invoices Kiabi (OneTouch).

Ce module permet de parser les fichiers CSV générés par le système OneTouch
et utilisés dans le système GCV (via itfwebkia.ec).

Format de fichier: invoice-XXX-YYYYMMDD-AAAAAAAAA.csv
"""

import csv
import structlog
from dataclasses import dataclass, field
from datetime import datetime
from pathlib import Path
from typing import Optional, List, Dict, Any

logger = structlog.get_logger()


@dataclass
class InvoiceKiabiLine:
    """
    Représente une ligne de facture Kiabi (équivalent de ttinvoicekia en C).

    Correspond aux champs extraits dans ITFWEBKIA_VENTES_decodekia() du fichier
    itfwebkia.ec lignes 430-546.
    """
    # Identifiants
    delivery_shop_id: str           # Shop_id
    invoice_id: int                 # Invoice_id (nocom)
    order_id: str                   # Order_id
    order_ref: str                  # Order_ref (refcom)

    # Date et heure
    invoice_date: datetime          # Invoice_date (dtcom + hrcom)

    # Client
    customer_lastname: str          # Customer_lastname (nomcli)
    customer_firstname: str         # Customer_firstname (prncli)
    customer_gender: str            # Customer_Gender (gender_of_customer) - souvent vide
    loyalty_card_id: str            # Loyalty_card_id (numcrt)
    loyalty_discount_flag: int      # Loyalty_discount_flag (ilyaremise)

    # Produit
    product_id: str                 # Product_id
    product_name: str               # Product_name (desig)
    product_ref: str                # Product_ref (code)
    product_ean13: str              # Product_ean13 (ean)
    product_unit_price_ht: float    # Product_unit_price_ht (pvht)
    product_unit_price_ttc: float   # Product_unit_price_ttc (pvttc)
    product_tax_rate: float         # Product_tax_rate (txtva)
    product_qty: float              # Product_qty (qt)
    product_total_price_ht: float   # Product_total_price_ht
    product_total_price_ttc: float  # Product_total_price_ttc (pvtttc)
    product_stock_shop_id: str      # Product_stock_shop_id (typtail)

    # Totaux facture
    invoice_product_ht: float       # Invoice_product_ht (totmhtav)
    invoice_tax: float              # Invoice_tax (tottva)
    invoice_product_ttc: float      # Invoice_product_ttc (totttc)

    # Paiement
    payment_method: str             # Payment_method (modepai)
    payment_amount: float           # Payment_amount (montant)
    payment_transaction_id: str     # Payment_transaction_id (numaut)
    payment_bank_remittance_date: str  # Payment_bank_remittance_date (dtrembanq)

    # Réduction
    invoice_reduction_ttc: float    # Invoice_reduction_ttc (totremttc)

    # Devise
    currency: str                   # Currency (devise)
    currency_rate_eur: float        # CurrencyRateEur (tauxdev)

    # Avoirs/Vouchers (ajoutés le 21/07/2025 - voir lignes 526-545 de itfwebkia.ec)
    voucher_id_1: str = ""          # Voucher_id_1 (id_avoir_utilise_1)
    voucher_amount_1: float = 0.0   # Voucher_amount_1 (montant_avoir_1)
    voucher_id_2: str = ""          # Voucher_id_2 (id_avoir_utilise_2)
    voucher_amount_2: float = 0.0   # Voucher_amount_2 (montant_avoir_2)
    voucher_id_3: str = ""          # Voucher_id_3 (id_avoir_utilise_3)
    voucher_amount_3: float = 0.0   # Voucher_amount_3 (montant_avoir_3)

    # Réduction spécifique web (calculée - voir ligne 542-545 de itfwebkia.ec)
    reduction_specif_web: float = 0.0  # dbreduc - (montant_avoir_1 + montant_avoir_2 + montant_avoir_3)

    # Données brutes pour référence
    raw_data: Dict[str, str] = field(default_factory=dict)


class KiabiInvoiceParser:
    """
    Parser pour les fichiers CSV d'invoices Kiabi.

    Équivalent Python de la fonction ITFWEBKIA_VENTES_decodekia() du fichier C
    (itfwebkia.ec lignes 430-546).
    """

    # Colonnes attendues dans le CSV (ordre exact du fichier - voir ligne 428 de itfwebkia.ec)
    # + colonnes voucher ajoutées lignes 233-238
    EXPECTED_COLUMNS = [
        "Shop_id",
        "Invoice_id",
        "Invoice_date",
        "Order_id",
        "Order_ref",
        "Customer_lastname",
        "Customer_firstname",
        "Loyalty_card_id",
        "Loyalty_discount_flag",
        "Product_id",
        "Product_name",
        "Product_ref",
        "Product_ean13",
        "Product_unit_price_ht",
        "Product_unit_price_ttc",
        "Product_tax_rate",
        "Product_qty",
        "Product_total_price_ht",
        "Product_total_price_ttc",
        "Product_stock_shop_id",
        "Invoice_product_ht",
        "Invoice_tax",
        "Invoice_product_ttc",
        "Payment_method",
        "Payment_amount",
        "Payment_transaction_id",
        "Payment_bank_remittance_date",
        "Invoice_reduction_ttc",
        "Currency",
        "CurrencyRateEur",
        # Colonnes voucher (ajoutées 21/07/2025)
        "Voucher_id_1",
        "Voucher_amount_1",
        "Voucher_id_2",
        "Voucher_amount_2",
        "Voucher_id_3",
        "Voucher_amount_3"
    ]

    def __init__(self, encoding: str = "utf-8", delimiter: str = ";"):
        """
        Initialise le parser.

        Args:
            encoding: Encodage du fichier CSV (utf-8 par défaut, cp850 en C)
            delimiter: Séparateur de colonnes (point-virgule par défaut)
        """
        self.encoding = encoding
        self.delimiter = delimiter

    @staticmethod
    def _parse_date(date_str: str) -> datetime:
        """
        Parse une date au format ISO: 2020-04-24 00:59:58.

        Correspond au parsing dans ITFWEBKIA_VENTES_decodekia ligne 442-448.
        """
        try:
            return datetime.strptime(date_str.strip(), "%Y-%m-%d %H:%M:%S")
        except ValueError as e:
            logger.warning(f"Date invalide: {date_str}, erreur: {e}")
            return datetime.now()

    @staticmethod
    def _to_float(value: str) -> float:
        """
        Convertit une chaîne en float, gère les virgules et points.

        Correspond à strtodot() en C.
        """
        if not value or value.strip() == "":
            return 0.0
        try:
            # Remplace virgule par point
            return float(value.strip().replace(",", "."))
        except ValueError:
            logger.warning(f"Impossible de convertir en float: {value}")
            return 0.0

    @staticmethod
    def _to_int(value: str) -> int:
        """Convertit une chaîne en int."""
        if not value or value.strip() == "":
            return 0
        try:
            return int(value.strip())
        except ValueError:
            logger.warning(f"Impossible de convertir en int: {value}")
            return 0

    def parse_line(self, row: Dict[str, str]) -> InvoiceKiabiLine:
        """
        Parse une ligne du CSV et retourne un objet InvoiceKiabiLine.

        Équivalent de ITFWEBKIA_VENTES_decodekia() (lignes 430-546).

        Args:
            row: Dictionnaire représentant une ligne du CSV

        Returns:
            InvoiceKiabiLine: Objet représentant la ligne parsée
        """
        # Parse des champs de base
        invoice_reduction_ttc = self._to_float(row.get("Invoice_reduction_ttc", "0"))

        # Parse des vouchers (lignes 527-538 de itfwebkia.ec)
        voucher_amount_1 = self._to_float(row.get("Voucher_amount_1", "0"))
        voucher_amount_2 = self._to_float(row.get("Voucher_amount_2", "0"))
        voucher_amount_3 = self._to_float(row.get("Voucher_amount_3", "0"))

        # Calcul de la réduction spécifique web (lignes 542-545 de itfwebkia.ec)
        # dbreduc inclut les montants des avoirs, il faut les déduire pour obtenir
        # la réduction "spécifique web" (BAFID chez Kiabi)
        reduction_specif_web = invoice_reduction_ttc - (
            voucher_amount_1 + voucher_amount_2 + voucher_amount_3
        )

        return InvoiceKiabiLine(
            delivery_shop_id=row.get("Shop_id", "").strip(),
            invoice_id=self._to_int(row.get("Invoice_id", "0")),
            order_id=row.get("Order_id", "").strip(),
            order_ref=row.get("Order_ref", "").strip()[:9],  # Limite à 9 caractères

            invoice_date=self._parse_date(row.get("Invoice_date", "")),

            customer_lastname=row.get("Customer_lastname", "").strip()[:25],
            customer_firstname=row.get("Customer_firstname", "").strip()[:25],
            customer_gender=row.get("Customer_Gender", "").strip(),  # Souvent vide
            loyalty_card_id=row.get("Loyalty_card_id", "").strip()[:12],
            loyalty_discount_flag=self._to_int(row.get("Loyalty_discount_flag", "0")),

            product_id=row.get("Product_id", "").strip(),
            product_name=row.get("Product_name", "").strip()[:60],
            product_ref=row.get("Product_ref", "").strip()[:12],
            product_ean13=row.get("Product_ean13", "").strip()[:13],
            product_unit_price_ht=self._to_float(row.get("Product_unit_price_ht", "0")),
            product_unit_price_ttc=self._to_float(row.get("Product_unit_price_ttc", "0")),
            product_tax_rate=self._to_float(row.get("Product_tax_rate", "0")),
            product_qty=self._to_float(row.get("Product_qty", "0")),
            product_total_price_ht=self._to_float(row.get("Product_total_price_ht", "0")),
            product_total_price_ttc=self._to_float(row.get("Product_total_price_ttc", "0")),
            product_stock_shop_id=row.get("Product_stock_shop_id", "").strip(),

            invoice_product_ht=self._to_float(row.get("Invoice_product_ht", "0")),
            invoice_tax=self._to_float(row.get("Invoice_tax", "0")),
            invoice_product_ttc=self._to_float(row.get("Invoice_product_ttc", "0")),

            payment_method=row.get("Payment_method", "").strip(),
            payment_amount=self._to_float(row.get("Payment_amount", "0")),
            payment_transaction_id=row.get("Payment_transaction_id", "").strip(),
            payment_bank_remittance_date=row.get("Payment_bank_remittance_date", "").strip(),

            invoice_reduction_ttc=invoice_reduction_ttc,

            currency=row.get("Currency", "EUR").strip() or "EUR",
            currency_rate_eur=self._to_float(row.get("CurrencyRateEur", "1")) or 1.0,

            # Vouchers
            voucher_id_1=row.get("Voucher_id_1", "").strip()[:15],
            voucher_amount_1=voucher_amount_1,
            voucher_id_2=row.get("Voucher_id_2", "").strip()[:15],
            voucher_amount_2=voucher_amount_2,
            voucher_id_3=row.get("Voucher_id_3", "").strip()[:15],
            voucher_amount_3=voucher_amount_3,

            reduction_specif_web=reduction_specif_web,

            raw_data=row
        )

    def parse_file(self, file_path: Path) -> List[InvoiceKiabiLine]:
        """
        Parse un fichier CSV complet et retourne la liste des lignes.

        Correspond à la lecture du fichier dans ITFWEBKIA_VENTES_getreftick().

        Args:
            file_path: Chemin vers le fichier CSV

        Returns:
            List[InvoiceKiabiLine]: Liste des lignes parsées
        """
        logger.info(f"Parsing du fichier: {file_path}")

        if not file_path.exists():
            logger.error(f"Fichier introuvable: {file_path}")
            raise FileNotFoundError(f"Fichier introuvable: {file_path}")

        lines = []

        with open(file_path, "r", encoding=self.encoding) as csvfile:
            reader = csv.DictReader(csvfile, delimiter=self.delimiter)

            # Vérifier que les colonnes attendues sont présentes
            if reader.fieldnames:
                missing_cols = set(self.EXPECTED_COLUMNS) - set(reader.fieldnames)
                if missing_cols:
                    logger.warning(f"Colonnes manquantes: {missing_cols}")

            for row_num, row in enumerate(reader, start=2):  # start=2 car ligne 1 = header
                try:
                    line = self.parse_line(row)
                    lines.append(line)
                except Exception as e:
                    logger.error(f"Erreur ligne {row_num}: {e}", exc_info=True)

        logger.info(f"Fichier parsé: {len(lines)} lignes trouvées")
        return lines

    def group_by_invoice(self, lines: List[InvoiceKiabiLine]) -> Dict[str, List[InvoiceKiabiLine]]:
        """
        Regroupe les lignes par numéro de facture (order_ref).

        Correspond à la logique dans ITFWEBKIA_VENTES_trtmagkia_universel()
        qui traite chaque ticket individuellement.

        Args:
            lines: Liste des lignes parsées

        Returns:
            Dict[str, List[InvoiceKiabiLine]]: Dictionnaire {order_ref: [lignes]}
        """
        invoices = {}
        for line in lines:
            if line.order_ref not in invoices:
                invoices[line.order_ref] = []
            invoices[line.order_ref].append(line)

        logger.info(f"Nombre de factures distinctes: {len(invoices)}")
        return invoices
