"""
Module de conversion des fichiers CSV d'invoices Kiabi vers différents formats.

Ce module fournit des fonctions pour convertir les fichiers CSV parsés vers:
- Excel (.xlsx)
- SQLite (.db) via SQLAlchemy
"""

import structlog
from pathlib import Path
from typing import List, Optional
from datetime import datetime

try:
    import pandas as pd
    PANDAS_AVAILABLE = True
except ImportError:
    PANDAS_AVAILABLE = False
    pd = None

try:
    from sqlalchemy import (
        create_engine, Column, Integer, String, Float, DateTime, Text
    )
    from sqlalchemy.orm import declarative_base, sessionmaker
    SQLALCHEMY_AVAILABLE = True
    Base = declarative_base()
except ImportError:
    SQLALCHEMY_AVAILABLE = False
    Base = None

from .invoice_parser import InvoiceKiabiLine, KiabiInvoiceParser

logger = structlog.get_logger()


if SQLALCHEMY_AVAILABLE:
    class InvoiceKiabiModel(Base):
        """
        Modèle SQLAlchemy pour les invoices Kiabi.

        Correspond à la structure ttinvoicekia en C.
        """
        __tablename__ = "invoices"

        # Clé primaire auto-incrémentée
        id = Column(Integer, primary_key=True, autoincrement=True)

        # Identifiants
        delivery_shop_id = Column(String(10), index=True)
        invoice_id = Column(Integer)
        order_id = Column(String(50))
        order_ref = Column(String(10), index=True)

        # Date et heure
        invoice_date = Column(DateTime, index=True)
        invoice_date_year = Column(Integer)
        invoice_date_month = Column(Integer)
        invoice_date_day = Column(Integer)
        invoice_time_hour = Column(Integer)
        invoice_time_minute = Column(Integer)
        invoice_time_second = Column(Integer)

        # Client
        customer_lastname = Column(String(25))
        customer_firstname = Column(String(25))
        customer_gender = Column(String(5))
        loyalty_card_id = Column(String(13), index=True)
        loyalty_discount_flag = Column(Integer)

        # Produit
        product_id = Column(String(50))
        product_name = Column(String(60))
        product_ref = Column(String(12), index=True)
        product_ean13 = Column(String(13), index=True)
        product_unit_price_ht = Column(Float)
        product_unit_price_ttc = Column(Float)
        product_tax_rate = Column(Float)
        product_qty = Column(Float)
        product_total_price_ht = Column(Float)
        product_total_price_ttc = Column(Float)
        product_stock_shop_id = Column(String(10))

        # Totaux facture
        invoice_product_ht = Column(Float)
        invoice_tax = Column(Float)
        invoice_product_ttc = Column(Float)

        # Paiement
        payment_method = Column(String(25))
        payment_amount = Column(Float)
        payment_transaction_id = Column(String(25))
        payment_bank_remittance_date = Column(String(10))

        # Réduction
        invoice_reduction_ttc = Column(Float)

        # Devise
        currency = Column(String(3))
        currency_rate_eur = Column(Float)

        # Vouchers
        voucher_id_1 = Column(String(15))
        voucher_amount_1 = Column(Float)
        voucher_id_2 = Column(String(15))
        voucher_amount_2 = Column(Float)
        voucher_id_3 = Column(String(15))
        voucher_amount_3 = Column(Float)

        # Calculé
        reduction_specif_web = Column(Float)

        # Métadonnées
        created_at = Column(DateTime, default=datetime.now)

        def __repr__(self):
            return f"<InvoiceKiabi(order_ref='{self.order_ref}', invoice_date='{self.invoice_date}')>"


class KiabiInvoiceConverter:
    """Convertisseur pour les invoices Kiabi vers différents formats."""

    def __init__(self, parser: Optional[KiabiInvoiceParser] = None):
        """
        Initialise le convertisseur.

        Parameters
        ----------
        parser : KiabiInvoiceParser, optional
            Parser à utiliser (crée un parser par défaut si None)
        """
        self.parser = parser or KiabiInvoiceParser()

    def _lines_to_dict_list(self, lines: List[InvoiceKiabiLine]) -> List[dict]:
        """
        Convertit une liste de InvoiceKiabiLine en liste de dictionnaires.

        Parameters
        ----------
        lines : List[InvoiceKiabiLine]
            Liste des lignes parsées

        Returns
        -------
        List[dict]
            Liste de dictionnaires représentant les données
        """
        data = []
        for line in lines:
            data.append({
                # Identifiants
                "delivery_shop_id": line.delivery_shop_id,
                "invoice_id": line.invoice_id,
                "order_id": line.order_id,
                "order_ref": line.order_ref,

                # Date et heure
                "invoice_date": line.invoice_date.strftime("%Y-%m-%d %H:%M:%S"),
                "invoice_date_year": line.invoice_date.year,
                "invoice_date_month": line.invoice_date.month,
                "invoice_date_day": line.invoice_date.day,
                "invoice_time_hour": line.invoice_date.hour,
                "invoice_time_minute": line.invoice_date.minute,
                "invoice_time_second": line.invoice_date.second,

                # Client
                "customer_lastname": line.customer_lastname,
                "customer_firstname": line.customer_firstname,
                "customer_gender": line.customer_gender,
                "loyalty_card_id": line.loyalty_card_id,
                "loyalty_discount_flag": line.loyalty_discount_flag,

                # Produit
                "product_id": line.product_id,
                "product_name": line.product_name,
                "product_ref": line.product_ref,
                "product_ean13": line.product_ean13,
                "product_unit_price_ht": line.product_unit_price_ht,
                "product_unit_price_ttc": line.product_unit_price_ttc,
                "product_tax_rate": line.product_tax_rate,
                "product_qty": line.product_qty,
                "product_total_price_ht": line.product_total_price_ht,
                "product_total_price_ttc": line.product_total_price_ttc,
                "product_stock_shop_id": line.product_stock_shop_id,

                # Totaux facture
                "invoice_product_ht": line.invoice_product_ht,
                "invoice_tax": line.invoice_tax,
                "invoice_product_ttc": line.invoice_product_ttc,

                # Paiement
                "payment_method": line.payment_method,
                "payment_amount": line.payment_amount,
                "payment_transaction_id": line.payment_transaction_id,
                "payment_bank_remittance_date": line.payment_bank_remittance_date,

                # Réduction
                "invoice_reduction_ttc": line.invoice_reduction_ttc,

                # Devise
                "currency": line.currency,
                "currency_rate_eur": line.currency_rate_eur,

                # Vouchers
                "voucher_id_1": line.voucher_id_1,
                "voucher_amount_1": line.voucher_amount_1,
                "voucher_id_2": line.voucher_id_2,
                "voucher_amount_2": line.voucher_amount_2,
                "voucher_id_3": line.voucher_id_3,
                "voucher_amount_3": line.voucher_amount_3,

                # Calculé
                "reduction_specif_web": line.reduction_specif_web,
            })
        return data

    def _line_to_model(self, line: InvoiceKiabiLine) -> "InvoiceKiabiModel":
        """
        Convertit une InvoiceKiabiLine en modèle SQLAlchemy.

        Parameters
        ----------
        line : InvoiceKiabiLine
            Ligne parsée

        Returns
        -------
        InvoiceKiabiModel
            Instance du modèle
        """
        if not SQLALCHEMY_AVAILABLE:
            raise ImportError("SQLAlchemy n'est pas disponible")

        return InvoiceKiabiModel(
            delivery_shop_id=line.delivery_shop_id,
            invoice_id=line.invoice_id,
            order_id=line.order_id,
            order_ref=line.order_ref,

            invoice_date=line.invoice_date,
            invoice_date_year=line.invoice_date.year,
            invoice_date_month=line.invoice_date.month,
            invoice_date_day=line.invoice_date.day,
            invoice_time_hour=line.invoice_date.hour,
            invoice_time_minute=line.invoice_date.minute,
            invoice_time_second=line.invoice_date.second,

            customer_lastname=line.customer_lastname,
            customer_firstname=line.customer_firstname,
            customer_gender=line.customer_gender,
            loyalty_card_id=line.loyalty_card_id,
            loyalty_discount_flag=line.loyalty_discount_flag,

            product_id=line.product_id,
            product_name=line.product_name,
            product_ref=line.product_ref,
            product_ean13=line.product_ean13,
            product_unit_price_ht=line.product_unit_price_ht,
            product_unit_price_ttc=line.product_unit_price_ttc,
            product_tax_rate=line.product_tax_rate,
            product_qty=line.product_qty,
            product_total_price_ht=line.product_total_price_ht,
            product_total_price_ttc=line.product_total_price_ttc,
            product_stock_shop_id=line.product_stock_shop_id,

            invoice_product_ht=line.invoice_product_ht,
            invoice_tax=line.invoice_tax,
            invoice_product_ttc=line.invoice_product_ttc,

            payment_method=line.payment_method,
            payment_amount=line.payment_amount,
            payment_transaction_id=line.payment_transaction_id,
            payment_bank_remittance_date=line.payment_bank_remittance_date,

            invoice_reduction_ttc=line.invoice_reduction_ttc,

            currency=line.currency,
            currency_rate_eur=line.currency_rate_eur,

            voucher_id_1=line.voucher_id_1,
            voucher_amount_1=line.voucher_amount_1,
            voucher_id_2=line.voucher_id_2,
            voucher_amount_2=line.voucher_amount_2,
            voucher_id_3=line.voucher_id_3,
            voucher_amount_3=line.voucher_amount_3,

            reduction_specif_web=line.reduction_specif_web,
        )

    def to_excel(
        self,
        csv_file: Path,
        output_file: Optional[Path] = None,
        sheet_name: str = "Invoices"
    ) -> Path:
        """
        Convertit un fichier CSV d'invoices vers Excel.

        Parameters
        ----------
        csv_file : Path
            Chemin vers le fichier CSV source
        output_file : Path, optional
            Chemin vers le fichier Excel de sortie (auto si None)
        sheet_name : str, default="Invoices"
            Nom de la feuille Excel

        Returns
        -------
        Path
            Chemin vers le fichier Excel créé

        Raises
        ------
        ImportError
            Si pandas n'est pas installé
        """
        if not PANDAS_AVAILABLE:
            raise ImportError(
                "pandas est requis pour l'export Excel. "
                "Installez-le avec: pip install pandas openpyxl"
            )

        logger.info(f"Conversion vers Excel: {csv_file}")

        # Parser le fichier CSV
        lines = self.parser.parse_file(csv_file)

        # Convertir en dictionnaires
        data = self._lines_to_dict_list(lines)

        # Créer un DataFrame
        df = pd.DataFrame(data)

        # Générer le nom du fichier de sortie si non fourni
        if output_file is None:
            output_file = csv_file.with_suffix(".xlsx")

        # Exporter vers Excel
        df.to_excel(output_file, sheet_name=sheet_name, index=False, engine="openpyxl")

        logger.info(f"Fichier Excel créé: {output_file} ({len(lines)} lignes)")
        return output_file

    def to_sqlite(
        self,
        csv_file: Path,
        db_file: Optional[Path] = None,
        overwrite: bool = False
    ) -> Path:
        """
        Convertit un fichier CSV d'invoices vers SQLite via SQLAlchemy.

        Parameters
        ----------
        csv_file : Path
            Chemin vers le fichier CSV source
        db_file : Path, optional
            Chemin vers la base SQLite de sortie (auto si None)
        overwrite : bool, default=False
            Si True, écrase la table existante

        Returns
        -------
        Path
            Chemin vers le fichier SQLite créé

        Raises
        ------
        ImportError
            Si SQLAlchemy n'est pas installé
        """
        if not SQLALCHEMY_AVAILABLE:
            raise ImportError(
                "SQLAlchemy est requis pour l'export SQLite. "
                "Installez-le avec: pip install sqlalchemy"
            )

        logger.info(f"Conversion vers SQLite: {csv_file}")

        # Parser le fichier CSV
        lines = self.parser.parse_file(csv_file)

        # Générer le nom du fichier de sortie si non fourni
        if db_file is None:
            db_file = csv_file.with_suffix(".db")

        # Créer l'engine SQLAlchemy
        engine = create_engine(f"sqlite:///{db_file}")

        # Créer la table
        if overwrite:
            Base.metadata.drop_all(engine)
        Base.metadata.create_all(engine)

        # Créer une session
        Session = sessionmaker(bind=engine)
        session = Session()

        try:
            # Convertir et insérer les lignes
            for line in lines:
                model = self._line_to_model(line)
                session.add(model)

            # Commit
            session.commit()
            logger.info(f"Base SQLite créée: {db_file} ({len(lines)} lignes)")

        except Exception as e:
            session.rollback()
            logger.error(f"Erreur lors de l'insertion: {e}", exc_info=True)
            raise

        finally:
            session.close()

        return db_file

    def to_sqlite_bulk(
        self,
        csv_files: List[Path],
        db_file: Path,
        overwrite: bool = False
    ) -> Path:
        """
        Convertit plusieurs fichiers CSV vers une seule base SQLite.

        Parameters
        ----------
        csv_files : List[Path]
            Liste des chemins vers les fichiers CSV sources
        db_file : Path
            Chemin vers la base SQLite de sortie
        overwrite : bool, default=False
            Si True, écrase la table existante

        Returns
        -------
        Path
            Chemin vers le fichier SQLite créé

        Raises
        ------
        ImportError
            Si SQLAlchemy n'est pas installé
        """
        if not SQLALCHEMY_AVAILABLE:
            raise ImportError(
                "SQLAlchemy est requis pour l'export SQLite. "
                "Installez-le avec: pip install sqlalchemy"
            )

        logger.info(f"Conversion en masse vers SQLite: {len(csv_files)} fichiers")

        # Créer l'engine SQLAlchemy
        engine = create_engine(f"sqlite:///{db_file}")

        # Créer la table
        if overwrite:
            Base.metadata.drop_all(engine)
        Base.metadata.create_all(engine)

        # Créer une session
        Session = sessionmaker(bind=engine)
        session = Session()

        total_lines = 0

        try:
            for csv_file in csv_files:
                logger.info(f"Traitement de: {csv_file}")

                # Parser le fichier CSV
                lines = self.parser.parse_file(csv_file)
                total_lines += len(lines)

                # Convertir et insérer les lignes
                for line in lines:
                    model = self._line_to_model(line)
                    session.add(model)

            # Commit
            session.commit()
            logger.info(f"Base SQLite créée: {db_file} ({total_lines} lignes au total)")

        except Exception as e:
            session.rollback()
            logger.error(f"Erreur lors de l'insertion: {e}", exc_info=True)
            raise

        finally:
            session.close()

        return db_file
