# """Les CSVs à traiter sont:
#         - *archive_facture.csv (mélange l'entête et le pied de facture)
#         - *monttva.csv (contient les totaux TVA du pied de facture)
# """ 
import copy

from .tools import split_emc2_csv_line
from .to_html import lire_archive, table, table_wrapper
    
CSV_ENCODING="cp850"

# Entete du fichier *archive_facture.csv
FACTURE_COLNAMES = [
    "Identifiant",
    "Numéro facture",
    # ------- Logiciel
    "Version du logiciel",
    "Nom Logiciel",

    "Nombre d'édition",
    "Raison sociale",
    "Adresse",
    "Code postal",
    "Ville",
    "Pays",
    "Siret",
    "NAF",
    "TVA Intra.",

    # ------- Vendeur -------
    "Vendeur",
    "Nom",
    "Opérateur",
    "Nom opérateur",
    "Caisse",

    "Regroupement de consommation",
    "Nombre de client",
    "Date",
    "Date Livraison",
    "Opération",
    "Document",
    "Nombre de lignes",
    "Restitution signature",
    "Report",
    "Signature",
    "Signature précédente"
]

MONTTVA_COLNAMES = [
    "id",
    "Numéro facture",
    "Base TVA HT",
    "FAC-TAX-TOT-MHT",
    "FAC-TAX-TVA-NID",
    "FAC-TAX-TVA-TXX",
    "FAC-TAX-TVA-LIB",
    "FAC-ECT-ID",
    "FAC-ECT-TAUX",
    "FAC-ECT-MTN",
    "FAC-AUT-ID",
    "FAC-AUT-TAUX",
    "FAC-AUT-MTN",
    "FAC-TOT-MHT",
    "FAC-ESC-MTT",
    "FAC-RIS-MTT",
    "FAC-FRA-MTT",
    "FAC-POR-MTT",
    "FAC-TOT-TVA",
    "FAC-TOT-TAX",
    "FAC-TOT-TTC-MTN",
    "FAC-REG-ACPT",
    "FAC-MTN-NET",
    "FAC-REG-DAT",
    "FAC-REG-MEN-IND",
    "FAC-REG-TXX",
    "FAC-HASH",
    "FAC-SIG",
    "FAC-SIG-RES",
    "FAC-VER",
    "FAC-VER-NOM",
    "FAC-REG-ID",
    "FAC-REG-MODE",
    "FAC-REG-MODE-LIB",
    "FAC-REG-MODE-REF",
    "FAC-REG-MODE-MTN",
    "ENC-TIK-DON",
    "ENC-TIK-DON-MTN",
    "ENC-TIK-DON-ORG",
    "FAC-SPE",
    "FAC-SPE"
    ]

def get_no_factures(data):
    """Liste les numéros de facture dans le fichier *archive_facture.csv
    Parameters
    ----------
    data: list
        Liste des lignes du fichier *archive_facture.csv
    Returns
    -------
    set
        Ensemble des numéros de facture (sans doublons)
    """
    idx_no_fac = FACTURE_COLNAMES.index("Numéro facture")
    return set([row[idx_no_fac] for row in data])


def inserer_tva_entete(factures, tva_csvname):

    with open(tva_csvname, encoding=CSV_ENCODING) as fp:
        tvas = [split_emc2_csv_line(line) for line in fp if line.strip()]

    idx_fac_tot_mht = MONTTVA_COLNAMES.index("FAC-TOT-MHT")
    idx_fac_tot_tva = MONTTVA_COLNAMES.index("FAC-TOT-TVA")
    idx_tva_no_fac = MONTTVA_COLNAMES.index("Numéro facture")

    factures_copy = copy.deepcopy(factures)
    idx_no_facture = FACTURE_COLNAMES.index("Numéro facture")
    idx_res_sign = FACTURE_COLNAMES.index("Restitution signature")

    # Ajoute deux colonnes pour stocker le total TVA et le total HT
    FACTURE_COLNAMES.insert(idx_res_sign,"Montant Total HT")
    FACTURE_COLNAMES.insert(idx_res_sign+1,"Montant Total TVA")
    for idx, fac in enumerate(factures_copy):
        no_facture = fac[idx_no_facture]

        # The list to update
        facture = factures[idx]

        for tva in tvas:
            if tva[idx_tva_no_fac] == no_facture:
                facture.insert(idx_res_sign, tva[idx_fac_tot_mht])
                facture.insert(idx_res_sign+1, tva[idx_fac_tot_tva])
                break
        else:
            # La facture n'a pas été trouvée dans le fichier *monttva.csv
            facture.insert(idx_res_sign  , "")
            facture.insert(idx_res_sign+1, "")

    return FACTURE_COLNAMES, factures

  
def generer_entete_facture(idvent, dt):
    """
    
    Returns
    -------
        navigation: list
            Liste des éléments de navigation
        html: str
            Texte HTML contenant section, titre et tableau

    Notes
    -----
        La fonction C qui génère ce fichier est **BATCH_export_archivefac**
    """

    facture_csvname = "output/{:03}{}_archive_facture.csv".format(idvent, dt)
    tva_csvname     = "output/{:03}{}_archive_monttva.csv".format(idvent, dt)

    id_nav = "gfacture"
    titre = "Entête de facture"

    with open(facture_csvname, encoding=CSV_ENCODING) as fp:
        factures = [split_emc2_csv_line(line) for line in fp if line.strip()]
    
    no_fac = get_no_factures(factures)
    # navigation, html = lire_archive(csvname, FACTURE_COLNAMES, id_nav, titre)

    colnames, factures = inserer_tva_entete(factures, tva_csvname)

    navigations = [{"id": id_nav, "titre": titre}]

    html = table_wrapper(table(colnames, factures), id_nav, titre)

    return navigations, html


def generer_pied_facture(idvent, dt):
    """
    
    Returns
    -------
        navigation: list
            Liste des éléments de navigation
        html: str
            Texte HTML contenant section, titre et tableau

    Notes
    -----
        La fonction C qui génère ce fichier est **BATCH_export_archivefac**
    """

    tva_csvname     = "output/{:03}{}_archive_monttva.csv".format(idvent, dt)

    id_nav = "pfacture"
    titre = "Pied de facture"

    with open(tva_csvname, encoding=CSV_ENCODING) as fp:
        tvas = [split_emc2_csv_line(line) for line in fp if line.strip()]
    
    navigations = [{"id": id_nav, "titre": titre}]

    html = table_wrapper(table(MONTTVA_COLNAMES, tvas), id_nav, titre)

    return navigations, html


def archive_facture(idvent, dt):
    navigations, html = generer_entete_facture(idvent, dt)
    nav, txt = generer_pied_facture(idvent, dt)
    navigations.extend(nav)
    html += txt
    return navigations, html