from datetime import datetime
from pathlib import Path
from typing import Dict


def write_labels_maintexfile(
    etiquettes_dict: Dict,
    main_texfile: Path,
    design_directory: Path,
    design_shortname: str
) -> None:
    """
    Écrit le fichier LaTeX principal pour les étiquettes à partir d'un template.

    Parameters
    ----------
    etiquettes_dict : Dict
        Données globales nécessaires à la génération du document (marges, paysage, etc).
    main_texfile : Path
        Chemin du fichier .tex de sortie.
    design_directory : Path
        Répertoire contenant le fichier main.tex de base avec les placeholders.
    design_shortname : str
        Nom court du design (injecté dans le template).
    """
    options = [f"paper={etiquettes_dict['paper']}"]
    if etiquettes_dict.get("landscape", False):
        options.append("landscape")
    options_str = ",".join(options)

    if etiquettes_dict.get("paper_equal_format"):
        margin_definitions = r"\usepackage[margin=0cm,nomarginpar,ignoreall,noheadfoot,nomarginpar]{geometry}"
    else:
        margins = etiquettes_dict.get("margins", {"top": "0cm", "left": "0cm", "right": "0mm", "bottom": "0cm"})
        margin_definitions = r"\usepackage[{},nomarginpar,ignoreall,noheadfoot]{{geometry}}".format(
            ",".join([
                f"top={margins.get('top','0cm')}",
                f"left={margins.get('left','0cm')}",
                f"right={margins.get('right','0mm')}",
                f"bottom={margins.get('bottom','0cm')}"
            ])
        )

    preamble = f"""
%
% Ce fichier est généré automatiquement par le générateur d'étiquettes
% Veuillez modifier le template principal si nécessaire.
% Date de génération: {datetime.now().strftime("%Y-%m-%d %H:%M:%S")}
%
\\documentclass[10pt,headinclude=false,footinclude=false,parskip=never,usegeometry,{options_str}]{{scrartcl}}
\\usepackage{{polyglossia}}
\\setdefaultlanguage{{french}}
{margin_definitions}
"""

    replacements = {
        "@PREAMBLE_AUTOGENERATED@": preamble,
        "@DESIGN_DIRECTORY@": str(design_directory),
        "@DESIGNSHORTNAME@": design_shortname
    }

    template_path = Path(__file__).parent.absolute().resolve() / "main.tex"
    with open(template_path, "r", encoding="utf8") as inp, open(main_texfile, "w", encoding="utf8") as oup:
        for line in inp:
            for key, value in replacements.items():
                if key in line:
                    if key == "@PREAMBLE_AUTOGENERATED@":
                        line = value
                        break
                    else:
                        line = line.replace(key, value)
            oup.write(line)

def clean_latex(texfile):
    for ext in [".log", ".aux", ".fls", ".fdb_latexmk"]:
        rm_me = texfile.with_suffix(ext)
        if rm_me.is_file(): rm_me.unlink()