#!/usr/bin/python3
import sys
import time

from datetime import datetime
import json
import logging
from pathlib import Path
import pprint as pp
import shutil
import subprocess
import sys
import tarfile
import tempfile

from gescokit.etiquettes import label_validator

from .old_archi.etiquettes import prix as price_utils
from gescokit.etiquettes.fonts import write_font_preamble
from gescokit.etiquettes.tex import write_labels_maintexfile, clean_latex
from gescokit.etiquettes.config import read_design_configs
from gescokit.etiquettes.layout import auto_detection_config_papier, \
    calcul_positions_etiquettes, get_nb_pages_from_pdf, PaperSize
from gescokit.etiquettes.label_model import LabelModel
from .shared_progress import EtiqSharedProgress, EtiqGenerationStep, get_step_name

logging.basicConfig(
    stream=sys.stdout,
    format="%(asctime)s [etiquettes] - %(message)s",
    datefmt="%d-%b-%Y %H:%M:%S",
    level=logging.DEBUG,
)

logger = logging.getLogger(__name__)


def generate_labels_from_json(
    json_file: Path,
    config_file: Path,
    max_labels: int = 0,
    encoding: str = "utf8",
    debug: int = 0,
    no_latex: bool = False,
    build_pngs: bool = True,

    output: Path|None = None,
    design_shortname: str|None = None,
    paper_size: str|None = None,
    verbose: bool = False,
    format: str|None = None,
    tarname: str|None = None,
    attached_pid: int = 0
):
    """
    Génère des étiquettes à partir d'un fichier JSON et d'une configuration YAML.
    """

    use_latexmk = False
    use_arara = True

    # Initialisation du gestionnaire de progression
    progress_manager = None
    if attached_pid > 0:
        try:
            progress_manager = EtiqSharedProgress(attached_pid)
            # logger.info(f"Création de la mémoire partagée pour le PID {progress_manager.shm_name}")
            progress_manager.start_step(EtiqGenerationStep.INIT)
            logger.info(f"Gestionnaire de progression enrichie initialisé pour PID {attached_pid}")
        except Exception as e:
            logger.error(f"Erreur lors de l'initialisation de la progression: {e}")
            attached_pid = 0

    try:
        if not json_file.is_file():
            raise FileNotFoundError(f"Le fichier JSON n'existe pas {json_file}")

        # Lecture du fichier JSON décrivant toutes les étiquettes à imprimer.
        # Contient par exemple: la description de l'article, le prix, le prix promo, le codebarres...
        # Ce fichier est généré par GCV6 ou manuellement voir le dossier exemples/
        with open(str(json_file), encoding=encoding) as fp:
            etiq_to_print_as_json = json.load(fp)

        # Attention ! les paramètres rentrés par CLI sont prioritaire
        # le design est defini dans le fichier JSON mais le paramètre design_shortname a la précédence
        # si il est défini.
        _design_shortname = design_shortname or etiq_to_print_as_json.get("design", "default")
        _paper_size = (paper_size or etiq_to_print_as_json["papier"]).lower()
        _output = Path(output or etiq_to_print_as_json["output"])
        _format = format or etiq_to_print_as_json["format"]

        # user_config contient les configurations des étiquettes de la société
        # ce dictionnaire définit toutes les étiquettes (et propriétés) que EMC2 a autorisé
        # à la société courante.
        # design_config est le fichier de configuration YAML qui a été utilisé pour extraire la user_config.
        # user_config est donc un sous-ensemble de design_config.
        design_config, design_config_file, user_config = read_design_configs(config_file, _design_shortname )


        # Mise à jour progression: parsing JSON terminé
        if progress_manager:
            progress_manager.start_step(EtiqGenerationStep.PARSE_JSON)

        if debug > 0:
            with open(Path().home() / "etiquettes.json", "w", encoding="utf8") as fp:
                json.dump(etiq_to_print_as_json, fp, indent=4, ensure_ascii=False)

        if _format == _paper_size:
            margins = {"top":"0mm","left":"0mm","eti_bot":"0mm","eti_right":"0mm"}
        else:
            # TODO: Vérifier que "format" ai encore un intérêt
            margins = design_config[_format].get("margins",{}).get(_paper_size, None)

        if margins is None:
            logger.info(
                "Aucune marges n'a été définies pour le format {} sur le papier {}".format(
                    _format, _paper_size
                )
            )
            logger.info(
                'Le format a respecté est : margins: {"a4":{"top":"6mm","left":"6mm","eti_bot":"5mm","eti_right":"5mm"}}'
            )
            margins = None

        landscape = False
        if "landscape" in etiq_to_print_as_json:
            # Attention ! la valeur paramétrée dans le JSON est prioritaire
            # à celle du fichier de configuration
            landscape = etiq_to_print_as_json["landscape"]
        else:
            if _format:
                try:
                    landscape = design_config[_format]["papiers_paysage"][_paper_size]
                except Exception:
                    pass

        # if debug > 0:
        #     # working_directory = Path(tempfile.mkdtemp(dir=str(Path(__file__).parent.parent)))
        #     working_directory = Path(__file__).parent.parent / "tmp"
        #     if not working_directory.is_dir():
        #         working_directory.mkdir(parents=True, exist_ok=True)
        # else:
        #     working_directory = Path(tempfile.mkdtemp())
        working_directory = Path(tempfile.mkdtemp())

        main_texfile = working_directory / "etiquettes.tex"
        barcode_directory = working_directory / "barcodes"
        if not barcode_directory.is_dir():
            barcode_directory.mkdir(parents=True, exist_ok=True)
        if progress_manager: progress_manager.end_current_step()


        if progress_manager: progress_manager.start_step(EtiqGenerationStep.PARSE_JSON)
        # Les formats disponibles pour la société courante. Par exemple: a4, a7, gondole...
        # formats_dispo_societe = utils.get_formats_dispo(user_design_config)
        available_formats = user_config.get("formats", {}).keys()
        etiquettes_data = []
        nb_articles_in_json = len(etiq_to_print_as_json["articles"])
        for idx,article in enumerate(etiq_to_print_as_json["articles"]):
            # Mise à jour progression pendant la validation des articles
            progress = int((idx + 1) / nb_articles_in_json * 100)
            if article := label_validator.validate_article(
                                article, _design_shortname, _format, available_formats,
                                design_config, barcode_directory
                            ):

                # print(f"article {article['code']} barcode {article['barcode']}", end="\r", flush=True)
                # On ajoute l'étiquette à la liste des étiquettes à générer
                etiquettes_data.extend(article["nombex"] * [article])
                if progress_manager:
                    progress_manager.update_step_progress(progress,
                        f"article [{article['code']}] "
                    )
                    logger.debug(f"Article {article['code']} valide, progression: {progress}%")
            else:
                if progress_manager:
                    progress_manager.update_step_progress(progress,
                        f"Article {article['code']} invalide"
                    )
        if progress_manager: progress_manager.end_current_step()

        if max_labels > 0:
            etiquettes_data = etiquettes_data[: max_labels]
        if not etiquettes_data:
            logger.info("aucune étiquette à générer")
            if progress_manager:
                progress_manager.set_success("Aucune étiquette à générer")
            sys.exit(0)

        if progress_manager: progress_manager.end_current_step()


        if progress_manager:
            progress_manager.start_step(EtiqGenerationStep.GENERATE_BARCODE)
        articles = []
        total_data = len(etiquettes_data)
        for idx,etiquette in enumerate(etiquettes_data):
            progress = int((idx+1)/total_data*100)
            eti = LabelModel(**etiquette)
            eti.update_config(design_config)
            eti.generate_barcode()

            design_of_format = design_config.get( article["format"])
            # if design_of_format is None:
            #     logger.warning(
            #         "format {} non reconnu dans le fichier de configuration {}".format(
            #             format, etiquette_config_yamlfile
            #         )
            #     )
            #     continue
            article = etiquette.copy() | design_of_format
            article["border"] = False if article["format"]==_paper_size else eti.border
            if article.get("bgcolor", None) is None:
                article["bgcolor"] = (255, 255, 255)
            if article.get("fgcolor", None) is None:
                article["fgcolor"] = (0, 0, 0)

            # ======= Travail sur les prix
            article["devise"] = eti.devise
            prix = article.pop("prix")
            prix_ancien = article.pop("prix_ancien", {})

            _prix_ht_et_ttc = price_utils.PrixHtTtc(
                contenance=article["contenance"],
                contenance_unit=article["unite"],
                **prix
                )
            if prix_ancien:
                # Si l'ancien prix est défini, on le convertit aussi en PrixHtTtc
                # pour pouvoir comparer les prix et éventuellement supprimer l'ancien prix
                # si il n'est pas strictement supérieur au nouveau prix.
                _ancien_prix_ht_et_ttc = price_utils.PrixHtTtc(**prix_ancien)
                if _ancien_prix_ht_et_ttc.ttc <= _prix_ht_et_ttc.ttc:
                    article["prix_ancien"] = None # on supprime l'ancien prix pour éviter de l'afficher
                else:
                    article["prix_ancien"] = _ancien_prix_ht_et_ttc.to_dict()
            article["prix"] = _prix_ht_et_ttc.to_dict()

            articles.append(article)

            # print(f'article {article["code"]}', end="\r", flush=True)
            if progress_manager:
                progress_manager.update_step_by_count(idx + 1, total_data,
                    f"Contrôle de l'article {article['code']}"
                )
        if progress_manager: progress_manager.end_current_step()
        # print()
        # Détermine si le papier est exactement de la même taille que les étiquettes.
        # Si c’est le cas, le template LaTeX doit être adapté pour éviter les marges.
        config_papier = auto_detection_config_papier(articles, _paper_size)

        nbetiquettes = len(articles)

        eti_positions, nbetiquettes_par_page = calcul_positions_etiquettes(
            paperSize=_paper_size,
            fmt=_format,
            margins=margins,
            landscape=landscape,
            nbetiquettes=nbetiquettes,
        )


        # print("nb etiquettes: {}".format(nbetiquettes))
        # print("nb etiquettes par page: {}".format(nbetiquettes_par_page))
        if progress_manager:
            progress_manager.start_step(EtiqGenerationStep.GENERATE_LATEX)

        for idx, eti in enumerate(articles):
            pos = next(eti_positions)
            # eti["border"] = not config_papier["paper_equal_format"] # Mettre un cadre de découpage
            eti["position"] = pos

        if progress_manager:
            progress_manager.update_progress(
                EtiqGenerationStep.GENERATE_LATEX,
                30,
                f"Calcul positions {idx + 1}/{nbetiquettes}"
            )
        # print()

        # A faire avant de réorganiser les articles par groupes
        write_font_preamble(articles, "/home/local/fonts", working_directory)

        etiquettes_dict = {
            "paper": PaperSize(_paper_size).as_scrartcl_opt(),
            "debug": debug,
            "margins": margins,
            "landscape":landscape,
            "font_dossier": "/home/local/fonts/",
            "devise":"€",
            "format": _format, # On part du principe que le format est le même pour toutes les étiquettes
            **config_papier
        }

        etiquettes_by_page = [
            articles[i : i + nbetiquettes_par_page]
            for i in range(0, nbetiquettes, nbetiquettes_par_page)
        ]

        articles_by_page_json = working_directory / "articles_by_page.json"
        with open( articles_by_page_json, "w", encoding="utf8") as fp:
            json.dump(etiquettes_dict | {"pages":etiquettes_by_page}, fp, ensure_ascii=False, indent=2)

        if not _output:
            _output = main_texfile.with_suffix(".pdf")

        write_labels_maintexfile(etiquettes_dict, main_texfile, design_config_file.parent.parent, _design_shortname)

        # Mise à jour progression: génération LaTeX terminée
        if progress_manager:
            progress_manager.update_progress(
                EtiqGenerationStep.GENERATE_LATEX,
                50,
                get_step_name(EtiqGenerationStep.GENERATE_LATEX)
            )

        if not no_latex:
            if use_latexmk:
                latexmk_cmd = "latexmk -lualatex -outdir={}".format(working_directory)
                if not verbose:
                    latexmk_cmd += " --quiet"
                texcommand = latexmk_cmd + f" {main_texfile}"

            if use_arara:
                options = []
                options.append("-v" if verbose else "-s")
                options.append("--log" if debug > 0 else "")
                texcommand = f"arara {' '.join(options)} --working-directory={working_directory} {main_texfile}"

            # Mise à jour progression: compilation PDF terminée
            if progress_manager:
                progress_manager.update_progress(
                    EtiqGenerationStep.COMPILE_PDF,
                    80,
                    get_step_name(EtiqGenerationStep.COMPILE_PDF)
                )

            _process = subprocess.run(texcommand.split(), check=True, capture_output=True)

            # nbPDFpages = get_nbPDFpages(str(main_texfile.with_suffix(".log")))
            # logger.info("LaTeX a créé un pdf de {} pages".format(nbPDFpages))

            if progress_manager: progress_manager.end_current_step()

            pdffile = main_texfile.with_suffix(".pdf")
            if pdffile.is_file() and _output != pdffile:
                shutil.copy(str(pdffile), str(_output))

            logger.info(f"LaTeX a créé un pdf de {get_nb_pages_from_pdf(str(_output))} pages")


            # Finalisation et nettoyage du gestionnaire de progression
            if progress_manager:
                nb_pages = get_nb_pages_from_pdf(str(_output))
                if nb_pages == 1:
                    message = f"1 page avec {nbetiquettes} etiquettes"
                else:
                    message = f"{nb_pages} pages avec {nbetiquettes} etiquettes et {nbetiquettes_par_page} etiquettes par pages"

                progress_manager.set_success(message)

            if debug < 0:
                clean_latex(main_texfile)


        # def archiver_fichiers_sources(working_directory, tarname):
        #     archive_path = Path.home() / f"{tarname}.tar"
        #     entryname = archive_path.stem

        #     # modifie le chemin absolue vers le dossier temporaire en './'
        #     for fichier in working_directory.rglob("*.tex"):
        #         # Lire en mémoire le fichier
        #         filedata = Path(fichier).read_text()
        #         # Remplacer toutes les occurences
        #         filedata = filedata.replace(str(working_directory), ".")
        #         # Écrire le fichier
        #         with open(str(fichier), "w") as f:
        #             f.write(filedata)



        #     with tarfile.open(str(archive_path), "w") as tar:
        #         tar.add(
        #             str(working_directory), arcname="{0}/latex/".format(entryname)
        #         )  # renomme et simplifie le chemin dans le tar
        #         if json_file:
        #             tar.add(str(json_file), arcname="{0}/{1}".format(entryname, str(json_file)))

        # if tarname:
        #     archiver_fichiers_sources(working_directory, tarname)

        # # if debug < 0 and not no_latex:
        # #     main_texfile.unlink()

        # # if debug < 0 and working_directory.is_dir() and not no_latex:
        # #     shutil.rmtree(working_directory, ignore_errors=True)
        # # else:
        # #     logger.info("dossier intermedaire {}".format(str(working_directory)))

        # if build_pngs and not no_latex:
        #     # TODO utiliser les fichiers temporaires
        #     subprocess.call(
        #         "convert -density 300 etiquettes.pdf -quality 90 etiquette_%d.png".split()
        #     )
        #     # convert $(basename $@)-0.png -flatten -trim +repage $(basename $@).png
        #     if debug < 0:
        #         main_texfile.with_suffix(".pdf").unlink()

        #     if _output.is_file() and not no_latex:
        #         logger.info("PDF : {}".format(_output))
        #         print("PDF : {}".format(_output))

        #     # Finalisation et nettoyage du gestionnaire de progression
        #     if progress_manager:
        #         progress_manager.update_progress(
        #             EtiqGenerationStep.FINALIZE,
        #             100,
        #             # get_step_name(EtiqGenerationStep.FINALIZE)
        #             f"PDF: {_output} {get_nb_pages_from_pdf(str(_output))} pages"
        #         )
        #         progress_manager.set_success()

        #     # shutil.rmtree(str(working_directory), ignore_errors=True)

    except Exception as e:
        # Gestion d'erreur avec progression enrichie
        error_msg = f"Erreur : {str(e)}"
        logger.error(error_msg)

        if progress_manager:
            progress_manager.set_error(error_msg)

        raise  # Re-lancer l'exception pour que l'appelant puisse la gérer


