import logging
from typing import Dict, Any, List
from pathlib import Path


logger = logging.getLogger(__name__)

def validate_nombre_exemplaires(article: Dict[str, Any]) -> int:
    """
    Validate the 'nombex' field in the article dictionary.
    If 'nombex' is not present or invalid, set it to 1.
    """
    nombex = article.get("nombex", 1)
    if not isinstance(nombex, int) or nombex < 1:
        logger.warning(f"Nombre d'exemplaires invalide pour l'article {article['code']}: {nombex}. Défini à 1.")
        return 1
    return nombex

def validate_format(article: Dict[str, Any], available_formats: Dict[str, Any], default_format:str) -> str:
    """
    Validate the 'format' field in the article dictionary.
    If 'format' is not present or invalid, set it to the default format.
    """
    art_format = article.get("format", default_format)
    if art_format not in available_formats:
        logger.warning(f"Format de l'étiquette non reconnu {art_format} pour l'article {article['code']}. Défini à {default_format}.")
        return default_format
    return art_format

def validate_categorie(article: Dict[str, Any], design_config: Dict[str, Any]) -> Dict[str, Any]:
    """
    Validate the 'categorie' field in the article dictionary.
    If the category is not defined in the design configuration, log a warning and return an empty dictionary.
    """
    ctgry = article.get("categorie")
    if ctgry not in design_config.get(article["format"], {}).get("categories", []):
        logger.warning(f"Catégorie étiquette inconnue {ctgry} dans le format {article['format']}")
        logger.warning(article)
        return {}
    return article

def validate_barcode(article: Dict[str, Any], barcode_directory: Path) -> Dict[str, Any]:
    """
    Validate the 'barcode' field in the article dictionary.
    If 'barcode' is present, generate a barcode image and return its path.
    """
    if article.get("barcode"):
        article["barcode"] = article.pop("barcode").strip()
        article["barcode_path"] = str(barcode_directory / f'{article["barcode"]}.png')
    return article

def validate_designation(article: Dict[str, Any]) -> Dict[str, Any]:
    """
    Validate the 'designation' field in the article dictionary.
    If 'designation' is not present, set it to an empty string.
    """
    if "designation" not in article:
        article["designation"] = ""
    else:
        article["designation"] = article["designation"].strip()
        if article.get("designationCom", "").strip() == "":
            article["designationCom"] = article["designation"]

    return article

def validate_article(
        article: Dict[str, Any],
        design_shortname: str,
        default_format: str,
        available_formats: Dict[str, Any],
        design_config: Dict[str, Any],
        barcode_directory: Path
        ) -> Dict[str, Any]:
    """
    Validate the article dictionary.
    """
    article["nombex"] = validate_nombre_exemplaires(article)
    article["format"] = validate_format(article, available_formats, default_format)
    article = validate_categorie(article, design_config)
    if not article:
        return {}

    article["design"] = design_shortname
    article = validate_barcode(article, barcode_directory)
    article = validate_designation(article)

    return article