import logging
from pathlib import Path
from copy import deepcopy
from typing import Optional
from collections.abc import Sequence
from yaml import load, Loader

from gescokit.etiquettes.barcode import barcode_as_img_with_zint

from .old_archi.etiquettes import prix as price

logger = logging.getLogger(__name__)

class PromoTarif:
    def __init__(self, taux=None, prix=None, gratuit=None, **kwargs):
        self.taux = int(taux) if taux is not None else None
        self.prix = prix or None
        self.gratuit = int(gratuit) if gratuit is not None else None

    @staticmethod
    def from_dict(dic=None):
        return PromoTarif(**dic) if dic else PromoTarif()

    def to_dict(self):
        return {
            key: getattr(self, key)
            for key in dir(self)
            if not key.startswith("__") and not callable(getattr(self, key))
        }

class LabelModel:
    def __init__(
        self,
        design: str,
        prix: dict,
        designation: str,
        barcode: str,
        barcode_path: Path,
        code: str,
        format: str = "",
        devise: str = "€",
        nombex: int = 1, # Nombre d'exemplaire de cette étiquette à imprimer mais non utilisé pour l'instant
        prix_ancien: Optional[dict] = None,
        promo_tarif: Optional[dict] = None,
        designationCom: Optional[str] = None,
        lettrage: Optional[str] = None,
        font_designation: Optional[str] = None,
        font_devise: Optional[str] = None,
        margins: Optional[dict] = None,
        limit_designation: Optional[int] = None,
        categorie: Optional[str] = None,
        contenance: Optional[float] = None,
        unite: Optional[str] = None,
        font_directory: str = "/home/local/fonts/",
        **kwargs
    ):

        self.design_name = design
        self.design_directory = Path(__file__).parent / "design" / design
        if not self.design_directory.is_dir():
            raise NotADirectoryError(f"Le dossier de design '{self.design_directory}' n'existe pas ou n'est pas un dossier")

        self.config_file = self.design_directory / "config.yaml"
        if not self.config_file.is_file():
            raise FileNotFoundError(f"Fichier de configuration introuvable : {self.config_file}")
        self.promo_tarif = PromoTarif.from_dict(promo_tarif) if promo_tarif else PromoTarif()

        self.designation = designation[:limit_designation] if limit_designation and designation else designation
        self.designationCom = designationCom[:limit_designation] if limit_designation and designationCom else designationCom
        self.prix = price.PrixHtTtc(contenance=contenance, contenance_unit=unite, **prix)
        self.prix_ancien = price.PrixHtTtc(**prix_ancien) if prix_ancien else None
        if self.prix_ancien and self.prix_ancien.ttc <= self.prix.ttc:
            self.prix_ancien = None

        self.format = format.lower()
        self.barcode = barcode
        self.barcode_path = barcode_path
        self.codeInterne = code
        self.lettrage = lettrage
        self.font_designation = font_designation
        self.font_devise = font_devise
        self.devise = devise
        self.border = True
        # Marges par rapport au coin supérieur gauche (top, left)
        # et entre 2 étiquettes (eti_right, eti_bot)
        self.margins = margins or {"top":"1cm","left":"1cm","eti_right":"3mm","eti_bot":"1cm"}
        self.unite = unite
        self.contenance = contenance
        self.font_directory = font_directory

        self.config = self._load_config()
        self.config = self._drop_categories(self.config, categorie)
        self.nombex = nombex
        self._sanity_check_config()

    def _load_config(self):
        if not self.config_file or not self.config_file.is_file():
            raise FileNotFoundError(f"Fichier de config introuvable : {self.config_file}")
        with open(self.config_file, encoding="utf8") as cfg:
            data = load(cfg, Loader=Loader)

        if self.format not in data.get(self.design_name,{}):
            raise ValueError(f"Format '{self.format}' non trouvé dans la configuration")
        return self._add_path_to_img(data[self.design_name][self.format])

    def _add_path_to_img(self, data):
        """Si une des categories contient une image il faut y rajouter le chemin vers cette image
        qui dépend du modèle
        """
        if "categories" in data:
            for value in data["categories"].values():
                if "img" in value:
                    img_path = self.design_directory / "img" / value["img"]
                    if img_path.is_file():
                        value["img"] = img_path
                    else:
                        logger.warning("Image introuvable: %s", img_path)
                        value["img"] = None
        return data

    def _drop_categories(self, config, cat_key):
        """Supprime toutes les catégories non utilisees pour l'étiquette"""

        if "categories" in config:
            categories = config.pop("categories")
            selected = categories.get(cat_key)

            config["categorie"] = selected
            config["categorie"]["text"] = selected.get("text", "")
            config["categorie"]["key"] = cat_key
        return config

    def _sanity_check_config(self):
        ctgry = self.config.get("categorie", {})
        rgb = ctgry.get("RGB")
        if rgb and (not isinstance(rgb, (list, tuple)) or len(rgb) != 3):
            logger.warning("RGB invalide pour categorie: %s", rgb)
            ctgry["RGB"] = None

        # Permet d'afficher le taux de remise de la promottion
        ctgry.setdefault("show_taux", False)
        ctgry.setdefault("text", "")
        self.config.setdefault("bgcolor", (255, 255, 255))
        self.config.setdefault("fgcolor", (0, 0, 0))

    def update_config(self, user_config):
        """Mets à jour la configuration par défaut avec celle de l'utilisateur.
        Seule la catégorie du format de cette étiquette est affectée.

        Permet une légère personnalisation de l'étiquette sans aller toucher à design/config.yaml
        """
        custom = deepcopy(user_config.get(self.format, {}))
        self._update_dict(self.config, self._drop_categories(self._add_path_to_img(custom), self.config["categorie"].get("key")))
        self._sanity_check_config()

    def _update_dict(self, base, updates):
        """Met à jour le dictionnaire de base avec les valeurs du dictionnaire updates."""
        for k, v in updates.items():
            if isinstance(v, dict) and isinstance(base.get(k), dict):
                self._update_dict(base[k], v)
            else:
                base[k] = v

    def get_formats(self) -> list[str]:
        """Tous les fichiers contenus dans le dossier design_directoryr/*.lua sont
        des formats (tailles) d'étiquettes disponibles pour ce modèle/design
        Returns
        -------
            formats: list of str
                exemples: ["15x15", "a4"]
        """
        return [
            luafile.name.replace(".lua", "")
            for luafile in self.design_directory.glob("*.lua")
            if luafile.is_file()
        ]


    def __str__(self):
        return f"Etiquette {self.designation} - Prix: {self.prix}"

    def generate_barcode(self, output: Optional[Path] = None) -> Optional[Path]:
        """Génère le code-barres de l'étiquette et le sauvegarde dans le fichier output."""
        if not self.barcode:
            logger.warning("Aucun code-barres à générer pour l'étiquette %s", self.designation)
            return None
        if output is None:
            output = self.barcode_path or Path(f"{self.barcode}.png")
        return barcode_as_img_with_zint(self.barcode, output)