#!/usr/bin/env python3
"""
Script pour générer des aperçus PNG pour chaque format d'étiquette et chaque design.

Ce script génère des aperçus pour tous les formats disponibles dans tous les designs
configurés dans le système d'étiquettes. Les aperçus sont sauvegardés dans le format
attendu par l'interface utilisateur GTK.
"""

import json
import logging
import subprocess
import tempfile
from pathlib import Path
from typing import Dict, Any, List, Optional
import yaml
import typer
from .config import get_designs, DESIGN_DIRECTORY
from .main import generate_labels_from_json

logger = logging.getLogger(__name__)

# Répertoire où les aperçus doivent être stockés pour l'interface GTK
PREVIEW_OUTPUT_DIR = Path("/home/local/etc/gcv6/etiquettes/design")

# Article exemple pour générer les aperçus
SAMPLE_ARTICLE = {
    "categorie":	"gondole",
    "format":	"gondole",
    "nombex":	1,
    "designation":	"Désignation de test pour aperçus",
    "designationCom":	"Désignation de test pour aperçus",
    "barcode":	"1234567890123",
    "code":	"PREVIEW-CODE",
    "lettrage":	"A",
    "contenance":	200,
    "unite":	"ml",
    "prix":	{
        "ht":	10.94009,
        "ttc":	11.87,
        "deee":	0,
        "ecomeuble":	0
    },
    "prix_ancien":	{
        "ht":	15.6221,
        "ttc":	16.95
    },
    "promo_tarif":	{
        "gratuit":	0,
        "lot":	0,
        "taux":	0,
        "prix":	10.94009
    }
}

def create_sample_json_file(design_name: str, format_name: str, category: str, papier: str = 'a4') -> Path:
    """
    Crée un fichier JSON temporaire avec un article d'exemple.

    Args:
        format_name: Format de l'étiquette (a4, a7, 15x15, gondole)
        category: Catégorie de l'étiquette (PROMO, PUB, etc.)

    Returns:
        Path vers le fichier JSON temporaire
    """
    # Créer l'article en modifiant le format et la catégorie
    article = {
        **SAMPLE_ARTICLE,
        "categorie": category,
        "format": format_name
    }

    sample_data = {
        "output": f"/tmp/preview_{design_name}_{format_name}_{category}.pdf",
        "papier": papier,  # Format papier par défaut pour les aperçus
        "design": design_name,
        "format": format_name,
        "articles": [article]
    }

    temp_file = tempfile.NamedTemporaryFile(mode='w', prefix='eti_preview_', suffix='.json', delete=False, encoding='utf-8')
    json.dump(sample_data, temp_file, ensure_ascii=False, indent=2)
    temp_file.close()

    return Path(temp_file.name)

def create_sample_config_file(design_name: str, format_name: str, category: str) -> Path:
    """
    Crée un fichier de configuration temporaire pour la génération d'aperçu.

    Args:
        design_name: Nom du design
        format_name: Format de l'étiquette
        category: Catégorie de l'étiquette

    Returns:
        Path vers le fichier de configuration temporaire
    """
    config_data = {
        "designs": {
            design_name: {
                "display_nom": f"Aperçu {design_name}",
                "formats": {
                    format_name: {
                        "categories": [category]
                    }
                }
            }
        }
    }

    temp_file = tempfile.NamedTemporaryFile(mode='w', prefix='eti_preview_', suffix='.json', delete=False, encoding='utf-8')
    json.dump(config_data, temp_file, ensure_ascii=False, indent=2)
    temp_file.close()

    return Path(temp_file.name)

def get_format_categories(design_config: Dict[str, Any]) -> Dict[str, List[str]]:
    """
    Extrait tous les formats et leurs catégories disponibles d'un design.

    Args:
        design_config: Configuration du design

    Returns:
        Dictionnaire avec les formats comme clés et listes de catégories comme valeurs
    """
    format_categories = {}

    for format_name, format_config in design_config.items():
        if isinstance(format_config, dict) and "categories" in format_config:
            categories = list(format_config["categories"].keys())
            if categories:
                format_categories[format_name] = categories

    return format_categories

def convert_pdf_to_preview_png(pdf_path: Path, output_path: Path, resolution: int = 300, max_height: int = 300) -> bool:
    """
    Convertit un PDF en PNG d'aperçu avec pdftocairo, en limitant la hauteur maximale.

    Args:
        pdf_path: Chemin vers le fichier PDF source
        output_path: Chemin vers le fichier PNG de sortie
        resolution: Résolution en DPI pour l'image de sortie
        max_height: Hauteur maximale en pixels (défaut: 300)

    Returns:
        True si la conversion a réussi, False sinon
    """
    try:
        # Créer le répertoire de sortie si nécessaire
        output_path.parent.mkdir(parents=True, exist_ok=True)

        # Commande pdftocairo pour convertir en PNG avec limitation de hauteur
        cmd = [
            "pdftocairo",
            "-png",
            "-singlefile",           # Une seule page
            "-r", str(resolution),   # Résolution en DPI
            "-scale-to-y", str(max_height),  # Limite la hauteur à max_height pixels
            "-scale-to-x", "-1",     # Garde les proportions (largeur automatique)
            str(pdf_path),           # Fichier PDF source
            str(output_path.with_suffix(''))  # Fichier de sortie sans extension (pdftocairo ajoute .png)
        ]

        result = subprocess.run(cmd, capture_output=True, text=True, timeout=30)

        if result.returncode == 0:
            logger.info(f"Aperçu généré (max {max_height}px hauteur): {output_path}")
            return True
        else:
            logger.error(f"Erreur conversion pdftocairo: {result.stderr}")
            return False

    except subprocess.TimeoutExpired:
        logger.error(f"Timeout lors de la conversion de {pdf_path}")
        return False
    except Exception as e:
        logger.error(f"Erreur lors de la conversion {pdf_path}: {e}")
        return False

def generate_preview_for_format(
        design_name: str, format_name: str,
        category: str, force: bool = False,
        output_dir: Optional[Path] = None,
        no_compile: bool = False,
        verbose: bool = False,
        attached_pid: int = 0
        ) -> bool:
    """
    Génère un aperçu PNG pour un format et une catégorie spécifiques.

    Args:
        design_name: Nom du design
        format_name: Format de l'étiquette
        category: Catégorie de l'étiquette
        force: Force la régénération même si l'aperçu existe
        enabled_shared_memory: Active la communication par mémoire partagée pour suivi de progression

    Returns:
        True si l'aperçu a été généré avec succès, False sinon
    """
    # Chemin de sortie pour l'aperçu avec la catégorie
    base_output_dir = output_dir if output_dir else PREVIEW_OUTPUT_DIR
    preview_path = base_output_dir / design_name / "images" / f"{format_name}_{category}.png"

    # Vérifier si l'aperçu existe déjà
    if preview_path.exists() and not force:
        logger.info(f"Aperçu existant ignoré: {preview_path}")
        return True

    logger.info(f"Génération aperçu: {design_name}/{format_name} (catégorie: {category})")

    # Fichiers temporaires
    json_file = None
    config_file = None

    try:
        # Créer les fichiers temporaires
        json_file = create_sample_json_file(design_name, format_name, category, format_name)
        config_file = create_sample_config_file(design_name, format_name, category)

        with open(json_file, 'r', encoding='utf-8') as f:
            pdf_path = Path(json.load(f)["output"])

        # Générer le PDF avec gescokit
        try:
            _result = generate_labels_from_json(
                json_file=json_file,
                config_file=config_file,
                design_shortname=design_name,
                output=pdf_path,
                build_pngs=False,  # On génère d'abord le PDF
                max_labels=1,
                debug=0,
                verbose=verbose,
                no_latex=no_compile,
                attached_pid=attached_pid
            )

            # Si no_compile est activé, on ne génère que les fichiers sources
            if no_compile:
                # Essayer de trouver le répertoire de travail temporaire
                working_dir = pdf_path.parent
                logger.info(f"Fichiers sources générés pour {design_name}/{format_name} dans {working_dir}")
                return True

            # Vérifier si le PDF existe, même si generate_labels_from_json retourne False
            # (peut arriver avec des warnings LaTeX qui ne sont pas critiques)
            if not pdf_path.exists():
                logger.error(f"Échec génération PDF pour {design_name}/{format_name}: fichier non créé")
                return False
            else:
                logger.info(f"PDF généré avec succès: {pdf_path} ({pdf_path.stat().st_size} bytes)")

        except Exception as e:
            logger.error(f"Erreur lors de la génération: {e}")
            # Si no_compile, on considère que c'est un succès (fichiers sources générés)
            if no_compile:
                working_dir = pdf_path.parent
                logger.info(f"Fichiers sources générés pour {design_name}/{format_name} dans {working_dir}")
                return True
            # Sinon, vérifier quand même si le PDF a été créé malgré l'exception
            if not pdf_path.exists():
                return False
            else:
                logger.warning(f"PDF créé malgré l'erreur: {pdf_path}")

        # Convertir le PDF en PNG d'aperçu (seulement si compilation activée)
        # Limiter la hauteur à 300 pixels maximum
        success = convert_pdf_to_preview_png(pdf_path, preview_path, max_height=300)

        # Nettoyer le fichier PDF temporaire
        try:
            pdf_path.unlink()
        except:
            pass

        return success

    except Exception as e:
        logger.error(f"Erreur lors de la génération de l'aperçu: {e}")
        return False

    finally:
        # Nettoyer les fichiers temporaires
        for temp_file in [json_file, config_file]:
            if temp_file and temp_file.exists():
                try:
                    temp_file.unlink()
                except:
                    pass

def generate_all_previews(
        force: bool = False, design_filter: Optional[str] = None,
        output_dir: Optional[Path] = None, no_compile: bool = False,
        verbose: bool = False,
        debug: bool = False,
        attached_pid: int = 0
        ) -> Dict[str, Dict[str, Dict[str, Any]]]:
    """
    Génère tous les aperçus pour tous les designs, formats et catégories disponibles.

    Args:
        force: Force la régénération même si les aperçus existent
        design_filter: Ne traiter que ce design spécifique (optionnel)
        no_compile: Ne compile pas les fichiers LaTeX (génère uniquement les fichiers sources)
        verbose: Affichage détaillé
        attached_pid: PID du processus attaché pour la communication par mémoire partagée (0 = désactivé)

    Returns:
        Dictionnaire avec les résultats de génération par design, format et catégorie
        Format: {design_name: {"design_path": Path, "formats": {format_name: {"categories": {category: {"success": bool, "path": Path}}}}}}
    """
    results = {}

    # Initialisation du gestionnaire de progression si attached_pid fourni
    progress_manager = None
    total_operations = 0
    current_operation = 0

    # if attached_pid > 0:
    #     try:
    #         from .shared_progress import EtiqSharedProgress, EtiqGenerationStep, get_step_name
    #         progress_manager = EtiqSharedProgress(attached_pid)
    #         progress_manager.update_progress(
    #             EtiqGenerationStep.GENERATE_LATEX,
    #             0,
    #             "Initialisation génération aperçus"
    #         )
    #         logger.info(f"Gestionnaire de progression initialisé pour PID {attached_pid}")
    #     except Exception as e:
    #         logger.warning(f"Erreur lors de l'initialisation de la progression: {e}")
    #         attached_pid = 0

    try:
        # Récupérer tous les designs disponibles avec leurs chemins
        designs_with_paths = {}
        from .config import get_design_config

        # Utiliser get_designs() pour obtenir la liste des designs disponibles
        try:
            designs = get_designs()
            for design_name, design_data in designs.items():
                try:
                    # Utiliser get_design_config pour obtenir le vrai chemin du design
                    _, design_config_file = get_design_config(design_name)
                    # Le chemin du design est le répertoire parent du config.yaml
                    design_path = design_config_file.parent
                    designs_with_paths[design_name] = {
                        "data": design_data,
                        "path": design_path
                    }
                except Exception as design_error:
                    logger.warning(f"Erreur lors du chargement du design {design_name}: {design_error}")
        except Exception as e:
            logger.warning(f"Impossible de récupérer les designs: {e}")

        if not designs_with_paths:
            logger.error("Aucun design trouvé")
            return results

        # Filtrer par design si spécifié
        if design_filter:
            if design_filter not in designs_with_paths:
                logger.error(f"Design '{design_filter}' introuvable")
                return results
            designs_with_paths = {design_filter: designs_with_paths[design_filter]}

        # # Calculer le nombre total d'opérations pour la progression
        # if progress_manager:
        #     for design_name, design_info in designs_with_paths.items():
        #         design_data = design_info["data"]
        #         if design_name in design_data:
        #             design_config = design_data[design_name]
        #             format_categories = get_format_categories(design_config)
        #             for format_name, categories in format_categories.items():
        #                 total_operations += len(categories)

        #     logger.info(f"Progression initialisée pour {total_operations} aperçus")

        for design_name, design_info in designs_with_paths.items():
            design_data = design_info["data"]
            design_path = design_info["path"]
            logger.info(f"Traitement du design: {design_name}")
            results[design_name] = {
                "design_path": design_path,
                "formats": {}
            }

            # Récupérer la configuration du design
            if design_name not in design_data:
                logger.warning(f"Configuration manquante pour le design {design_name}")
                continue

            design_config = design_data[design_name]
            format_categories = get_format_categories(design_config)

            if not format_categories:
                logger.warning(f"Aucun format trouvé pour le design {design_name}")
                continue

            # Générer un aperçu pour chaque combinaison (format, catégorie)
            for format_name, categories in format_categories.items():
                results[design_name]["formats"][format_name] = {
                    "categories": {}
                }

                # Générer un aperçu pour chaque catégorie du format
                for category in categories:
                    # # Mettre à jour la progression
                    # if shared_memory_manager:
                    #     progress_message = f"Génération {design_name}/{format_name}_{category}"
                    #     shared_memory_manager.update_progress(0, progress_message)

                    # Calculer le chemin de l'aperçu avec la catégorie
                    base_output_dir = output_dir if output_dir else PREVIEW_OUTPUT_DIR
                    preview_path = base_output_dir / design_name / "images" / f"{format_name}_{category}.png"

                    # # Mettre à jour la progression avant génération
                    # if progress_manager:
                    #     current_operation += 1
                    #     progress_percent = int((current_operation * 100) / total_operations) if total_operations > 0 else 0
                    #     progress_message = f"Aperçu {current_operation}/{total_operations}: {design_name}/{format_name}_{category}"
                    #     progress_manager.update_progress(
                    #         EtiqGenerationStep.GENERATE_LATEX,
                    #         progress_percent,
                    #         progress_message
                    #     )

                    success = generate_preview_for_format(
                        design_name, format_name, category, force, output_dir, no_compile, verbose,
                        attached_pid
                    )

                    # Si no_compile, on ne génère pas de PNG mais on retourne quand même le chemin logique
                    result_path = None
                    if success:
                        if no_compile:
                            # Retourner un chemin vers le répertoire de fichiers sources
                            result_path = f"fichiers sources LaTeX dans répertoire temporaire"
                        else:
                            result_path = preview_path

                    results[design_name]["formats"][format_name]["categories"][category] = {
                        "success": success,
                        "path": result_path
                    }


    except Exception as e:
        logger.error(f"Erreur lors de la génération des aperçus: {e}")
        if progress_manager:
            progress_manager.set_error(f"Erreur lors de la génération des aperçus: {str(e)}")
        return results

    # Finaliser la progression avec succès
    if progress_manager:
        progress_manager.set_success()

    return results

# Interface CLI avec typer
app = typer.Typer(help="Générateur d'aperçus PNG pour les étiquettes")

@app.command("generate-previews")
def cli_generate_previews(
    force: bool = typer.Option(False, "--force", "-f", help="Force la régénération même si les aperçus existent"),
    design: Optional[str] = typer.Option(None, "--design", "-d", help="Ne traiter que ce design spécifique"),
    verbose: bool = typer.Option(False, "--verbose", "-v", help="Affichage détaillé"),
    output_dir: Optional[Path] = typer.Option(None, "--output-dir", "-o", help="Répertoire de sortie (par défaut: /home/local/etc/gcv6/etiquettes/design)")
):
    """
    Génère des aperçus PNG pour tous les formats d'étiquettes disponibles.
    """
    # Configuration du logging
    log_level = logging.DEBUG if verbose else logging.INFO
    logging.basicConfig(level=log_level, format='%(levelname)s: %(message)s')

    # Modifier le répertoire de sortie si spécifié
    global PREVIEW_OUTPUT_DIR
    if output_dir:
        PREVIEW_OUTPUT_DIR = output_dir

    logger.info(f"Génération des aperçus dans: {PREVIEW_OUTPUT_DIR}")

    # Vérifier que pdftocairo est disponible
    try:
        subprocess.run(["pdftocairo", "-v"], capture_output=True, check=True)
    except (subprocess.CalledProcessError, FileNotFoundError):
        typer.echo("Erreur: pdftocairo n'est pas disponible", err=True)
        typer.echo("Veuillez installer poppler-utils: yum install poppler-utils ou dnf install poppler-utils", err=True)
        raise typer.Exit(1)

    # Générer les aperçus
    results = generate_all_previews(force=force, design_filter=design, output_dir=output_dir, verbose=verbose)

    # Afficher les résultats
    total_generated = 0
    total_failed = 0

    # results[design_name]["formats"][format_name]["categories"][category] = {
    #     "success": success,
    #     "path": result_path
    # }
    for design_name, design_info in results.items():
        design_path = design_info["design_path"]
        design_results = design_info["formats"]
        typer.echo(f"\nDesign: {design_name} → {design_path}")
        for format_name, format_info in design_results.items():
            categories_results = format_info["categories"]
            for category, result_info in categories_results.items():
                success = result_info["success"]
                path = result_info["path"]
                status = "✓" if success else "✗"
                if success and path:
                    typer.echo(f"  {format_name}_{category}: {status} → {path}")
                else:
                    typer.echo(f"  {format_name}_{category}: {status}")
                if success:
                    total_generated += 1
                else:
                    total_failed += 1

    typer.echo(f"\nRésumé: {total_generated} aperçus générés, {total_failed} échecs")

    if total_failed > 0:
        raise typer.Exit(1)

if __name__ == "__main__":
    app()