from pathlib import Path
from typing import List, Dict



def int_to_base26_letters(n: int) -> str:
    """
    Convertit un entier 1-indexé en représentation alphabétique base-26 (A, B, ..., Z, AA, AB, ...).
    """
    result = ''
    while n > 0:
        n, remainder = divmod(n - 1, 26)
        result = chr(65 + remainder) + result
    return result


def write_font_preamble(articles: List[Dict], font_directory: str, working_directory: Path) -> None:
    """
    Génère un fichier LaTeX "fontspreamble.tex" déclarant toutes les polices utilisées dans les articles.

    Parameters
    ----------
    articles : List[Dict]
        Liste des articles contenant des références à des polices (font_designation, font_devise, etc).
    font_directory : str
        Répertoire des polices utilisé dans LaTeX (ex. /home/local/fonts).
    working_directory : Path
        Répertoire temporaire de travail où écrire le fichier "fontspreamble.tex".
    """
    if not font_directory:
        return

    polices: Dict[str, str] = {}
    idx_police = 0

    for article in articles:
        for key in list(article.keys()):
            if key.startswith("font") and key != "font_dossier":
                font_name = article[key]
                if font_name not in polices:
                    idx_police += 1
                    macro_name = "\\police" + int_to_base26_letters(idx_police)
                    polices[font_name] = macro_name
                    article[key] = macro_name
                else:
                    article[key] = polices[font_name]

    output_file = working_directory / "fontspreamble.tex"
    with open(output_file, "w", encoding="utf8") as fp:
        fp.write("\\usepackage{fontspec}\n")
        fp.write(f"\\defaultfontfeatures{{Path={font_directory}/}}\n")
        for font_name, macro_name in polices.items():
            if "/" in font_name:
                subdir, font_stem_ext = font_name.rsplit("/", 1)
                font_stem, font_ext = font_stem_ext.rsplit(".", 1)
                fp.write(fr"\newfontfamily{{{macro_name}}}[" + "\n")
                fp.write(fr"    Path=/home/local/fonts/{subdir}/," + "\n")
                fp.write(f"    Extension=.{font_ext}\n")
                fp.write(f"]{{{font_stem}}}\n")
            else:
                fp.write(f"\\newfontfamily{{{macro_name}}}{{{font_name}}}\n")
