from pathlib import Path
from typing import Any, Dict, Tuple, List
import json
import logging

from yaml import safe_load, YAMLError
logger = logging.getLogger(__name__)

DESIGN_DIRECTORY = Path("/home/local/etc/gcv6/etiquettes/design")

def read_design_configs(config_file: Path, design_shortname: str) -> Tuple[Dict[str, Any], Path, Dict[str, Any]]:
    """
    Lit un fichier YAML de configuration de la société et retourne la structure associée.

    Parameters
    ----------
    config_file : Path
        Chemin absolu vers le fichier `config.yaml`

    Returns
    -------
    Dict[str, Any]
        - le dictionnaire de configuration (avec tous les formats d’étiquettes définis pour la société)

    Raises
    ------
    FileNotFoundError
        Si le fichier de configuration n'existe pas.

    """
    if not config_file.is_file():
        raise FileNotFoundError(f"Fichier de configuration introuvable: {config_file}")

    design_config, design_config_file = get_design_config(design_shortname)

    with open(config_file, "r", encoding="utf-8") as f:
        data = json.load(f)

    if design_shortname not in data.get("designs", {}):
        raise ValueError(f"Le design '{design_shortname}' n'est pas défini dans le fichier de configuration.")

    return design_config, design_config_file, data["designs"][design_shortname]


def get_design_config(design_shortname: str) -> Tuple[Dict[str, Any], Path]:

    # On recherche le fichier YAML de configuration du design dans 2 répertoires:
    # 1) le répertoire contenant ce fichier que vous lisez (EN DEV)
    # 2) le répertoire de l'application (par exemple /home/local/etc/etiquettes/design) EN PROD

    current_dir = Path(__file__).parent / "design"
    for directory in [current_dir, DESIGN_DIRECTORY]:
        design_config_file = directory / design_shortname / "config.yaml"
        if design_config_file.is_file():
            with open(design_config_file, "r", encoding="utf-8") as f:
                try:
                    design_config = safe_load(f)
                except YAMLError as e:
                    logger.error(f"Erreur lors de la lecture du fichier de configuration YAML: {e}")
                    raise
            return design_config[design_shortname], design_config_file
    else:
        # Si le fichier n'est pas trouvé dans les deux répertoires, on lève une exception
        raise FileNotFoundError(f"Fichier de configuration du design introuvable: {design_shortname}")

def get_designs() -> Dict[str, Any]:
    """
    Retourne un dictionnaire contenant les designs disponibles.

    Returns
    -------
    Dict[str, Any]
        Dictionnaire avec les noms des designs comme clés et leurs configurations comme valeurs.
    """
    designs = {}

    # Chercher d'abord dans le répertoire de production
    # FIXME: installer les fichiers de design dans le répertoire de production
    directories_to_check = [DESIGN_DIRECTORY]

    # Ajouter le répertoire de développement comme fallback
    dev_design_dir = Path(__file__).parent / "design"
    if dev_design_dir.exists():
        directories_to_check.append(dev_design_dir)

    for base_dir in directories_to_check:
        if not base_dir.exists():
            logger.debug(f"Répertoire de designs inexistant: {base_dir}")
            continue

        for design_dir in base_dir.iterdir():
            if design_dir.is_dir():
                config_file = design_dir / "config.yaml"
                if config_file.is_file():
                    try:
                        with open(config_file, "r", encoding="utf-8") as f:
                            designs[design_dir.name] = safe_load(f)
                        logger.debug(f"Design chargé: {design_dir.name} depuis {base_dir}")
                    except YAMLError as e:
                        logger.error(f"Erreur lors de la lecture du fichier de configuration YAML: {e}")

    return designs