__VERSION__="2.0.2"
from pathlib import Path

import pandas as pd
import numpy as np


from . import utils
from . import pyisql_adapter as pyisql


def clear_data(line):
    """Converti line en liste de str et supprime les " ainsi que les espaces superflux"""
    return [l.replace('"', "").strip() for l in line.split(";")]


def fill_adEvent(shopId, ticket_header, ticket_details):

    # shopId, numfac, date, total_ticket, details=None, numcrt=None, idMember=None
    """Construit l'objet AdEvent.
    L'AdEvent est la representation du ticket dans Adelya
    """

    adEvent = {
        "type": "addCA",
        "idExternal": ticket_header["numfac"],
        "date": ticket_header["date"],
        "fvalue": ticket_header["total_ticket"],
        "group": {"loadFromKeys": {"idExternal": shopId}},
    }

    if ticket_details is not None:
        adEvent["details"] = convert_ticket_to_AdEventDetails_object(ticket_details)

    if offres := ticket_header.get("offres"):
        details = adEvent.get("details")
        if details is None:
            details = []

        details.extend(
            [
                {
                    "AdEventDetail": {
                        "detailType": "C",
                        # FIXME: VErifier que l'on n'a pas inverser class1 et class10
                        "class1": offre["desig"],
                        "class10": offre["code"],
                    }
                }
                for offre in offres
            ]
        )
        adEvent["details"] = details

    if ref_piece := ticket_header.get("ref_piece"):
        adEvent["sparam"] = ref_piece

    if (idMember := ticket_header["idMember"]) is not None:
        adEvent["member"] = {"id": idMember}
    else:
        adEvent["member"] = {"loadFromKeys": {"cardnumber": ticket_header["numcrt"]}}

    data = {"AdEvent": adEvent}

    return data


def get_login(idMember):
    """Extract the login in the fake card number
    In Debug mode, the card number is following this rules
    1st char is 1
    2nd char is the length of the login
    3rd to 3+length is the login
    """
    return idMember[2 : 2 + int(idMember[1])]


def convert_ticket_to_dict(filepath:str) -> list[dict]:
    """Converti le fichier filepath en liste de dictionnaires.

    Les clés du dictionnaires sont les colonnes du fichier CSV. Les noms des colonnes sont
    déduites de la première ligne du fichier.
    """
    with open(filepath, "r", encoding="cp850") as f:
        lines = [line.strip() for line in f.readlines()]
    # le header attendu est :
    # "num_ticket";"date_ticket";"heure_ticket";"client";"ref_article";"libelle";
    # "quantité";"pu_rev";"tx_tva";"pu_vente_ttc";"total_ttc";"reference";"theme";
    # "histoire";"famille";"departement";"sous-departement";"remise_15";"total_ticket";
    # "code";"fam";"offre";"offre_desig";"ref_piece"
    # TODO ajouter un sanity_check_ticket ici pour vérifier que les colonnes sont bien présentes

    header = clear_data(lines[0])
    lines = [clear_data(line) for line in lines[1:]]
    csv_data = [{h: l for h, l in zip(header, line)} for line in lines]
    # Remarque : dans le cas du ticket le code client est en fait le numéro de carte fidélité
    csv_data = [line | {"numcrt": line["client"]} for line in csv_data]
    return header+["numcrt"], csv_data


def sanity_check_ticket(header, keys, filename):
    for key in keys.keys():
        assert key in header, "{} not in header".format(key)
    assert "date_ticket" in header, "{}: date_ticket not in header".format(filename)
    assert "num_ticket" in header, "{}: num_ticket not in header".format(filename)
    assert "ref_article" in header, "{}: ref_article not in header".format(filename)
    assert "client" in header, "{}: client not in header".format(filename)
    assert "total_ticket" in header, "{}: total_ticket not in header".format(filename)


def convert_ticket_to_AdEventDetails_object(ticket_data):
    """Converti toutes les lignes du ticket en une liste d'AdEventDetail."""
    # Colonnes extraites du fichier CSV et leur equivalent dans Adelya
    replace = {
        "libelle": "class1",
        "code": "class10",
        "fam": "class2",
        "quantité": "quantity",
        "pu_vente_ttc": "unitValue",
        "total_ttc": "finalValue",
    }

    # sanity_check_ticket(header, replace, filepath)

    # Ne conserver de ticket_data
    # que les colonnes qui nous intéressent puis les renommer.
    # { libelle: "xxx", code: "yyy", ...} => { class1: "xxx", class10: "yyy", ...}
    details = [{v: d[k] for k, v in replace.items()} for d in ticket_data]
    for detail in details:
        detail["detailType"] = "I"
    return [{"AdEventDetail": detail} for detail in details]


def group_tickets_by_numfac(filepath):

    _, csvdata = convert_ticket_to_dict(filepath)

    # On suppose qu'il ne peut avoir qu'une seule date par ticket (num_ticket)
    # idem pour le client, idem pour ref_piece
    # On rassemble les lignes par ticket. Un ticket est identifié par son numéro de facture
    tickets = {}
    for data in csvdata:
        ticket = data["num_ticket"]
        if ticket not in tickets.keys():
            date = data["date_ticket"]  # TODO: convertir en date ou datetime
            # FIXME : Ceci est le fuseau horaire de Guyane
            date = "{}-{}-{}T{}-03:00".format(date[6:],date[3:5],date[:2],data["heure_ticket"])
            tickets[ticket] = {
                "date": date,
                "client": data["client"],
                "numcrt": data["numcrt"],
                "total_ticket": data["total_ticket"],
                "ref_piece": data.get("ref_piece"),
                "lines": [],
                "offres": [],
            }

        tickets[ticket]["lines"].append(data)
        # Pour 1 ligne, il n'y a qu'une seule offre ou 0.
        if offre := data.get("offre"):
            tickets[ticket]["offres"].append(
                {"code": offre, "desig": data["offre_desig"]}
            )

    for num_ticket in list(tickets.keys()):
        ticket = tickets[num_ticket]
        # Rendre unique les offres
        ticket["offres"] = list( set(ticket["offres"]) )

    return tickets

def group_tickets_by_client_and_numfac(filepath:str) -> dict[str, dict[str, dict]]:
    """Tris les tickets du CSV par clients et par numfac.

    Parameters
    ----------
    filepath : str
        Chemin du fichier CSV contenant les tickets.

    Returns
    -------
    dict[str, dict[str, dict]]


    Retourne un dictionnaire de la forme:
    {
        client1: {
            numfac1: {date: date, total_ticket: total_ticket, offres: offres, ref_piece: ref_piece, lines: [lines]},
            numfac2: {date: date, total_ticket: total_ticket, offres: offres, ref_piece: ref_piece, lines: [lines]},
            ...
        },
        client2: {
            numfac1: {date: date, total_ticket: total_ticket, offres: offres, ref_piece: ref_piece, lines: [lines]},
            numfac2: {date: date, total_ticket: total_ticket, offres: offres, ref_piece: ref_piece, lines: [lines]},
            ...
        },
        ...
    }
    """
    tickets_by_numfac = group_tickets_by_numfac(filepath)
    tickets_by_clients_numfac = {}
    for numfac, data in tickets_by_numfac.items():
        client = data["client"]
        if client not in tickets_by_clients_numfac.keys():
            tickets_by_clients_numfac[client] = {}
        tickets_by_clients_numfac[client][numfac] = data

    return tickets_by_clients_numfac

def write_AdEvent_to_file(filepath, numfac, data):
    output = f"{filepath}-{numfac}.data"
    json_data = utils.format_data(data)
    with open(output, "w") as f:
        print(json_data, file=f)

    print(f"{output} written")
    return json_data

def convert_tickets_to_AdEvent(filepath, crm, DEBUG=False):
    """Lis le fichier filepath et converti les tickets en AdEvent, prêts à être envoyés à Adelya."""

    tickets = group_tickets_by_numfac(filepath)

    for numfac, data in tickets.items():
        ticket_header = {
            "numfac": numfac,
            "date": data["date"],
            "total_ticket": data["total_ticket"],
            "offres": data["offres"],
            "ref_piece": data["ref_piece"],
            "idMember": get_login(data["client"]) if DEBUG else None,
            "numcrt": None if DEBUG else data["numcrt"],
        }
        details = data["lines"]

        client = ticket_header["numcrt"] or ticket_header["idMember"]

        yield (numfac, client, fill_adEvent(crm.shopId, ticket_header, details))

# ======================================================================================================
# ======================================================================================================
# ======================================================================================================
# ======================================================================================================




# TODO: deplacer read_invoices hors de Adelya car c'est un element généré par la gestion
# et non dépendante d'adelya
def read_invoices(filepath, sep=';'):
    """Lis le fichier filename et retourne un DataFrame des lignes de facture."""
    assert Path(filepath).is_file(), f"Le fichier {filepath} n'existe pas"
    df = pd.read_csv(
        filepath, sep=sep, encoding="cp850",
        header=0, index_col=False,
        dtype={ "client":str, "numcrt":str, "numdep":int, "code art.":str, "barcode":str} | {f"fam. {i}p":str for i in range(1,5)}
        )

    # Modifier chaque cellule de la dataframe pour supprimer les espaces supplémentaires
    df = df.map(lambda x: x.strip() if isinstance(x, str) else x)
    df["dtfac"] = pd.to_datetime(df["dtfac"], format="%Y%m%d%H:%M:%S", errors="coerce")
    df.replace({"nan": None, np.nan:None}, inplace=True)
    return df


def group_by_article(df):
    """Groupe les lignes de facture par article et retourne un DataFrame."""
    df = df.groupby(["code art.", "desig. art."], as_index=False).agg(
        {"quantity": np.sum, "PU TTC": np.mean, "Total TTC": np.sum, "Mont. fac TTC": np.sum}
    )
    return df

def group_by_nofac(df):
    """Groupe les lignes de facture par numero de facture et retourne un DataFrame."""
    df = df.groupby(["nofac"], as_index=False).agg(
        {"Total TTC": np.sum, "Mont. fac TTC": np.mean}
    )
    return df



def convert_invoices_to_AdEvent(filepath, crm, with_details=True, DEBUG=False, superDbg=False):
    """Lis le fichier filepath et converti les tickets en AdEvent, prêts à être envoyés à Adelya."""

    tickets = read_invoices(filepath)

    for name, grp in tickets.groupby(["nofac"]):
        numfac = name[0]
        # print("=================== Facture:", numfac)

        first_row = grp.iloc[0]

        code_client = first_row["client"]
        numcrt = first_row["numcrt"]
        numdep = int(first_row["numdep"]) # convert to int because read_invoices convert to int64

        code_client = first_row["client"]
        # line0 = grp[["nofac","nomdep","client", "numcrt", "Mont. fac TTC", "piece orig."]].iloc[0]

        is_fid, client = pyisql.is_crtf_and_active_client(code_client)
        if not is_fid:
            print(f"ERREUR: Ce client [code:{code_client},numcrt:{numcrt}] n'est pas actif ou n'est pas un client CRTF")
            continue

        if numcrt is None or client["numcrt"] is None or client["numcrt"] != numcrt:
            print(f"ERREUR: Donneee invalide pour le client [code:{code_client},numcrt:{numcrt}]")
            continue

        date = f"{first_row['dtfac']:%Y-%m-%d}"
        total_ticket = first_row["Mont. fac TTC"]
        #print(is_fid, code_client, numcrt, numdep)
        #pp.pprint(client)
        #sys.exit(0)

        print("-------", "sendtick.py")

        ticket_header = {
            "numfac": numfac,
            "date": date,
            "total_ticket": total_ticket,
            "ref_piece": first_row["piece orig."],
            "idMember": get_login(client["numcrt"]) if DEBUG else None,
            "numcrt": None if DEBUG else client["numcrt"],
            "numcais": first_row["numcais"],
        }

        if numcrt:
            exist, fidMember = crm.get_fidMember_by_cardnumber(numcrt)
            if not exist:
                print(f"ERREUR: La carte de fidélité [{numcrt}] n'existe pas dans Adelya. Code client [{code_client}]")
                continue
            ticket_header["idMember"] = fidMember["id"]
        else:
            print(f"ERREUR: Le client [{code_client}] n'a pas de numéro de carte. Impossible de l'envoyer")
            continue

        if superDbg:
            ticket_header["idMember"] = str(int(ticket_header["idMember"])-9)
        # if DEBUG:
        #     ticket_header.pop("idMember")
        #     ticket_header["loadFromKeys"] = {"idExternal": client["numcrt"]}

        if with_details:
            #  grp["type dossier"]==4 Ne gardez que les offres commerciales
            offres = grp[ grp["type dossier"]==4 ][["num. dossier", "desig. dossier"]] \
                        .rename(columns={"num. dossier":"code", "desig. dossier":"desig"}) \
                        .drop_duplicates() \
                        .dropna() \
                        .to_dict(orient="records")
            ticket_header["offres"] = offres
            print("------- offres", offres)

            # Renommer les colonnes pour correspondre aux attentes de la fonction
            # fill_adEvent
            columns = {
                "desig. art.": "libelle",
                "code art.": "code",
                "fam. 4p.": "fam",
                "PU TTC": "pu_vente_ttc",
                "Mont. fac TTC": "total_ttc",
                "quantity": "quantité"
            }
            ticket_details = grp[columns.keys()].copy()
            # ticket_details["desig. art."] = ticket_details["desig. art."].apply(lambda x: urllib.parse.quote(x, safe=""))
            ticket_details = ticket_details.rename(columns=columns).to_dict(orient="records")
        else:
            ticket_details = None

        data = fill_adEvent(numdep, ticket_header, ticket_details)
        yield data

