from typing import Optional
from gescokit.common.compat import BaseModel, validator, Field
from pathlib import Path
import tomllib


class CRMConfig(BaseModel):
    """
    Modèle de configuration pour la connexion à l'API Adelya.

    Attributs
    ----------
    nom : str
        Nom du CRM ou du magasin.
    https : str
        "1" si HTTPS est activé, "0" sinon.
    login : str
        Identifiant API, au format `clé;nom_compte`.
    password : str
        Mot de passe API.
    url : str
        URL de l'API, doit contenir "clientId=".
    timeout : int
        Délai d'expiration des requêtes HTTP en secondes.
    idmagasin : str
        Identifiant du magasin.
    debug : bool
        Active ou désactive le mode debug.
    country : Optional[str]
        Code pays du magasin (GP, MQ, GF...). Optionnel.
    """

    nom: str
    https: str
    login: str
    password: str
    url: str
    timeout: int
    idmagasin: str
    debug: bool
    country: Optional[str] = Field(default=None)

    @validator("login")
    def login_must_have_two_parts(cls, v):
        if ";" not in v or len(v.split(";")) != 2:
            raise ValueError("Le champ 'login' doit contenir exactement deux parties séparées par un ';'")
        return v

    @validator("url")
    def url_must_contain_clientid(cls, v):
        if "clientId=" not in v:
            raise ValueError("Le champ 'url' doit contenir 'clientId='")
        return v

    @validator("https")
    def https_must_be_0_or_1(cls, v):
        if v not in ("0", "1"):
            raise ValueError("Le champ 'https' doit être '0' ou '1'")
        return v

    @classmethod
    def from_file(cls, path: Path, section: str = "crm.adelya") -> "CRMConfig":
        """
        Charge un fichier TOML et extrait la section [crm.adelya].

        Parameters
        ----------
        path : Path
            Chemin vers le fichier TOML.
        section : str
            Chemin dans la hiérarchie TOML, par défaut 'crm.adelya'

        Returns
        -------
        CRMConfig
        """
        if not path.exists():
            raise FileNotFoundError(f"Fichier de configuration introuvable : {path}")

        with path.open("rb") as f:
            raw = tomllib.load(f)

        # Naviguer dans les sections hiérarchiques (crm -> adelya)
        data = raw
        for key in section.split("."):
            data = data.get(key)
            if data is None:
                raise ValueError(f"Section '{section}' non trouvée dans le fichier {path}")

        return cls(**data)
