from datetime import datetime
from typing import Dict, Any, List, Tuple, Optional
import requests

import pandas as pd

from . import sendtick
from . import utils
from . import pyisql_adapter as pyisql
import structlog

import pprint as pp

logger = structlog.get_logger()

# Désactiver les avertissements de sécurité SSL
requests.packages.urllib3.disable_warnings()

# Mapping des genres utilisés pour les membres fidélité
gender_mapping: Dict[str, str] = {
    "mr": "M",
    "m": "M",
    "m.": "M",
    "mle": "MLLE",
    "mme": "MME"
}

class Adelya:
    """
    Classe permettant d'interagir avec l'API Adelya pour la gestion des membres de fidélité et des transactions.
    """

    def __init__(self, active_crm: Dict[str, Any], magasinCountry: str) -> None:
        """
        Initialise la connexion à l'API Adelya.

        Parameters
        ----------
        active_crm : dict
            Dictionnaire contenant les informations de connexion au CRM actif.
        """
        self.shopId: str = active_crm["idmagasin"]  # FIXME: utiliser numdep à la place
        self.DEBUG: bool = active_crm["debug"]
        self.API_URL: str = "http" + "s" if active_crm["https"] == "1" else "" + "://"
        self.API_URL += "://{login}:{password}@{url}".format(**active_crm)
        self.login: str = active_crm["login"]
        self.password: str = active_crm["password"]
        self.headers: Dict[str, str] = {"Content-Type": "application/x-www-form-urlencoded"}
        self.magasinCountry = magasinCountry

    def put(self, data: Dict[str, Any], addReturnType: bool = False) -> Dict[str, Any]:
        """
        Envoie une requête PUT à l'API Adelya.

        Parameters
        ----------
        data : dict
            Données à envoyer.
        addReturnType : bool, optional
            Si True, ajoute un type de retour à la requête.

        Returns
        -------
        dict
            Réponse JSON de l'API.
        """
        json_data = utils.format_data(data, addReturnType)

        # Ne pas vérifier le certificat SSL
        response = requests.put(
            self.API_URL,
            headers=self.headers,
            data=json_data,
            auth=(self.login, self.password),
            verify=False
        )
        if response.status_code == 200:
            return response.json() | {"sent": True}
        else:
            try:
                return response.json() | {"status_code": response.status_code, "sent": False}
            except:
                return {"error": response.text, "status_code": response.status_code, "sent": False}

    def get(self, data: Dict[str, Any]) -> Dict[str, Any]:
        """
        Envoie une requête GET à l'API Adelya.

        Parameters
        ----------
        data : dict
            Données à envoyer.

        Returns
        -------
        dict
            Réponse JSON de l'API.

        Raises
        ------
        Exception
            En cas d'erreur dans la requête.
        """
        json_data = utils.format_data(data)
        response = requests.get(f"{self.API_URL}&{json_data}", headers=self.headers)
        if response.status_code == 200:
            return response.json()
        else:
            raise Exception(response.text)

    def _check_fidelityMember_result(self, result: Dict[str, Any]) -> Tuple[bool, Dict[str, Any]]:
        """
        Vérifie si un membre fidélité est retourné dans la réponse.

        Parameters
        ----------
        result : dict
            Résultat de la requête.

        Returns
        -------
        tuple
            Booléen indiquant si un membre fidélité est trouvé et le dictionnaire correspondant.
        """
        fidMember = result.get("FidelityMember")
        if fidMember is None:
            result2 = result.get("ApiReturn")
            if result2 is None:
                return False, result
            return False, result2
        return True, fidMember

    def get_fidMember_by_mobile(self, mobile: str) -> Tuple[bool, Dict[str, Any]]:
        """
        Récupère un membre fidélité par son numéro de téléphone.

        Parameters
        ----------
        mobile : str
            Numéro de téléphone du membre.

        Returns
        -------
        tuple
            Résultat de la requête.
        """
        data = {"FidelityMember": {"mobile": mobile}}
        return self._check_fidelityMember_result(self.get(data))

    def get_fidMember_by_cardnumber(self, cardnumber: str) -> Tuple[bool, Dict[str, Any]]:
        """
        Récupère un membre fidélité par son numéro de carte.

        Parameters
        ----------
        cardnumber : str
            Numéro de carte du membre.

        Returns
        -------
        tuple
            Résultat de la requête.
        """
        data = {"FidelityMember": {"cardnumber": cardnumber}}
        return self._check_fidelityMember_result(self.get(data))

    def get_fidMember_by_id(self, idExternal: str) -> Tuple[bool, Dict[str, Any]]:
        """
        Récupère un membre fidélité par son identifiant externe.

        Parameters
        ----------
        idExternal : str
            Identifiant externe du membre.

        Returns
        -------
        tuple
            Résultat de la requête.
        """
        data = {"FidelityMember": {"idExternal": idExternal}}
        return self._check_fidelityMember_result(self.get(data))


    def get_new_fidMembers(self, from_date:str, to_date:str=None):
        """
        from_date: str
            Date à partir de laquelle on recherche les nouveaux clients.
            Attention le format est celui-ci'2024-02-01'
        """
        # FIXME Vérifier que from_date est bien une date et avec le bon format
        if to_date is None:
            data = {'FidelityMember':{'dateCreate_ge':from_date}}
        else:
            data = {'FidelityMember':{"and": [{'dateCreate_ge':from_date,'dateCreate_lt':to_date}]}}

        result = self.get(data)

        if _list := result.get("list"):
            return _list
        return []


    def get_updated_fidMembers(self, from_date:str, to_date:str=None):
        """
        exemple de valeur : '2024-02-22 16:27:00', '2024-02-22'
        """
        if to_date is None:
            data = {'FidelityMember':{'actif':'1', 'dateUpdate_ge':from_date}}
        else:
            data = {'FidelityMember':{'actif':'1', "and": [{'dateUpdate_ge':from_date,'dateUpdate_lt':to_date}]}}

        rslt = self.get(data)
        if rslt:
            return rslt["list"]


    def send_customer(
        self,
        debug,
        lastname,
        firstname,
        tel,
        gender,
        numcrt,
        ad1,
        ad2,
        cpv,
        codpost,
        email,
        birth,
        **kwargs,
    ):

        civilite = gender.lower()
        if civilite not in gender_mapping:
            civilite = "m"
            logger.warning(f"{__file__} Le 'gender' {gender} de {numcrt} n'est pas valide. Veuillez le corriger.")
            # raise ValueError(f"{__file__} Le 'gender' {gender} n'est pas valide. Veuillez le corriger.")

        fm = {
            "actif": 1,
            "type": "C",
            "name": lastname,
            "firstname": firstname,
            "mobile": tel,
            "gender": gender_mapping[civilite],
            "birthday": birth,
            "email": email,
            "group": {"loadFromKeys": {"idExternal": self.shopId}},
        }

        if not debug:
            fm["cardnumber"] = numcrt
        else:
            if "idExternal" in kwargs:
                fm["idExternal"] = kwargs["idExternal"]

        town = pyisql.get_town_by_zip(codpost)
        fm["address"] = {"line1": f"{ad1.strip()} {ad2.strip()} {cpv.strip()}","zip": codpost,"town": town,"country": self.magasinCountry}
        data = {"FidelityMember": fm}

        return self.put(data)


    def create_customer(
        self,
        lastname,
        firstname,
        mobile,
        gender,
        line1,
        line2,
        zipcode,
        email,
        birthday,
        **kwargs,
    ):

        if line2:
            logger.warning("Attention ! La ligne 2 de l'adresse est ignorée par EMC2 lors de la creation du client.")
        civilite = gender.lower()
        if civilite not in gender_mapping:
            civilite = "m"
            logger.warning(f"{__file__} Le 'gender' {gender} de {lastname} {firstname} n'est pas valide. Veuillez le corriger.")
            # raise ValueError(f"{__file__} Le 'gender' {gender} n'est pas valide. Veuillez le corriger.")

        assert isinstance(birthday, datetime), f"{__file__} La date de naissance doit être de type datetime."

        # client['birth'] = datetime.strptime(client['birth'], "%d-%m-%Y").strftime("%Y-%m-%d")


        fm = {
            "actif": 1,
            "type": "C",
            "name": lastname,
            "firstname": firstname,
            "mobile": utils.add_prefix_to_tel(mobile),
            "gender": gender_mapping[civilite],
            "birthday": f"{birthday:%Y-%m-%d}",
            "email": email,
            "group": {"loadFromKeys": {"idExternal": self.shopId}},
        }

        if "idExternal" in kwargs:
            fm["idExternal"] = kwargs["idExternal"]

        # if cardnumber := kwargs.get("cardnumber"):
        #     fm["cardnumber"] = cardnumber

        import pyisql

        nbrows, data = pyisql.get_ville(zipcode)
        town = data[0]["nom"] if nbrows > 0 else ""
        fm["address"] = {"line1": line1,"zip": zipcode, "town": town, "country": self.magasinCountry}
        data = {"FidelityMember": fm}

        r = self.put(data, addReturnType=True)

        if self.check_apireturn_is_ok(r):
            apiReturn = r.get("ApiReturn")
            data = apiReturn.get("data", None)
            if not data:
                message = f"Erreur lors de la création du client {lastname} {firstname}.\n"
                message += "La réponse de l'API ne contient pas le champs 'data'.\n"
                message += str(r)
                raise Exception(message)
            data = data.get("FidelityMember", None)
            if not data:
                message = f"Erreur lors de la création du client {lastname} {firstname}.\n"
                message += "La réponse de l'API ne contient pas le champs 'FidelityMember'.\n"
                message += str(r)
                raise Exception(message)

            return True, data

        return False, r

    def convert_tickets_to_AdEvent(self, ticket_path, debug=False):
        return sendtick.convert_tickets_to_AdEvent(ticket_path, self, debug)

    def check_customer_is_valid(self, cardnumber):
        # 1 - Vérifier que le client est un client fidélité
        return pyisql.is_valid_customer(cardnumber)



    def send_tickets(self, ticket_path:str, pre_prod:bool=False) -> Tuple[list, list, list]:
        """Envoie à Adelya les ventes contenues dans le fichier ticket_path.

        Retourne les numéros de factures envoyées avec succès, ceux non envoyées et ceux dont le client n'est pas fidélisé.

        Parameters
        ----------
        ticket_path: str
            Chemin du fichier contenant les tickets.
        pre_prod: bool
            True signifie que le cardnumber est un numéro fictif composé de l'id du fidelityMember envoyé par Adelya.

        Returns
        -------
        numfac_OK: list
            Liste des numéros de factures envoyées avec succès
        numfac_KO: list
            Liste des numéros de factures non envoyées
        invalid_customers_numfac: list
            Liste des numéros de factures dont le client n'est pas fidélisé
        logfiles: dict
            Liste des fichiers log générés par facture
        """

        root_path = ticket_path.parent

        numfac_OK = []
        numfac_KO = []
        invalid_customers_numfac = []
        logfiles = {}

        # Remarque : dans le meilleur des cas cardnumber est le gc_cli2.numcrt
        # et dans le pire des cas c'est le gc_cli.code ou en preprod une valeur numcrt
        # fictive composée de l'id du fidelityMember envoyé par Adelya.
        for numfac, cardnumber, adEvent in self.convert_tickets_to_AdEvent(ticket_path, pre_prod):
            logfile = f"{ticket_path}-{numfac}.log"
            if self.check_customer_is_valid(cardnumber):
                rslt = self.put(adEvent)
                if rslt["sent"]:
                    numfac_OK.append(numfac)
                else:
                    numfac_KO.append(numfac)
                with open(logfile, "w") as f:
                    print(rslt, file=f)
            else:
                invalid_customers_numfac.append(numfac)
                with open(logfile, "w") as f:
                    print(f"Le client {cardnumber} n'est pas un client valide pour Adelya.", file=f)
            logfiles[numfac] = logfile

        return numfac_OK, numfac_KO, invalid_customers_numfac, logfiles


    def send_invoices(self, invoices_path, with_details=True, debug=False, raise_exception=False, superDbg=False):
        """L'envoie des factures s'arrête dès la première erreur. Si raise_exception est True, l'exception est levée. Sinon, la fonction retourne une liste des réponses des requêtes.

        Returns
        -------
            responses: list
                Liste des réponses des requêtes
            hasErr: bool
                True si une erreur s'est produite, False sinon
        """
        responses = []
        for data in sendtick.convert_invoices_to_AdEvent(
            invoices_path, self, with_details, debug, superDbg
        ):
            logdata = data
            pp.pprint(logdata)

            try:
                rslt = self.put(data)
                logdata["response"] = rslt
                responses.append(rslt)
                pp.pprint(rslt)
                if 'ApiReturn' in rslt:
                    rtrn = rslt['ApiReturn']
                    logdata["ApiReturn"] = rtrn
                    pp.pprint(logdata)

                    if rtrn['code'] != 'OK':
                        print(invoices_path, "Erreur lors de l'envoi de la facture", data['AdEvent']['idExternal'], rtrn)
                        if raise_exception:
                            raise Exception(f"Erreur lors de l'envoi de la facture {data['AdEvent']['idExternal']}: {rtrn}")
                        return responses, True
                else:
                    if raise_exception:
                        raise Exception(f"Erreur lors de l'envoi de la facture {data['AdEvent']['idExternal']}: {rslt}")
                    return responses, True
                # break
            except Exception as e:
                logdata["error"] = str(e)
                pp.pprint(logdata)
                responses.append(logdata)
                numfac = data['AdEvent']['idExternal']
                logger.error(invoices_path, "Erreur lors de l'envoi de la facture", numfac, e)
                if raise_exception:
                    raise e
                return responses, True

        return responses, False


    def get_adEvent_by_numfac(self, numfac, shopId=None, with_details=False):
        adEvent = {
                "type": "addCA",
                "idExternal": numfac,
        }

        if shopId is not None:
            adEvent["group"] = {"idExternal": shopId}

        data = {"AdEvent": adEvent}
        rslt = self.get(data)
        compactAdEvent = rslt.get("CompactAdEvent", None)
        if not with_details:
            if compactAdEvent:
                return compactAdEvent
            return rslt

        # {'CompactAdEvent': {'date': '2024-05-30 22:07:52',
        #             'dateUpdateStatus2': '2024-08-07 12:38:19',
        #             'fparam': '67.82',
        #             'fvalue': '67.82',
        #             'group': {'codeGroup': 'G6449519086',
        #                       'descrGroup': 'Yves Rocher MONTJOLY',
        #                       'id': '59124',
        #                       'idExternal': '6',
        #                       'salesDescr': 'Yves Rocher MONTJOLY',
        #                       'uniqueId': 'G-8c008b60-be36-4516-86ef-ea9fafb9f8ba'},
        #             'id': '608880404',
        #             'idExternal': 'FG24001566',
        #             'member': '600099769',
        #             'status1': False,
        #             'status2': True,
        #             'status3': False,
        #             'type': 'addCA',
        #             'user': '235134'}}

        data ={
            "AdEventDetail":{
                "event":{
                    "id": compactAdEvent["id"]
                }
            }
        }
        details = self.get(data)
        lines = details.get("list", [])
        # {
        # "list":[
        #     {
        #         "id":"605473042",
        #         "event":"608880404",
        #         "detailType":"I",
        #         "class1":"CN3 OMBRE LIFEPROOF STICK KAKI 1.4G",
        #         "class2":"I110",
        #         "class10":"52560",
        #         "unitValue":"11.61",
        #         "quantity":"1.0",
        #         "value":"0.0",
        #         "finalValue":"11.61"
        #     },
        #     ...
        #     }
        return compactAdEvent | {"list": lines}

    def get_adEvent_by_cardnumber(self, cardnumber):
        adEvent = {
                "type": "addCA",
                "FidelityMember": {"cardnumber": cardnumber},
        }

        data = {"AdEvent": adEvent}
        return self.get(data)

    def check_apireturn_is_ok(self, json_data):
        apiReturn = json_data.get("ApiReturn")
        if not apiReturn:
            return False

        code = apiReturn.get("code")
        if not code or code == "ERROR":
            return False

        if code == "OK":
            return True

        raise NotImplementedError(f"Le code {code} n'est pas encore géré.")



def convert_json_adEvent_to_dataframe(adEvent):

#     {
# 'date': '2024-05-30 22:07:52',
#  'dateUpdateStatus2': '2024-08-07 12:38:19',
#  'fparam': '67.82',
#  'fvalue': '67.82',
#  'group': {'codeGroup': 'G6449519086',
#            'descrGroup': 'Yves Rocher MONTJOLY',
#            'id': '59124',
#            'idExternal': '6',
#            'salesDescr': 'Yves Rocher MONTJOLY',
#            'uniqueId': 'G-8c008b60-be36-4516-86ef-ea9fafb9f8ba'},
#  'id': '608880404',
#  'idExternal': 'FG24001566',
#  'list': [{'class1': 'CN3 OMBRE LIFEPROOF STICK KAKI 1.4G',
#            'class10': '52560',
#            'class2': 'I110',
#            'detailType': 'I',
#            'event': '608880404',
#            'finalValue': '11.61',
#            'id': '605473042',
#            'quantity': '1.0',
#            'unitValue': '11.61',
#            'value': '0.0'},
#           {'class1': 'CN3 OMBRE LIFEPROOF STICK BLEU NUIT 1.4G',
#            'class10': '52806',
#            'class2': 'I110',
#            'detailType': 'I',
#            'event': '608880404',
#            'finalValue': '19.35',
#            'id': '605473044',
#            'quantity': '1.0',
#            'unitValue': '19.35',
#            'value': '0.0'},
#           {'class1': 'OMBRE LIFEPROOF PRUNE 1.4G',
#            'class10': '53136',
#            'class2': 'I110',
#            'detailType': 'I',
#            'event': '608880404',
#            'finalValue': '13.86',
#            'id': '605473043',
#            'quantity': '1.0',
#            'unitValue': '13.86',
#            'value': '0.0'},
#           {'class1': 'HUILE NACR�E HYDRATANTE MONO� 100 ML',
#            'class10': '56423',
#            'class2': 'K100',
#            'detailType': 'I',
#            'event': '608880404',
#            'finalValue': '23.0',
#            'id': '605473045',
#            'quantity': '1.0',
#            'unitValue': '23.0',
#            'value': '0.0'}],
#  'member': '600099769',
#  'status1': False,
#  'status2': True,
#  'status3': False,
#  'type': 'addCA',
#  'user': '235134'}

    data_to_duplicate = { k:v for k,v in adEvent.items() if k != "list"}

    rows = adEvent.get("list", [])
    csv_data = [{**data_to_duplicate, **row} for row in rows]
    return pd.DataFrame(csv_data)