__VERSION__="1.0.0"
__CDC_VERSION__="1.3.6"
__DESCRIPTION__="""Flux E6 ? Mouvements divers

L?int�gration de ce fichier permet de mettre a jour le stock des points de vente suite aux d�clarations des mouvements suivants :
?	CAM	Casse magasin			(Sortie marchandise)
?	VOL	Vol magasin			(Sortie marchandise)
?	TRF 	Transfert			(Entree/Sortie marchandise)
?	INV	R�gularisation d?inventaire	(Entree/Sortie marchandise)

Mise � disposition 0,1 fichier E6 dans le repertoire d'echange SFTP 'Export' dedie au client.

S'il existe deux mouvements identiques (m�me code client soci�t�, m�me num�ro de site, m�me date de cr�ation, m�me code article, m�me type de mouvement, m�me sens du mouvement), il faut les agr�ger en cumulant les quantit�s.
"""

"""
Nom zone	Type	Lg	Dont déc 	Déb	Fin	O/F	Description
Numéro de mini-centrale	| CHAR	| 10 |	  |	1	|   10  | 	O	Code Client 'société'
Numéro de site	        | CHAR	| 10 |	  |	11	|   20  | 	O	Code Client 'site'
Date de création	    | CHAR	| 8	 |	21|	28	|   O	| Date du jour de génération du mouvement
Code Article	        | NUM	| 13 |	  |	29	|   41  | 	O	Code article
Quantité mouvementée	| NUM	| 13 |	3 |	42	|   54  | 	O
Sens du mouvement	    | CHAR	| 1	 |	55|	55	|   O	| E=Entrée, S=Sortie
Type de mouvement	    | CHAR	| 3	 | 	56|	58	|   O	| TRF = Transfert, CAM = Casse magasin, VOL = Vol magasin, INV = Régularisation d’inventaire
Commentaire	            | CHAR	| 30 |	  |	59	|   88  | 	F


"""


    # if (begselvmltyp (vmlGifiV2))
    # {
    #   while (nextselvml ())
    #   {
    #     if (vml.lg1 == verc.nodep)
    #     {
    #       fidentok = TRUE;
    #       break;
    #     }
    #   }
    #   endselvml ();
    # }

import csv
from dataclasses import dataclass
import datetime
from pathlib import Path
import pprint as pp
import sys
from gescokit.api.gifi.base import FixedWidthParser
from typing import Any



# FIXME: Interrogation de la base de données pour récupérer les dépôts dans gc_ml type 9.
Magasin = {"ducos": "33", "martinique": "28", "robert": "64", "caribdl": "5005"}

class Mvtp1Parser(FixedWidthParser):
    """
    Parser pour les fichiers MVTP1 (mouvements divers).

    Ce parser traite les lignes de fichiers à largeur fixe pour les mouvements
    de stock (TRF, CAM, VOL, INV) issus des points de vente. Il convertit les lignes
    en dictionnaires et peut être utilisé pour la conversion CSV.

    Attributes
    ----------
    fields : list of tuple
        Champs définis avec leur largeur et, si nécessaire, une fonction de transformation.
        Liste des champs (nom, largeur ou (largeur, fonction de transformation))
    """

    fields: list[tuple[str, int | tuple[int, Any]]] = [
        ("id_centrale", 10),
        ("id_depot", 10),
        ("date", (8, lambda s: f"{s[0:4]}-{s[4:6]}-{s[6:8]}")),
        ("code", 13),
        ("quatite", (13, lambda s: int(s)/1000)),
        ("sens", 1),
        ("typ", 3),
        ("comment", 30),
    ]


@dataclass
class Mvtp1:
    centrale_id: str
    depot_id: str
    date: str
    code: str
    qt: int
    sens: str
    typ: str
    comment: str=""

    centrale_id_len = 10

    @staticmethod
    def __read_line(line):
        centrale_id = line[:10]
        depot_id = line[10:20]
        date = line[20:28]
        code = line[28:41]
        qt = int(line[41:54])/1000
        sens = line[54]
        typ = line[55:58]
        comment = line[58:].strip()
        return centrale_id, depot_id, date, code, qt, sens, typ, comment

    @staticmethod
    def read_line(line):
        data =  Mvtp1.__read_line(line)
        return Mvtp1(*data)

    @staticmethod
    def read_file(filename):
        mvtp1s = []
        with open(filename, "r") as fp:
            for line in fp:
                mvtp1s.append(Mvtp1.read_line(line))
        return mvtp1s

    @staticmethod
    def to_csv(filename, mvtp1s, qt_signed=False):
        with open(filename, "w") as fp:
            for m in mvtp1s:
                qt = m.qt
                if qt_signed:
                    qt = m.qt if m.sens == "E" else -m.qt
                print(";".join(map(str,[m.centrale_id, m.depot_id, m.date, m.code, qt, m.sens, m.typ, m.comment])), file=fp)


    def write_line(self, stream):
        print(f"{self.centrale_id:0>10s}", end="", file=stream)
        print(f"{self.depot_id:0>10s}", end="", file=stream)
        print(f"{self.date:8s}", end="", file=stream)
        print(f"{self.code:0>13s}", end="", file=stream)
        print(f"{self.qt:0>13d}", end="", file=stream)
        print(f"{self.sens:1}", end="", file=stream)
        print(f"{self.typ:3}", end="", file=stream)
        print(f"{self.comment:30}", file=stream, end="\r\n")
        # Numéro de mini-centrale	| CHAR	| 10 |	  |	1	|   10  | 	O	Code Client ‘société’
        # Numéro de site	        | CHAR	| 10 |	  |	11	|   20  | 	O	Code Client ‘site’
        # Date de création	        | CHAR	| 8	 | 	  | 21  |	28	|   O	| Date du jour de génération du mouvement
        # Code Article	            | NUM	| 13 |	  |	29	|   41  | 	O	Code article
        # Quantité mouvementée	    | NUM	| 13 |	3 |	42	|   54  | 	O
        # Sens du mouvement	        | CHAR	| 1	 |	55|	55	|   O	| E=Entrée, S=Sortie
        # Type de mouvement	        | CHAR	| 3	 | 	56|	58	|   O	| TRF = Transfert, CAM = Casse magasin, VOL = Vol magasin, INV = Régularisation d’inventaire
        # Commentaire	            | CHAR	| 30 |	  |	59	|   88  | 	F
# Mvtp1("1005", "28", "20240308", "", 1, "E", "INV", "comment")

    def __str__(self) -> str:
        return self.write_line(None)
        pass

def get_codart(data):
    return data[1]

def get_stock(data):
    if len(data)>9:
        print(data)
        # la designation de l'article comporte des '|' (element séparateur du fichier ascii)
        stock = data[5 + len(data)-9]
        print(stock)
    else:
        stock = data[5]
    return int(stock) * 1000 if stock else 0

def codart_13(codart):
    return f"{codart:0>13s}"

# import sys
# from pathlib import Path
# invfile = Path(sys.argv[1])



def lire_inventaire_emc2(ascii_file):
    inventaire_rows = {}
    art_13_doublons = {}
    articles_ignores = []
    with open(inventaire_emc2, "r", encoding="cp850") as csvfile:
        inventaire = csv.reader(csvfile, delimiter = '|')
        stocks = {}
        header_1 = next(inventaire)
        header_2 = next(inventaire)
        for row in inventaire:
            data = list( map(str.strip, row) )
            # if (len(data)>9):
            #     articles_ignores.append(data)
            #     print(f"Problème ligne", "|".join(data))
            #     continue
            codart = get_codart(data)
            codart = codart_13(codart)
            if codart not in stocks:
                stocks[codart] = get_stock(data)
                inventaire_rows[codart] = data
            else:
                if codart not in art_13_doublons:
                    art_13_doublons[codart] = [inventaire_rows[codart], data]
                else:
                    art_13_doublons[codart].append(data)
                # print(f"Ce code [{codart}] est déjà compté")

    return stocks, art_13_doublons


def traitement_des_doublons_sur_13_positions(stocks, doublons, output="doublons.csv"):
    # Traitement des articles avec codification similaire as sens GIFI France
    # exemple: les codes articles 001078 et 1078 sont identiques pour GIFI France
    print(len(doublons), "doublons")
    with open(output, "w") as fp:
        for rows in doublons.values():
            for row in rows:
                print(";".join(row), end="\r\n", file=fp)

    # Déterminer l'article qui sera envoyé à GIFI France
    for codart, rows in doublons.items():
        articles = {get_codart(row):get_stock(row) for row in rows}
        keep = [k for k in articles if k.startswith("0")]
        if len(keep) == 1:
            codart = codart_13(keep[0])
            stocks[codart] = articles[keep[0]]
        else:
            print("Je ne sais pas quel article envoyé a GIFI", codart, articles)

    return stocks




# pb = [m6 for m6 in moins_de_6 if f"{m6:0>6s}" in stocks]
# print(len(pb))
# for m6 in pb:
#     print()
# for m6 in moins_de_6:
#     if f"{m6:0>6s}" in stocks:
#         print(m6)
# sys.exit()


def calcul_ecart_de_stock(inventaire_gifi, stocks_emc2, depot):
    # "PRD_28_Stock non nul_20240215_extrait.csv"
    with open(inventaire_gifi, "r", encoding="utf-8-sig") as csvfile:

        gstock = {}
        ecart_stock = stocks_emc2.copy()
        inventaire = csv.reader(csvfile, delimiter = ';')

        errors = {}
        rejets = {}
        nblines = 0
        for row in inventaire:
            line = "".join(row)
            if not line.strip(): # ignorer les lignes vides
                continue
            nblines += 1
            depot_id = str(int(row[0]))
            if depot_id != depot:
                print(f"Ce dépôt [{depot_id} != {depot}] n'est pas le bon")
                continue

            codart = row[1]
            codart = codart_13(codart)

            try:
                qt = int(row[2].replace(',','').replace(' ',''))

                if codart in stocks_emc2:
                    ecart_stock[codart] -= qt
                    # # Retirer les articles dont les stocks sont identiques entre EMC2 et GIFI Distr.
                    # if new_stocks[codart] == 0:
                    #     new_stocks.pop(codart)
                else:
                    print(f"Ce code [{codart}] n'existe pas pour ce dépôt")
                    rejets[codart] = -qt # mettre l'inverse pour le supprimer
                gstock[codart] = qt
            except ValueError as e:
                print(f"Problème ligne {nblines} : {row}")
                errors[codart] = str(e)

        # Retirer les ecarts nuls.
        # Cause écart nul:
        # stock identique entre EMC2 et GIFI Distr. (x des 2 côtés ou x peut-être 0)
        for codart in list(ecart_stock.keys()):
            if ecart_stock[codart] == 0:
                ecart_stock.pop(codart)
        return ecart_stock, rejets, nblines, errors


def ecrire_flux_ecart_stock(centrale_id, depot_id, ecart_stock, filename="tcxmvtp1.dat"):
    date = datetime.date.today().strftime("%Y%m%d")

    with open(filename, "w") as fp:
        for key, val in ecart_stock.items():
            sens = "E" if val > 0 else "S"

            mvtp1 = Mvtp1(centrale_id=centrale_id,
                        depot_id=depot_id,
                        date=date,
                        code=key,
                        qt=abs(val),
                        sens=sens,
                        typ="INV",
                        comment="")
            mvtp1.write_line(fp)

    print(f"Flux {filename} généré")
if __name__ == "__main__":

    centrale_id = "100005"
    depot_id = "33"


    assert len(sys.argv) == 5, f"Usage: {sys.argv[0]} <centrale_id> <depot_id> <gifi_stock_filepath> <inventaire_emc2>"
    # Lecture du fichier envoyé par GIFI
    centrale_id = sys.argv[1]
    depot_id = sys.argv[2]
    gifi_stock_filepath = Path(sys.argv[3])
    inventaire_emc2 = Path(sys.argv[4])
    

    assert inventaire_emc2.is_file(), f"Fichier {inventaire_emc2} introuvable"
    assert gifi_stock_filepath.is_file(), f"Fichier {gifi_stock_filepath} introuvable"


    stocks_emc2, art_13_doublons = lire_inventaire_emc2(inventaire_emc2)
    stocks_emc2 = traitement_des_doublons_sur_13_positions(stocks_emc2, art_13_doublons)

    print(10*"#/")
    print(f"{len(art_13_doublons)} articles doublons sur 13 positions")
    print(10*"#/")

    print(f"{len(stocks_emc2)} articles comptés dans l'inventaire EMC2")

    with open("gifidu.art", "w", encoding="utf8") as fp:
        for key, val in stocks_emc2.items():
            print(f"{key};{val}", file=fp)

    ecart_stock, rejets, nbarticles, errors = calcul_ecart_de_stock(gifi_stock_filepath, stocks_emc2, depot_id)


    print(f"{len(rejets)} rejets / {nbarticles} articles gifi" )
    print(f"{len(errors)} erreurs / {nbarticles} articles gifi" )

    ecrire_flux_ecart_stock(centrale_id, depot_id, ecart_stock)
    ecrire_flux_ecart_stock(centrale_id, depot_id, rejets, "tcxmvtp1.dat.rejet")

    mvtp1s = Mvtp1.read_file("tcxmvtp1.dat")
    Mvtp1.to_csv("tcxmvtp1.csv", mvtp1s, qt_signed=True)

    mvtp1s = Mvtp1.read_file("tcxmvtp1.dat.rejet")
    Mvtp1.to_csv("tcxmvtp1.csv.rejet", mvtp1s, qt_signed=True)

    with open("erreurs.csv", "w") as fp:
        for key, val in errors.items():
            print(f"{key};{val}", file=fp)