from gescokit.api.gifi.base import FixedWidthParser
from gescokit.api.gifi.utils import lambda_int, lambda_YYYYMMDD
from typing import Any

description = """Flux S6 - Articles

Le fichier des articles contient les produits distribués par GIFI au client.
Il est alimenté en mode dit “delta”, c’est à dire qu’il ne contient que les articles créés ou modifiés depuis la dépose précédente sur le serveur SFTP.
Les articles concernés sont ceux appartenant au plan de vente du client.
Le plan de vente du client est la réunion des plans de vente de ses sites.
"""

class Artp2Parser(FixedWidthParser):
    """
    Parser for ARTP2 files.
    """

    version = "1.3.6"
    label = description
    filename = "tcxartp2.dat"
    flux_code = "S6"
    flux_name = "Ventes magasin détaillées"
    flux_type = "ARTP2"
    encoding = "utf-8"

    fields: list[tuple[str, int | tuple[int, Any]]] = [
    ("code_operation", 1),
    ("code_article", 13),
    ("ean",13),
    ("designation",20), #: str    # Désignation article (courte)
    # ("ecotaxes",20) #: str       # Caractères 1 à 6 : montant hors taxes de l’écotaxe DEEE (à blanc si non applicable). Partie entière sur les caractères 1 à 4, partie décimale sur les caractères 5 à 6.
    #                     # |Caractères 7 à 12 : montant hors taxes de l’écotaxe écomobilier (à blanc si non applicable). Partie entière sur les caractères 7 à 10, partie décimale sur les caractères 11 à 12.
    #                     # |Caractères 13 à 20 : inutilisés.
    ("deee", (6, lambda s: lambda_int(s) / 100)), #: int           # montant hors taxes de l’écotaxe DEEE (à blanc si non applicable). Partie entière sur les caractères 1 à 4, partie décimale sur les caractères 5 à 6.
    ("ecomeuble", (6, lambda s: lambda_int(s) / 100)), #: int      #
    ("unused", 8), # non utilis
    ("designation_3",50),#: str  # Désignation article (longue)
    ("code_famille",9),#: str   # famille (hiérarchie groupe marchandise, 3ème niveau)
    ("sous_classe",9),#: str       #hiérarchie groupe marchandise, 2ème niveau
    ("classe",9),#:str             #hiérarchie groupe marchandise, premier niveau
    ("gamme",3),#: str             #hiérarchie commerciale, 3ème niveau
    ("univers",3),#: str           #hiérarchie commerciale, 2ème niveau
    ("categorie",3),#: str         #hiérarchie commerciale, premier niveau
    ("cycle_vie",1),#: str         #Cycle de vie du produit
    ("code_fournisseur",10),#: int   #vide si le fournisseur est GIFI
    ("ref_fournisseur",15),#: int   #Référence du produit chez le fournisseur
    ("pcb",5),#:int                #Nombre d’articles par carton
    ("sous_pcb",5),#: int          #Nombre d’articles par sous-carton
    ("prix_achat",(13, lambda s: int(s)/1000)),#: str        #Prix d’achat à la centrale GIFI ou au fournisseur direct (HT)
    ("prix_vente",(13, lambda s: int(s)/1000)),#: str        #Prix de vente conseillé
    ("tva",1),#: float             #Code classification fiscale (TVA) Franceð  Voir liste de valeurs
    ("unused2",1),#:bool    #O=Oui, N ou vide=Non
    ("unite_vente",1),#: int       #U=Unité, K=Kilo, M=Mètre, L=Litre
    ("blocage_vente", (1,lambda s: s=="Oui")),#: bool    #O=Oui, N=Non. Indique si l’article est interdit à la vente en magasin (non-conformité, dangereux, etc.)
    ("unused3",1),#: bool        #'N'
    ("mode_enregistrement",1),#: int  #C=Consigné, D=Déconsigné, N=Négatif, S=Standard, T=Taxé
    ("date_maj", (8,lambda_YYYYMMDD)),#: int           #Date du traitement
    ("unite_conditionnement",1),#: int #Indique si le prix est au mètre, au litre, etc.Ex : M=Mètre, L=Litre, etc.
    ("contenance",7),#: int         #Nombre d’unités selon l’unité de conditionnement
    ]

    metadata: dict[str, dict[str, Any]] = {
        "code_operation": {
            "fullname": "Code opération",
            "description": "Code de l'opération (A=Ajout, M=Modification, S=Suppression)",
            "obligatoire": True
        },
        "code_article": {
            "fullname": "Code article",
            "description": "Code article GIFI",
            "obligatoire": True
        },
        "ean": {
            "fullname": "EAN principal",
            "description": "EAN principal GIFI",
            "obligatoire": True
        },
        "designation": {
            "fullname": "Désignation courte",
            "description": "Désignation courte de l'article",
            "obligatoire": True
        },
        "deee": {
            "fullname": "Écotaxe DEEE",
            "description": "Montant hors taxes de l'écotaxe DEEE (à blanc si non applicable). Partie entière sur les caractères 1 à 4, partie décimale sur les caractères 5 à 6.",
            "obligatoire": False
        },
        "ecomeuble": {
            "fullname": "ecomeuble",
            "description": "",
            "obligatoire": False
        },
        "unused": {
            "fullname": "Non utilisé",
            "description": "Caractères inutilisés",
            "obligatoire": False
        },
        "designation_3": {
            "fullname": "Désignation longue",
            "description": "Désignation longue de l'article",
            "obligatoire": True
        },
        "code_famille": {
            "fullname": "Code famille",
            "description": "Famille (hiérarchie groupe marchandise, 3ème niveau)",
            "obligatoire": True
        },
        "sous_classe": {
            "fullname": "Sous-classe",
            "description": "Hiérarchie groupe marchandise, 2ème niveau",
            "obligatoire": True
        },
        "classe": {
            "fullname": "Classe",
            "description": "Hiérarchie groupe marchandise, premier niveau",
            "obligatoire": True
        },
        "gamme": {
            "fullname": "Gamme",
            "description": "Hiérarchie commerciale, 3ème niveau",
            "obligatoire": True
        },
        "univers": {
            "fullname": "Univers",
            "description": "Hiérarchie commerciale, 2ème niveau",
            "obligatoire": True
        },
        "categorie": {
            "fullname": "Catégorie",
            "description": "Hiérarchie commerciale, premier niveau",
            "obligatoire": True
        },
        "cycle_vie": {
            "fullname": "Cycle de vie",
            "description": "Cycle de vie du produit",
            "obligatoire": True
        },
        "code_fournisseur": {
            "fullname": "Code fournisseur",
            "description": "Code du fournisseur (vide si le fournisseur est GIFI)",
            "obligatoire": False
        },
        "ref_fournisseur": {
            "fullname": "Référence fournisseur",
            "description": "Référence du produit chez le fournisseur",
            "obligatoire": False
        },
        "pcb": {
            "fullname": "Nombre d'articles par carton",
            "description": "Nombre d'articles par carton",
            "obligatoire": True
        },
        "sous_pcb": {
            "fullname": "Nombre d'articles par sous-carton",
            "description": "Nombre d'articles par sous-carton",
            "obligatoire": True
        },
        "prix_achat": {
            "fullname": "Prix d'achat",
            "description": "Prix d'achat à la centrale GIFI ou au fournisseur direct (HT)",
            "obligatoire": True
        },
        "prix_vente": {
            "fullname": "Prix de vente",
            "description": "Prix de vente conseillé",
            "obligatoire": True
        },
        "tva": {
            "fullname": "TVA",
            "description": "Code classification fiscale (TVA) France. Voir liste de valeurs",
            "obligatoire": True
        },
        "unite_vente": {
            "fullname": "Unité de vente",
            "description": "U=Unité, K=Kilo, M=Mètre, L=Litre",
            "obligatoire": True
        },
        "blocage_vente": {
            "fullname": "Blocage vente",
            "description": "O=Oui, N=Non. Indique si l’article est interdit à la vente en magasin (non-conformité, dangereux, etc.)",
            "obligatoire": True
        },
        "mode_enregistrement": {
            "fullname": "Mode d'enregistrement",
            "description": "C=Consigné, D=Déconsigné, N=Négatif, S=Standard, T=Taxé",
            "obligatoire": True
        },
        "date_maj": {
            "fullname": "Date de mise à jour",
            "description": "Date du traitement au format AAAAMMJJ",
            "obligatoire": True
        },
        "unite_conditionnement": {
            "fullname": "Unité de conditionnement",
            "description": "Indique si le prix est au mètre, au litre, etc. Ex : M=Mètre, L=Litre, etc.",
            "obligatoire": True
        },
        "contenance": {
            "fullname": "Contenance",
            "description": "Nombre d’unités selon l’unité de conditionnement",
            "obligatoire": True
        },
        "unused3": {
            "fullname": "Non utilisé",
            "description": "Caractère non utilisé",
            "obligatoire": False
        },
        "unused2": {
            "fullname": "Non utilisé",
            "description": "Caractère non utilisé",
            "obligatoire": False
        }

    }

from pathlib import Path
import subprocess
import json
import os
# Sur la centos9 on placera xbase dans /home/gestcom/bin/xbase alors que sur la centos6 on le placera dans /home/gestcom/xbase
XBASE = Path("/home/gestcom/bin/xbase") if Path("/home/gestcom/bin/xbase").is_file() else Path("/home/gestcom/xbase")

def recart(codeart, args, user, dbpath, rightpath):
    puht = puttc = pucaisse = None
    cmd = [ str(XBASE) ] + f"--db-path {dbpath} --rightpath {rightpath} -u {user} --ignore-db-vers ".split()
    cmd.extend(["--numsoc", args.numsoc, "--numdep", args.numdep])
    cmd.append("recart")
    cmd.append(f'{{\"motif\":\"{codeart}\",\"typ\":\"cod\",\"qt\":1}}')

    data = subprocess.run(cmd,env=os.environ, capture_output=True, text=True)
    try:
        if articles:=json.loads(data.stdout).get("articles"):
            puttc = articles[0].get("puttc")
            pucaisse = articles[0].get("pucaisse")
            puht = articles[0].get("puht")
        else:
            print(f"Erreur lors de la recherche de l'article {codeart}")
            print(data)
    except Exception as e:
        print(e)
        puttc = puht = pucaisse = None
    return puht, puttc, pucaisse